package com.cybelia.sandra.notifier;

import com.cybelia.sandra.EmailHelper;
import com.cybelia.sandra.SandraConfig;
import com.cybelia.sandra.SandraDAOHelper;
import com.cybelia.sandra.SandraSchedulerConfigHelper;
import com.cybelia.sandra.entities.Label;
import com.cybelia.sandra.entities.LabelDAO;
import com.cybelia.sandra.entities.notifier.Cron;
import com.cybelia.sandra.entities.notifier.CronDAO;
import com.cybelia.sandra.entities.notifier.Event;
import com.cybelia.sandra.entities.notifier.EventDAO;
import com.cybelia.sandra.entities.notifier.Queue;
import com.cybelia.sandra.entities.notifier.QueueDAO;
import com.cybelia.sandra.security.NotifierProfilManager;
import com.cybelia.sandra.security.NotifierSecurityHelper;
import freemarker.core.Environment;
import freemarker.template.Configuration;
import freemarker.template.Template;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.StringWriter;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.TopiaSecurityDAOHelper;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.topia.persistence.TopiaId;
import org.nuiton.topia.taas.entities.TaasUser;
import org.nuiton.topia.taas.entities.TaasUserDAO;
import org.nuiton.util.ApplicationConfig;

/**
 * @author sletellier
 */
public class SandraNotifier {

    protected static final Log log = LogFactory.getLog(SandraNotifier.class);

    protected void endTransaction(TopiaContext transaction) throws TopiaException {
        transaction.commitTransaction();
        transaction.closeContext();
    }

    /**
     * Notification d'un evenement
     */
    public void notifyEvent(String type, String... values) throws TopiaException {
        ThreadNotifyEvent threadNotifyEvent = new ThreadNotifyEvent(new SandraNotifier(), type, values);
        threadNotifyEvent.start();
    }

    /**
     * Notification pour un cron a partir de son identifiant, utilse par les jobs
     * de quartz
     */
    public void notifyCron(String cronTopiaId) throws TopiaException {
        ThreadNotifyCron threadNotifyCron = new ThreadNotifyCron(new SandraNotifier(), cronTopiaId);
        threadNotifyCron.start();
    }

    synchronized protected void doNotifyEvent(TopiaContext transaction, String type, String[] values) {
        List<Cron> crons = null;
        List<Queue> queues = null;
        try {
            QueueDAO queueDAO = SandraDAOHelper.getQueueDAO(transaction);
            EventDAO eventDAO = SandraDAOHelper.getEventDAO(transaction);

            // Ajout de l'evenement dans la queue
            Event event = eventDAO.findByType(type);

            Queue queue = queueDAO.create();
            queue.setEvent(event);
            queue.setDate(new Date());
            queue.addAllValues(Arrays.asList(values));
            queueDAO.update(queue);

            // Notification des crons directs seulement sur le dernier evenement
            queues = new ArrayList<Queue>(1);
            queues.add(queue);

            if (log.isDebugEnabled()) {
                log.debug("adding queue : " + queue.getTopiaId());
            }

            String hql = "SELECT cron FROM " + Cron.class.getName() + " cron WHERE " +
                    "cron.enable = :enable AND cron.event.type = :type AND " +
                    "(cron.expression IS NULL OR cron.expression = '') AND " +
                    "(cron.dateStart IS NULL OR cron.dateStart >= :date) AND " +
                    "(cron.dateEnd IS NULL OR cron.dateEnd <= :date)";

            crons = transaction.findAll(hql, "enable", true, "type", type, "date", new Date());

        } catch (TopiaException te) {
            log.error("Topia exception : ", te);
           // te.printStackTrace();
        } finally {
            // Fermeture de la transaction
            if (transaction != null) {
                try {
                    if (log.isDebugEnabled()){
                        log.debug("crons : " + crons + " queues not null : " + queues != null);
                    }
                    if (crons != null && queues != null) {
                        for (Cron cron : crons) {
                            if (log.isDebugEnabled()) {
                                log.debug("cron : " + cron.getTopiaId());
                            }

                            // Notification du cron
                            notifyCron(transaction, cron, queues);
                        }
                    } else {
                        if (transaction != null) {
                            endTransaction(transaction);
                        }
                    }
                } catch (Exception eee) {
                    log.error("Topia exception, trying to reopening connection", eee);

                } finally {
                    if (transaction != null) {
                        try {
                            endTransaction(transaction);
                        } catch (TopiaException eee) {
                            log.error("Topia exception, trying to reopening connection", eee);
                        }
                    }
                }
            }
        }
    }

    synchronized protected void doNotifyCron(TopiaContext transaction, String cronTopiaId) {
        try {
            CronDAO cronDAO = SandraDAOHelper.getCronDAO(transaction);

            Cron cron = cronDAO.findByTopiaId(cronTopiaId);

            // Recuperation de tout les evenementsdu meme type que le cron
            String hqlQueue = "SELECT queue FROM " + Queue.class.getName() + " queue WHERE " +
                    "queue.event = :event AND " +
                    "queue NOT IN (SELECT q FROM " + Queue.class.getName() + " q JOIN q.cron AS cron WHERE cron = :cron) AND " +
                    "queue.date >= :date";

            List<Queue> queues = transaction.findAll(hqlQueue,
                    "event", cron.getEvent(),
                    "cron", cron,
                    "date", cron.getTopiaCreateDate());

            if (!queues.isEmpty()) {
                // Notification du cron
                if (log.isDebugEnabled()) {
                    log.debug("NotifyCron " + cron.getTopiaId() + " in queue " + queues.size());
                }
                notifyCron(transaction, cron, queues);
            } else {
                if (log.isInfoEnabled()) {
                    log.info("Nothing to do for cron " + cron.getTopiaId());
                }
            }

        } catch (TopiaException te) {
            log.error("NotifyCron error", te);
        }  catch (Exception te) {
            log.error("NotifyCron error", te);
        } finally {
            try {
                endTransaction(transaction);
            } catch (TopiaException ex) {
                log.error("Topia exception : ", ex);
            }
        }
    }

    /**
     * Notification pour un cron sur une liste d'evenements
     */
    protected void notifyCron(TopiaContext transaction, Cron cron, List<Queue> queues) throws Exception {
        // Mise a jour des evenements envoyes
        for (Queue queue : queues) {
            queue.addCron(cron);
        }

        // Recuperation de l'auteur pour prendre ses droits pour l'envoi des
        // notification a des destinataires appartenant pas au systeme
        TaasUserDAO userDAO = TopiaSecurityDAOHelper.getTaasUserDAO(transaction);
        TaasUser author = userDAO.findByTopiaId(cron.getAuthor());

        // Recuperation des destinataires
        Collection<String> recipients = cron.getRecipients();
        Set<String> to = new HashSet<String>();
        if (log.isDebugEnabled()) {
            log.debug("Recipients : " + Arrays.toString(recipients.toArray()));
        }
        for (String recipient : recipients) {
            String topiaId = NotifierSecurityHelper.getTopiaIdRecipient(recipient);
            if (log.isDebugEnabled()) {
                log.debug("Recipient : " + topiaId);
            }
            if (TopiaId.isValidId(topiaId)) {
                TopiaEntity entity = transaction.findByTopiaId(topiaId);
                if (log.isDebugEnabled()) {
                    log.debug("entity instanceof TaasUser : " + recipient + " topiaId + " + topiaId);
                }
                // Destinataire de type utilisateur
                if (entity instanceof TaasUser) {
                    TaasUser user = (TaasUser) entity;
                    String email = user.getEmail();
                    if (log.isDebugEnabled()) {
                        log.debug("Dest type user : " + email);
                    }
                    if (to.add(email)) {
                        if (log.isDebugEnabled()) {
                            log.debug("Added");
                        }
                        notifyUser(transaction, cron, queues, user, email);
                    }

                 // Destinataire de type profil
                } else {
                    if (log.isDebugEnabled()) {
                        log.debug("Dest type profil");
                    }
                    String hql = "SELECT user FROM " + TaasUser.class.getName() + " user " +
                            "JOIN user.principals AS principal JOIN principal.authorizations AS authorization WHERE " +
                            "authorization.expression = :expression";

                    List<Object> args = new ArrayList<Object>();
                    args.add("expression");
                    args.add(topiaId);

                    // Recherche en plus un profil admin
                    if (NotifierSecurityHelper.isAdminProfil(recipient)) {
                        hql = "SELECT user FROM " + TaasUser.class.getName() + " user WHERE " +
                                "user IN (" + hql + ") AND " +
                                "user IN (SELECT user FROM " + TaasUser.class.getName() + " user " +
                                "JOIN user.principals AS principal WHERE " +
                                "principal.name = :profilAdmin)";

                        String profilAdmin = NotifierSecurityHelper.getAdminProfilName(topiaId);
                        args.add("profilAdmin");
                        args.add(profilAdmin);
                        if (log.isDebugEnabled()){
                            log.debug("Profil admin request : " + hql + " profilAdmin " + profilAdmin + " exp " + topiaId);
                        }
                    }
                    else {
                        log.info("Profil request : " + hql + " exp " + topiaId);
                    }
                    List<TaasUser> users = transaction.findAll(hql, args.toArray());
                    for (TaasUser user : users) {
                        String email = user.getEmail();
                        if (log.isDebugEnabled()) {
                            log.debug("Result mail " + email);
                        }
                        if (to.add(email)) {
                            if (log.isDebugEnabled()) {
                                log.debug(email + " added");
                            }
                            notifyUser(transaction, cron, queues, user, email);
                        }
                    }
                }

                // Destinataire libre
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("Dest libre : " + recipient);
                }
                if (to.add(recipient)) {
                    if (log.isDebugEnabled()) {
                        log.debug("added");
                    }
                    notifyUser(transaction, cron, queues, author, recipient);
                }
            }
        }
        if (log.isInfoEnabled()) {
            if (to.isEmpty()) {
                log.info("No recipient cron " + cron.getTopiaId());
            }
        }
    }

    /**
     * Notification pour un cron sur une liste d'evenements pour un destinataire
     * @param cron
     * @param queues
     * @param user
     * @param to
     */
    protected void notifyUser(TopiaContext transaction, Cron cron, List<Queue> queues, TaasUser user, String to) throws Exception  {
        if (log.isDebugEnabled()) {
            log.debug("NotifieUser queue size : " + queues.size() + " for user " + user.getLogin());
        }
        // Creation du context
        List<Map<String, Object>> events = new ArrayList<Map<String, Object>>(queues.size());
        List<String> pjs = new ArrayList<String>();
        for (Queue queue : queues) {
            if (log.isDebugEnabled()) {
                log.debug("Queue : " + queue.getTopiaId());
            }
            Map<String, Object> context = new HashMap<String, Object>();
            boolean secure = false;

            // Mise dans une map des valeurs de la Queue par rapport aux variables de l'Event
            List<String> variables = new ArrayList<String>(queue.getEvent().getVariables());

            // Copy to remove var without break for
            List<String> finalVariable = new ArrayList<String>(variables);
            List<String> values = new ArrayList<String>(queue.getValues());

            // Cached unauthorized entities indexes
            Map<Integer, boolean[]> cachedUnauthorized = new HashMap<Integer, boolean[]>();
            Map<String, List<TopiaEntity>> cachedEntities = new HashMap<String, List<TopiaEntity>>();
            try {
                int index = 0;
                for (String variable : variables) {
                    String value = values.get(index);

                    index++;

                    if (log.isDebugEnabled()){
                        log.debug("Variable : " + variable);
                    }

                    if (isPj(variable)) {

                        // Get notif dir
                        String statsPath = SandraSchedulerConfigHelper.getNotifPath(SandraConfig.getConfig());
                        File statsDir = new File(statsPath);

                        // Creating tmp dir with name of variable more date
                        SimpleDateFormat format = new SimpleDateFormat("yyMMdd");
                        String now = format.format(new Date());

                        String fileName = variable.replace("{date}", now);
                        String statPath = statsDir + File.separator + fileName;
                        File statFile = new File(statPath);

                        FileWriter writer = null;
                        try {
                            writer = new FileWriter(statFile, false);

                            writer.write(value);

                        } finally {
                            if (writer != null) {
                                writer.close();
                            }
                        }

                        pjs.add(statPath);

                        // Remove variable to make a secure context
                        finalVariable.remove(variable);
                    } else {

                        // List d'entities
                        if (value != null && isValidListOfId(value)) {
                            if (log.isDebugEnabled()){
                                log.debug("list entity");
                            }
                            String[] topiaIds = value.split("\\&\\&");
                            if (log.isDebugEnabled()){
                                log.debug("Found in list " + variable + " " + topiaIds.length + " values " + "(" + value + ")");
                            }
                            // Preparation du cache
                            boolean[] unauthorized = cachedUnauthorized.get(topiaIds.length);
                            if (unauthorized == null){
                                unauthorized = new boolean[topiaIds.length];
                            }

                            // Set all to false
                            Arrays.fill(unauthorized, false);

                            List<TopiaEntity> entities = new ArrayList<TopiaEntity>();
                            int lenght = topiaIds.length;
                            int cnt = 0;
                            for (String topiaid : topiaIds){
                                TopiaEntity entity;
                                if (log.isDebugEnabled()){
                                    log.debug("entity");
                                }
                                if(NotifierSecurityHelper.isAdmin(user)) {
                                   entity = transaction.findByTopiaId(topiaid);
                                } else {
                                   entity = NotifierProfilManager.getTopiaEntity(transaction, user, topiaid);
                                }
                                if (log.isDebugEnabled()){
                                    log.debug("Find entity : " + entity);
                                }

                                // Si une entite n'est pas autorise
                                if (entity == null) {
                                    unauthorized[cnt] = true;
                                    if (log.isDebugEnabled()){
                                        log.debug("step " + variable + " is unauthorized " + topiaid);
                                    }
                                }
                                entities.add(entity);
                                if (log.isDebugEnabled()){
                                    log.debug("step " + variable + " unauthorized " + (entity == null) + " " + cnt + " " + lenght);
                                }
                                cnt++;
                            }

                            // Mis en cache
                            cachedUnauthorized.put(unauthorized.length, unauthorized);
                            cachedEntities.put(variable, entities);

                            if (log.isDebugEnabled()){
                                log.debug("Found in list " + variable + " " + entities.size() + " entities");
                            }
                        // is topiaEntity
                        } else if (value != null && TopiaId.isValidId(value)) {
                            if (log.isDebugEnabled()){
                                log.debug("entity");
                            }
                            TopiaEntity entity = null;
                            if(NotifierSecurityHelper.isAdmin(user)) {
                               entity = transaction.findByTopiaId(value);
                            } else {
                                if (log.isDebugEnabled()){
                                    log.debug("Try to find entity : " + value + " from user " + user.getLogin());
                                }
                                entity = NotifierProfilManager.getTopiaEntity(transaction, user, value);
                            }

                            if (log.isDebugEnabled()){
                                log.debug("Find entity : " + entity);
                            }

                            // Si une entite n'est pas autorise, arret du traitement
                            if (entity == null) {
                                secure = true;
                                log.warn("l'entite n'est pas autorise, arret du traitement : " + value);
                                break;
                            } else {
                                context.put(variable, entity);
                            }

                        // Is list of string
                        } else if (isList(value)) {

                            String[] splited = value.split("&&");
                            context.put(variable, Arrays.asList(splited));

                        // Other
                        } else {
                            if (log.isDebugEnabled()){
                                log.debug("other : " + value);
                            }
                            context.put(variable, value);
                        }
                        if (log.isDebugEnabled()){
                            log.debug("Finished Index : " + index + "/" + finalVariable.size() + " variable " + variable + " value " + value);
                        }
                    }
                }
            } catch (Exception e) {
                log.error("Exception : ", e);
            }

            if (log.isDebugEnabled()){
                log.debug("Context to add : " + context);
            }
            // Adding listEntities in context
            for (String variable : cachedEntities.keySet()){
                List<TopiaEntity> entities = cachedEntities.get(variable);
                boolean[] unauthorized = cachedUnauthorized.get(entities.size());

                // Reverse to keep indexes
                for (int i = unauthorized.length - 1; i >= 0;i--){
                    if (unauthorized[i]){
                        if (log.isDebugEnabled()){
                            log.debug("Removing entity " + i + " from list : " + variable);
                        }
                        // Remove unautorized entities
                        entities.remove(i);
                    }
                }

                // Si une entite n'est pas autorise, arret du traitement
                if (entities.isEmpty()){
                    secure = true;
                    log.warn("Final list " + variable + " is empty");
                    break;
                } else {
                    log.info("Final list size " + variable + " size " + entities.size());
                    context.put(variable, entities);
                }
            }

            // Si une entite n'est pas autorise, l'utilisateur n'est pas concerne par l'evenement
            if (!secure && context.size() == finalVariable.size()) {
                addLabelMapInContext(transaction, context);

                // Add helper in context
                context.put("helper", new TemplateHelper());

                if (log.isDebugEnabled()) {
                    for (String key : context.keySet()) {
                        log.debug("context key[" + key + "] value[" + context.get(key) + "]");
                    }
                }

                events.add(context);
            } else {
                log.warn("Context is not secure : " + context.toString() +
                        " context size : " + context.size() +
                        " variable size : " + finalVariable.size());
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("sendTemplate");
        }
        // L'utilisateur est concerne, envoi de la notification
        if (!events.isEmpty()) {
            if (log.isDebugEnabled()) {
                log.debug("Send cron " + cron.getTopiaId() + " to " + to);
            }
            sendTemplate(cron, events, pjs, to);
        } else {
            if (log.isInfoEnabled()) {
                log.info("Nothing to do " + cron.getTopiaId() + " to " + to);
            }
        }
    }

    protected void addLabelMapInContext(TopiaContext transaction, Map<String, Object> context) throws Exception {

        Map<String, String> labelMap = new HashMap<String, String>();

        LabelDAO labelDAO = SandraDAOHelper.getLabelDAO(transaction);
        List<Label> labels = new ArrayList<Label>(labelDAO.findAll());

        for (Label lbl : new ArrayList<Label>(labels)) {
            labelMap.put(lbl.getCategorie() + "-" + lbl.getId(), lbl.getValeur());
        }
        if (log.isDebugEnabled()) {
            log.debug("Found " + labelMap.size() + " label");
        }
        context.put("lbl", labelMap);
    }

    protected boolean isPj(String variable) {
        return variable.matches(".+\\-\\{date\\}\\..+");
    }

    protected boolean isValidListOfId(String value) throws Exception  {
//        java heap space
//        return value.matches("(.*?#[0-9]+#[0-9.]+\\&\\&)+.*?#[0-9]+#[0-9.]+");
        return value != null && value.matches(".*?#[0-9]+#[0-9.]+\\&\\&.*");
    }

    protected boolean isList(String value) throws Exception  {
        return value != null && value.matches("(.*\\&\\&)+.*?");
    }

    public void sendTemplate(Cron cron, List<Map<String, Object>> events, List<String> pjs, String to) throws Exception {
        if (log.isDebugEnabled()) {
            log.debug("sendTemplateIn");
        }
        // Creation du context pour le template
        Map<String, Object> context = new HashMap<String, Object>();
        context.put("events", events);
        context.put("event", events.get(0));

        // Recuperation du template
        ApplicationConfig appConfig = SandraConfig.getConfig();
        String dirName = SandraSchedulerConfigHelper.getNotifierTemplateDir(appConfig);
        String fileName = dirName + cron.getTemplate() + SandraSchedulerConfigHelper.getNotifierTemplateExt(appConfig);

        if (log.isDebugEnabled()) {
            log.debug("dirName : " + dirName + " fileName : " + fileName);
        }

        // Rendu du template
        StringWriter content = new StringWriter();

        // Configuration
        Configuration config = new Configuration();

        // Encoding par default
        config.setDefaultEncoding("utf8");
        try {
            Template template = new Template(dirName, new FileReader(fileName), config, "utf8");
            if (log.isDebugEnabled()){
                log.debug("Template encoding : " + template.getEncoding());
            }
            Environment env = template.createProcessingEnvironment(context, content);

            env.setOutputEncoding("utf8");
            env.process();
        } catch (Exception e) {
            log.error("Generating failed fileName : " + fileName + " content : " + content.toString(), e);
        }

        // Envoi de la notification
        String subject = "";
        String body = "";
        StringBuffer message = content.getBuffer();
        int index = message.indexOf("\n");
        if (index != -1) {
            subject = message.substring(0, index).trim();
            body = message.substring(index).trim();
        } else {
            subject = "(no subject)";
            body = message.toString().trim();
        }
        if (!body.isEmpty()) {
            if (log.isInfoEnabled()) {
                log.info("sendEmail to : " + to + " subject : " + subject);
            }
            EmailHelper.sendEmail(to, subject, body, pjs.toArray(new String[pjs.size()]));
        }
    }
}
