/*
 * Class:        GeometricGen
 * Description:  random variate generator for the geometric distribution
 * Environment:  Java
 * Software:     SSJ 
 * Copyright (C) 2001  Pierre L'Ecuyer and Universite de Montreal
 * Organization: DIRO, Universite de Montreal
 * @author       
 * @since
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package umontreal.ssj.randvar;
import umontreal.ssj.rng.*;
import umontreal.ssj.probdist.*;

/**
 * This class implements a random variate generator for the *geometric*
 * distribution. Its has parameter @f$p@f$ and mass function
 * @anchor REF_randvar_GeometricGen_eq_fgeom
 * @f[
 *   p(x) = p(1-p)^x \mbox{ for } x=0,1,2,…, \tag{fgeom}
 * @f]
 * where @f$0\le p\le1@f$. Random variates are generated by calling
 * inversion on the distribution object.
 *
 * <div class="SSJ-bigskip"></div>
 *
 * @ingroup randvar_discrete
 */
public class GeometricGen extends RandomVariateGenInt {
   private double p;

   /**
    * Creates a geometric random variate generator with parameter @f$p@f$,
    * using stream `s`.
    */
   public GeometricGen (RandomStream s, double p) {
      super (s, new GeometricDist(p));
      setParams (p);
   }

   /**
    * Creates a new generator for the distribution `dist`, using stream
    * `s`.
    */
   public GeometricGen (RandomStream s, GeometricDist dist) {
      super (s, dist);
      if (dist != null)
         setParams (dist.getP());
   }
 
    
   public int nextInt () {
      return GeometricDist.inverseF (p, stream.nextDouble());
   }

/**
 * Generates a *geometric* random variate with parameter @f$p = @f$&nbsp;`p`,
 * using stream `s`, by inversion.
 */
public static int nextInt (RandomStream s, double p) {
      return GeometricDist.inverseF (p, s.nextDouble());
   }

   /**
    * Returns the parameter @f$p@f$ of this object.
    */
   public double getP() {
      return p;
   }

   /**
    * Sets the parameter @f$n@f$ and @f$p@f$ of this object.
    */
   protected void setParams (double p) {
      if (p < 0.0 || p > 1.0)
         throw new IllegalArgumentException ("p not in range [0, 1]");
      this.p = p;
   }
}