/*
 * #%L
 * Wikitty :: api
 * 
 * $Id: Search.java 643 2010-12-22 04:36:17Z bpoussin $
 * $HeadURL: http://svn.nuiton.org/svn/wikitty/tags/wikitty-3.0.2/wikitty-api/src/main/java/org/nuiton/wikitty/search/Search.java $
 * %%
 * Copyright (C) 2009 - 2010 CodeLutin, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.wikitty.search;

import org.nuiton.wikitty.search.operators.Restriction;
import org.nuiton.wikitty.search.operators.Like;
import org.nuiton.wikitty.search.operators.SubSearch;
import org.nuiton.wikitty.search.operators.Element;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.nuiton.wikitty.entities.FieldType;
import org.nuiton.wikitty.entities.Wikitty;
import org.nuiton.wikitty.WikittyUtil;

/**
 * Helper to create a criteria with a restriction
 *
 * Element :
 * <extensionName>.<fieldName>[.<fieldType>] : search on an extension and field with specific type (optional)
 * Criteria.ALL_EXTENSIONS.<fieldName>.<fieldType> : search on all extension and field name with specific type
 *
 * <fieldType> specify search on field as NUMERIC, STRING, WIKITTY, BOOLEAN, DATE. Use Element.ElementType to specify type.
 */
public class Search {

    /** Default operator type between all search condition. */
    public enum KIND {
        AND, OR, NOT
    }

    /** Defaut kind to {@link KIND#AND}. */
    protected KIND kind = KIND.AND;

    protected List<Restriction> restrictions;

    protected List<Search> subSearchs;

    public Search() {
        restrictions = new ArrayList<Restriction>();
        subSearchs = new ArrayList<Search>();
    }

    /**
     * Create Search query with field in wikitty argument.
     * Null field are not used in search request.
     *
     * @param wikitty example use to create query
     * @return query
     */
    static public Search query(Wikitty wikitty) {
        Search result = Search.query();
        result.kind = KIND.AND;
        // result object must have same extension that wikitty example
        for (String extName : wikitty.getExtensionNames()) {
            result.eq(Element.ELT_EXTENSION, extName);
        }

        for (String fqfieldName : wikitty.fieldNames()) {
            Object value = wikitty.getFqField(fqfieldName);
            if (value != null) {
                FieldType type = wikitty.getFieldType(fqfieldName);
                if (type.isCollection()) {
                    Collection<?> collection = (Collection<?>) value;
                    for (Object o : collection) {
                        String strValue = WikittyUtil.toStringForSearchEngine(type, o);
                        result.eq(fqfieldName, strValue);
                    }
                } else {
                    String strValue = WikittyUtil.toStringForSearchEngine(type, value);
                    result.eq(fqfieldName, strValue);
                }
            }
        }

        return result;
    }

    /**
     * Create new {@code Search} object with default kind to {@link KIND#AND}.
     * 
     * @return Search helper
     */
    public static Search query() {
        Search search = query(KIND.AND);
        return search;
    }

    /**
     * Create new {@code Search} object with custom kind.
     * 
     * @param kind kind
     * @return Search helper
     */
    public static Search query(KIND kind) {
        Search search = new Search();
        search.kind = kind;
        return search;
    }

    /**
     * Create new query on existent criteria to add new constraint to existent
     * criteria.
     * 
     * @param criteria
     * @return
     */
    public static Search query(Criteria criteria) {
        Search search = query();
        if (criteria != null) {
            search.restrictions.add(criteria.getRestriction());
        }
        return search;
    }
    
    protected static Element elt(String element) {
        Element elm = new Element();
        elm.setName(element);
        return elm;
    }

    /**
     * Contains.
     * 
     * @param element
     * @param values
     * @return
     */
    public Search contains(String element, Collection<String> values) {
        restrictions.add(RestrictionHelper.contains(elt(element),
                new ArrayList<String>(values)));
        return this;
    }

    /**
     * Contains.
     *
     * @param element
     * @param value
     * @return
     */
    public Search contains(String element, String value) {
        restrictions.add(RestrictionHelper.contains(elt(element),
                value));
        return this;
    }

    /**
     * Equals.
     * 
     * @param element
     * @param value
     * @return {@code this}
     */
    public Search eq(String element, String value) {
        restrictions.add(RestrictionHelper.eq(elt(element), value));
        return this;
    }

    /**
     * Equals each collection elements.
     * 
     * @param element
     * @param values
     * @return {@code this}
     */
    public Search eq(String element, Collection<String> values) {
        for (String value : values) {
            restrictions.add(RestrictionHelper.eq(elt(element), value));
        }
        return this;
    }

    /**
     * Like.
     * 
     * @param element
     * @param value
     * @param searchAs
     * @return {@code this}
     */
    public Search like(String element, String value, Like.SearchAs searchAs) {
        restrictions.add(RestrictionHelper.like(elt(element), value, searchAs));
        return this;
    }

    public Search like(String element, String value) {
        restrictions.add(RestrictionHelper.like(
                elt(element), value, Like.SearchAs.AsText));
        return this;
    }

    /**
     * Unlike.
     * 
     * @param element
     * @param value
     * @param searchAs
     * @return {@code this}
     */
    public Search unlike(String element, String value, Like.SearchAs searchAs) {
        restrictions.add(RestrictionHelper.unlike(elt(element), value, searchAs));
        return this;
    }

    public Search unlike(String element, String value) {
        restrictions.add(RestrictionHelper.unlike(
                elt(element), value, Like.SearchAs.AsText));
        return this;
    }

    /**
     * Not equals.
     * 
     * @param element
     * @param value
     * @return {@code this}
     */
    public Search neq(String element, String value) {
        restrictions.add(RestrictionHelper.neq(elt(element), value));
        return this;
    }

    /**
     * Greater than.
     * 
     * @param element
     * @param value
     * @return {@code this}
     */
    public Search gt(String element, String value) {
        restrictions.add(RestrictionHelper.great(elt(element), value));
        return this;
    }

    /**
     * Greater or equals.
     * 
     * @param element
     * @param value
     * @return {@code this}
     */
    public Search ge(String element, String value) {
        restrictions.add(RestrictionHelper.greatEq(elt(element), value));
        return this;
    }

    /**
     * Less than.
     * 
     * @param element
     * @param value
     * @return {@code this}
     */
    public Search lt(String element, String value) {
        restrictions.add(RestrictionHelper.less(elt(element), value));
        return this;
    }

    /**
     * Less or equals.
     * 
     * @param element
     * @param value
     * @return {@code this}
     */
    public Search le(String element, String value) {
        restrictions.add(RestrictionHelper.lessEq(elt(element), value));
        return this;
    }

    /**
     * Between.
     * 
     * @param element
     * @param lowerValue
     * @param upperValue
     * @return {@code this}
     */
    public Search bw(String element, String lowerValue, String upperValue) {
        restrictions.add(RestrictionHelper.between(elt(element), lowerValue, upperValue));
        return this;
    }

    /**
     * Starts with.
     * 
     * @param element
     * @param value
     * @return {@code this}
     */
    public Search sw(String element, String value) {
        restrictions.add(RestrictionHelper.start(elt(element), value));
        return this;
    }

    /**
     * Not starts with.
     * 
     * @param element
     * @param value
     * @return {@code this}
     */
    public Search nsw(String element, String value) {
        restrictions.add(RestrictionHelper.not(
                        RestrictionHelper.start(elt(element), value)));
        return this;
    }

    /**
     * Ends with.
     * 
     * @param element
     * @param value
     * @return {@code this}
     */
    public Search ew(String element, String value) {
        restrictions.add(RestrictionHelper.end(elt(element), value));
        return this;
    }

    /**
     * Not ends with.
     * 
     * @param element
     * @param value
     * @return {@code this}
     */
    public Search notew(String element, String value) {
        restrictions.add(RestrictionHelper.not(
                        RestrictionHelper.end(elt(element), value)));
        return this;
    }

    /**
     * Keyword.
     * 
     * @param value
     * @return {@code this}
     */
    public Search keyword(String value) {
        restrictions.add(RestrictionHelper.keyword(value));
        return this;
    }

    /**
     * Is null.
     *
     * @param fieldName
     * @return {@code this}
     */
    public Search isNull(String fieldName) {
        restrictions.add(RestrictionHelper.isNull(fieldName));
        return this;
    }

    /**
     * Is not null.
     *
     * @param fieldName
     * @return {@code this}
     */
    public Search isNotNull(String fieldName) {
        restrictions.add(RestrictionHelper.isNotNull(fieldName));
        return this;
    }

    /**
     * Not (sub query).
     * 
     * @return sub query
     */
    public Search not() {
        Search not = Search.query(KIND.NOT);
        subSearchs.add(not);

        Search search = Search.query(kind);
        not.subSearchs.add(search);

        return search;
    }

    /**
     * Or (sub query).
     * 
     * @return sub query
     */
    public Search or() {
        Search search = Search.query(KIND.OR);
        subSearchs.add(search);
        return search;
    }

    /**
     * And (sub query).
     * 
     * @return sub query
     */
    public Search and() {
        Search search = Search.query(KIND.AND);
        subSearchs.add(search);
        return search;
    }

    public Search associated(String foreignFieldName) {
        Search search = new SubSearch(foreignFieldName, this);
        return search;
    }
    
    /**
     * Return named criteria.
     * 
     * @param name name of criteria
     * @return new criteria
     */
    public Criteria criteria(String name) {
        Criteria criteria = new Criteria(name);
        Restriction result = getRestrictions();
        criteria.setRestriction(result);
        return criteria;
    }

    /**
     * Return unnamed criteria.
     * 
     * @return new criteria
     */
    public Criteria criteria() {
        Criteria criteria = criteria(null);
        return criteria;
    }

    protected Restriction getRestrictions() throws UnsupportedOperationException {
        Restriction result;
        if (restrictions.isEmpty() && subSearchs.isEmpty()) {
            result = RestrictionHelper.rFalse();
            
        } else if (restrictions.size() == 1 && subSearchs.isEmpty()) {
            result = restrictions.remove(0);

        } else if (subSearchs.size() == 1 && restrictions.isEmpty()) {
            Search subSearch = subSearchs.get(0);
            result = subSearch.getRestrictions();

            if (kind == KIND.NOT) {
                result = RestrictionHelper.not(result);
            }

        } else {
            List<Restriction> allRestrictions = new ArrayList<Restriction>(restrictions);
            for (Search subSearch : subSearchs) {
                Restriction restriction = subSearch.getRestrictions();
                allRestrictions.add(restriction);
            }
            switch (kind) {
                case OR:
                    result = RestrictionHelper.or(allRestrictions);
                    break;
                case AND:
                    result = RestrictionHelper.and(allRestrictions);
                    break;
                default:
                    throw new UnsupportedOperationException("Can't handle restriction kind " + kind.name());
            }
        }
        return result;
    }

    
}
