/*
 * #%L
 * Wikitty :: api
 * 
 * $Id: ImportExportCSVTest.java 659 2010-12-24 10:21:27Z sletellier $
 * $HeadURL: http://svn.nuiton.org/svn/wikitty/tags/wikitty-3.0.1/wikitty-api/src/test/java/org/nuiton/wikitty/addons/importexport/ImportExportCSVTest.java $
 * %%
 * Copyright (C) 2009 - 2010 CodeLutin, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.wikitty.addons.importexport;

import java.net.URL;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.nuiton.util.StringUtil;
import org.nuiton.wikitty.WikittyException;
import org.nuiton.wikitty.search.Criteria;
import org.nuiton.wikitty.search.PagedResult;
import org.nuiton.wikitty.WikittyConfig;
import org.nuiton.wikitty.entities.WikittyTreeNode;
import org.nuiton.wikitty.entities.WikittyTreeNodeImpl;
import org.nuiton.wikitty.entities.WikittyExtension;
import org.nuiton.wikitty.addons.WikittyImportExportService;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.WikittyService;
import org.nuiton.wikitty.WikittyServiceFactory;
import org.nuiton.wikitty.WikittyUtil;
import org.nuiton.wikitty.search.operators.Element;
import org.nuiton.wikitty.search.Search;

/**
 * Test for CSV import export class
 * 
 * @author chatellier
 * @version $Revision: 659 $
 * 
 * Last update : $Date: 2010-12-24 11:21:27 +0100 (ven., 24 déc. 2010) $
 * By : $Author: sletellier $
 */
public class ImportExportCSVTest {

    private final static Log log = LogFactory.getLog(ImportExportCSVTest.class);

    static protected WikittyService ws ;
    
    @BeforeClass
    static public void loadWikittyService() throws Exception {
        WikittyConfig config = new WikittyConfig(); // default config for in memory
//        String msg = "Try to passe test with "+ config.getPrintableConfig("wikitty.*", 50);
//        log.info(msg);
        ws = WikittyServiceFactory.buildWikittyService(config);
    }

    @Before
    public void clearStorage() throws Exception {
        ws.clear(null);
    }

    public WikittyService getWikittyService() {
        return ws;
    }
    
    public void setWikittyService(WikittyService wikittyService) {
        this.ws = wikittyService;
    }

    protected static final WikittyExtension extensionClient =
        new WikittyExtension("Client", "1.0", null,
                WikittyUtil.buildFieldMapExtension(
                    "String name"));

    protected static final WikittyExtension extensionTag =
        new WikittyExtension("Tag", "1.0", null,
                WikittyUtil.buildFieldMapExtension(
                    "String tags"));

    /**
     * Test que le pattern match les bonnes choses.
     */
    @Test
    public void testQueryPattern() {

        ImportExportCSV task = new ImportExportCSV();
        Pattern pattern = task.queryPattern;
        
        Matcher m = pattern.matcher("rtet");
        Assert.assertFalse(m.find());
        
        m = pattern.matcher("Client=4e2e4a93-c412-46b8-beb9-e76e0f16740d");
        Assert.assertFalse(m.find());
        
        m = pattern.matcher("Client=4e2e4a93-c412-46b8-beb9-e76e0f16740d");
        Assert.assertFalse(m.find());
        
        m = pattern.matcher("Client.name=toto");
        Assert.assertTrue(m.find());
        Assert.assertEquals("Client.name", m.group(1));
        Assert.assertEquals("Client", m.group(2));
        Assert.assertEquals("name", m.group(3));
        Assert.assertEquals("toto", m.group(4));
        
        m = pattern.matcher("Client.name=toto");
        Assert.assertTrue(m.find());
        Assert.assertEquals("toto", m.group(6));
        
        m = pattern.matcher("Client.name=\"toto\"");
        Assert.assertTrue(m.find());
        Assert.assertEquals("toto", m.group(5));
        
        m = pattern.matcher("Client.name=\"toto");
        Assert.assertFalse(m.find());
        
        m = pattern.matcher("Client.name=\"to\"to\"");
        Assert.assertTrue(m.find());
        Assert.assertEquals("to\"to", m.group(5));
        
        m = pattern.matcher("Client.name=toto\"");
        Assert.assertFalse(m.find());
    }
    
    /**
     * Test de parsing des valeurs multiples wikitty.
     */
    @Test
    public void testMultipleFieldsParsing() {
        String test = "(sdfsdf),(dfsdf),(fdsfsdfg),(sdfsdfqs)";
        String[] result = StringUtil.split(test);
        Assert.assertEquals(4, result.length);
        Assert.assertEquals("(sdfsdf)", result[0]);
        Assert.assertEquals("sdfsdf", result[0].substring(1, result[0].length() -1));
        Assert.assertEquals("(dfsdf)", result[1]);
        Assert.assertEquals("(fdsfsdfg)", result[2]);
        Assert.assertEquals("(sdfsdfqs)", result[3]);
        
        test = "(sdf\",\"sdf),(df()sdf),(fds\"(),()\"fsdfg),(sdfsdfqs)";
        result = StringUtil.split(test);
        Assert.assertEquals(4, result.length);
        Assert.assertEquals("(sdf\",\"sdf)", result[0]);
        Assert.assertEquals("(df()sdf)", result[1]);
        Assert.assertEquals("(fds\"(),()\"fsdfg)", result[2]);
        Assert.assertEquals("(sdfsdfqs)", result[3]);
        
        test = "toto ?";
        result = StringUtil.split(test);
        Assert.assertEquals(1, result.length);
        Assert.assertEquals("toto ?", result[0]);
    }

    /**
     * Test l'import export.
     */
    @Test
    public void testImport() {

        // store required import extension
        WikittyProxy proxy = new WikittyProxy(ws);
        proxy.storeExtension(extensionClient);
        proxy.storeExtension(WikittyTreeNodeImpl.extensionWikittyTreeNode);
        proxy.storeExtension(extensionTag);

        // declare import service
        WikittyConfig config = new WikittyConfig();
        WikittyImportExportService wsImport =
                new WikittyImportExportService(config, null, ws);
        String[] importFiles = {
            "/csv/importclient.csv",
            "/csv/importtree.csv",
            "/csv/importtree2.csv"};
        for (String importFile : importFiles) {
            URL importFileURL = ImportExportCSVTest.class.getResource(importFile);
            wsImport.syncImportFromUri(WikittyImportExportService.FORMAT.CSV,
                    importFileURL.toExternalForm());
        }

        // test extension support
        Criteria criteria = Search.query().eq(Element.ELT_EXTENSION, extensionTag.getName()).criteria();
        PagedResult<String> pagedResult = ws.findAllByCriteria(null, criteria);
        Assert.assertEquals(1, pagedResult.getNumFound());

        // test normal import
        Criteria criteria2 = Search.query().eq(extensionClient.getName() + ".name", "Toto").criteria();
        PagedResult<String> pagedResult2 = ws.findAllByCriteria(null, criteria2);
        Assert.assertEquals(1, pagedResult2.getNumFound());

        // import attachment id that not exist, must be not imported
        Criteria criteria3 = Search.query().eq(WikittyTreeNode.FQ_FIELD_WIKITTYTREENODE_NAME, "MyTreeNode").criteria();
        PagedResult<WikittyTreeNode> pagedResult3 = proxy.findAllByCriteria(WikittyTreeNode.class, criteria3);
        Assert.assertEquals(1, pagedResult3.getNumFound());
        WikittyTreeNode myTreeNode = pagedResult3.getFirst();
        Assert.assertEquals(1, myTreeNode.getAttachment().size());

        // test des requetes imbriquées
        Criteria criteria4 = Search.query().eq(WikittyTreeNode.FQ_FIELD_WIKITTYTREENODE_PARENT, myTreeNode.getWikittyId()).criteria();
        PagedResult<WikittyTreeNode> pagedResult4 = proxy.findAllByCriteria(WikittyTreeNode.class, criteria4);
        Assert.assertEquals(1, pagedResult4.getNumFound());
        WikittyTreeNode mySubNode = pagedResult4.getFirst();
        Assert.assertEquals("MySubNode", mySubNode.getName());

        // test wikitty is obselete exception
        try {
            for (String importFile : importFiles) {
                URL importFileURL = ImportExportCSVTest.class.getResource(importFile);
                wsImport.syncImportFromUri(WikittyImportExportService.FORMAT.CSV,
                        importFileURL.toExternalForm());
            }
        } catch (WikittyException eee) {
            Assert.fail();
        }
    }
}
