/*
 * #%L
 * Wikitty :: api
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2009 - 2010 CodeLutin, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.wikitty;


import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.util.Date;
import java.util.Enumeration;
import java.util.Properties;
import java.util.UUID;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jivesoftware.smack.PacketListener;
import org.jivesoftware.smack.XMPPConnection;
import org.jivesoftware.smack.packet.Message;
import org.jivesoftware.smack.packet.Packet;
import org.jivesoftware.smackx.muc.DiscussionHistory;
import org.jivesoftware.smackx.muc.MultiUserChat;
import org.nuiton.util.ApplicationConfig;

/**
 * Transporter d'event via xmpp. Pour que ca fonctionne il faut un serveur
 * xmpp avec une room
 * <p>
 * Configuration
 * <li>wikitty.xmpp.server = adresse du serveur (ex: im.codelutin.com)
 * <li>wikitty.xmpp.room = adresse de la room (ex: wikitty-event@im.codelutin.com)
 * <li>wikitty.notification.persistent = [true|false] indique si on doit
 * recevoir tous les messages sans en loupe un seul. Pour cela il faut que la
 * room soit archivee. (FIXME poussin 20101014 pas encore implantee)
 *
 * @author poussin
 * @version $Revision$
 *
 * Last update: $Date$
 * by : $Author$
 */
public class XMPPNotifierTransporter implements PacketListener,
        WikittyServiceNotifier.RemoteNotifierTransporter {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(XMPPNotifierTransporter.class);

    static final public String PROPERTY_EVENT_NAME = "wikitty-event";
    static final public String WIKITTY_XMPP_SERVER = "wikitty.xmpp.server";
    static final public String WIKITTY_XMPP_ROOM = "wikitty.xmpp.room";
    static final public String WIKITTY_NOTIFICATION_PERSISTENT =
            "wikitty.notification.persistent";

    /** Notifier service reference reference. */
    protected WikittyServiceNotifier ws;

    /** if persistent is true use Chat otherwize use MUC. Chat permit to
     received event missed after deconnection */
    protected boolean persistent = false;

    // Keep room addresse and pseudo to verify that messages event are not send by us
    protected String room;
    protected String pseudo;

    protected XMPPConnection connection;
    protected MultiUserChat muc;

    /**
     *
     * @param ws
     * @param props
     */
    public XMPPNotifierTransporter(WikittyServiceNotifier ws, Properties props) {
        this.ws = ws;
        initXMPP(props);
    }

    /**
     * Si persistent est vrai alors il faut toujours utilise le meme user id
     *
     * @param props
     */
    protected void initXMPP(Properties props) {
        ApplicationConfig config = new ApplicationConfig(props);

        persistent = config.getOptionAsBoolean(WIKITTY_NOTIFICATION_PERSISTENT);
        String server = config.getOption(WIKITTY_XMPP_SERVER);

        // Keep them to verify that is not us notifications
        room = config.getOption(WIKITTY_XMPP_ROOM);
        pseudo = getUniqueLoginName();
        try {
            if (log.isInfoEnabled()) {
                log.info("Try to connect to xmpp serveur " + server +
                     " with pseudo " + pseudo + " in room " + room +
                     " persistent " + persistent);
            }
            connection = new XMPPConnection(server);
            connection.connect();
            connection.loginAnonymously();

            DiscussionHistory history = new DiscussionHistory();
            if (persistent) {
                // FIXME poussin 20101113 compute amount of log history to retrieve
                // MUC must be archived
                Date date = new Date();
                history.setSince(date);
            } else {
                history.setMaxChars(0);
            }

            // connection to the volatile room
            muc = new MultiUserChat(connection, room);
            muc.join(pseudo, "", history, 4000);
            muc.addMessageListener(this);
        } catch (Exception eee) {
            throw new WikittyException("Can't connect to xmpp serveur", eee);
        }
    }

    @Override
    public void sendMessage(WikittyServiceEvent event) throws Exception {
        Message message = muc.createMessage();
        message.setBody(event.getType().toString());
        message.setProperty(PROPERTY_EVENT_NAME, event);

        muc.sendMessage(message);
    }

    /**
     * used for MUC message
     * @param packet
     */
    @Override
    public void processPacket(Packet packet) {

        // Dont listen own events
        String name = room + "/" + pseudo;
        if (!name.equals(packet.getFrom())) {

            Object event = packet.getProperty(PROPERTY_EVENT_NAME);

            if (log.isDebugEnabled()) {
                log.debug("Receive message : " + event);
            }

            if (event instanceof WikittyServiceEvent) {
                ws.processRemoteEvent((WikittyServiceEvent)event);
            }
        }
    }

    /**
     * Recherche le meilleur nom a utiliser comme pseudo pour cette machine.
     * Un UUID est toujours ajouter en debut, pour potentiellement lancer
     * plusieurs application en meme temps sur la meme machine et etre sur que
     * le pseudo soit bien unique.
     *
     * Le pseudo, n'est pas seulement un UUID, pour pouvoir facilement faire
     * du debug en entrant dans le channel et voir les ip des machiens connectees
     *
     * @return
     */
    static public String getUniqueLoginName() {
        String result = UUID.randomUUID().toString();
        try {
            String ipv4 = null;
            String ipv6 = null;
            for (Enumeration<NetworkInterface> e = NetworkInterface.getNetworkInterfaces(); e.hasMoreElements();) {
                NetworkInterface netint = e.nextElement();

                for (Enumeration<InetAddress> a = netint.getInetAddresses(); a.hasMoreElements();) {
                    InetAddress ip = a.nextElement();
                    if (ip instanceof Inet4Address) {
                        if ((ipv4 == null || "127.0.0.1".equals(ipv4))) {
                            ipv4 = ip.getHostAddress();
                        }
                    } else if (ip instanceof Inet6Address) {
                        if (ipv6 == null) {
                            ipv6 = ip.getHostAddress();
                        }
                    } else {
                        log.error("Can't get ip from no ipv4 and no ipv6: " + ip);
                    }
                }
            }

            if (ipv4 != null) {
                result = ipv4 + "=" + result;
            } else if (ipv6 != null) {
                result = ipv6 + "=" + result;
            }
        } catch (Exception eee) {
            log.error("Can't compute unique name from network interface", eee);
        }
        return result;
    }

}
