/*
 * #%L
 * Wikitty :: api
 * 
 * $Id: WikittyServiceInMemory.java 417 2010-10-15 15:24:44Z sletellier $
 * $HeadURL: http://svn.nuiton.org/svn/wikitty/tags/wikitty-2.2.2/wikitty-api/src/main/java/org/nuiton/wikitty/WikittyServiceInMemory.java $
 * %%
 * Copyright (C) 2009 - 2010 CodeLutin, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.wikitty;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.wikitty.FieldType.TYPE;
import org.nuiton.wikitty.search.And;
import org.nuiton.wikitty.search.BinaryOperator;
import org.nuiton.wikitty.search.Element;
import org.nuiton.wikitty.search.Restriction;

/**
 * In memory implementation of WikittyService, currently used for test only.
 *
 * @author poussin
 * @version $Revision: 417 $
 *
 * Last update: $Date: 2010-10-15 17:24:44 +0200 (ven., 15 oct. 2010) $
 * by : $Author: sletellier $
 */
public class WikittyServiceInMemory extends WikittyServiceImpl {

    //FIXME InMemory implementation is not usable for production. Must be reviewed.
    //FIXME The version increment must be done in 'prepare' method

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(WikittyServiceInMemory.class);

    public static class WikittyStorageInMemory implements WikittyStorage {
        protected Map<String, Wikitty> wikitties = new LinkedHashMap<String, Wikitty>();

        @Override
        public UpdateResponse store(WikittyTransaction transaction, Collection<Wikitty> wikitties, boolean force) {
            UpdateResponse result = new UpdateResponse();
            for (Wikitty wikitty : wikitties) {

                String actualVersion = null;
                Wikitty inMemoryWikitty = this.wikitties.get(wikitty.getId());
                if (inMemoryWikitty != null) {
                    actualVersion = inMemoryWikitty.getVersion();
                }
                String requestedVersion = wikitty.getVersion();

                String newVersion = null;
                if (force) {
                    if (actualVersion == null) { //no version in place
                        if (requestedVersion == null) { //no version requested
                            newVersion = WikittyUtil.DEFAULT_VERSION;
                        } else { //version requested
                            newVersion = requestedVersion;
                        }
                    } else { //version in place is not null
                        if (requestedVersion == null ||
                                requestedVersion.startsWith(WikittyUtil.DEFAULT_MAJOR_VERSION_PREFIX)) { //no version requested
                            newVersion = WikittyUtil.incrementMajorRevision(actualVersion);
                        } else if (WikittyUtil.versionEquals(actualVersion, requestedVersion)) { //same version
                            // wikitty is not modified, do nothing
                            continue;
                        } else if (WikittyUtil.versionGreaterThan(requestedVersion, actualVersion)) { //requested version is newer
                            newVersion = requestedVersion;
                        } else { //requested version is obsolete
                            throw new WikittyObsoleteException(String.format(
                                    "Your wikitty '%s' is obsolete (saving: '%s'; existing: '%s')", wikitty.getId(), requestedVersion, actualVersion));
                        }
                    }
                }
                else {
                    if (WikittyUtil.versionEquals(actualVersion, requestedVersion)) {
                        // no modification, continue
                        continue;
                    }
                    else if (WikittyUtil.versionGreaterThan(actualVersion, requestedVersion)) {
                        throw new WikittyObsoleteException(String.format(
                                "Your wikitty '%s' is obsolete", wikitty.getId()));
                    }
                    else {
                        newVersion = WikittyUtil.incrementMajorRevision(actualVersion);
                    }
                }
                wikitty.setVersion(newVersion);
                wikitty.clearDirty();

                try {
                    this.wikitties.put(wikitty.getId(), wikitty.clone());
                } catch (CloneNotSupportedException ex) {
                    if (log.isErrorEnabled()) {
                        log.error("Can't clone ?", ex);
                    }
                }
                result.addVersionUpdate(wikitty.getId(), wikitty.getVersion());
            }
            return result;
        }

        @Override
        public UpdateResponse delete(WikittyTransaction transaction, Collection<String> idList) throws WikittyException {
            UpdateResponse result = new UpdateResponse();
            Date now = new Date();
            for (String id : idList) {
                Wikitty w = restore(transaction, id);
                w.setDeleteDate(now);
                result.addDeletionDateUpdate(id, now);
            }
            return result;
        }

        @Override
        public boolean exists(WikittyTransaction transaction, String id) {
            boolean result = wikitties.containsKey(id);
            return result;
        }

        @Override
        public boolean isDeleted(WikittyTransaction transaction, String id) {
            boolean result = false;
            Wikitty w = wikitties.get(id);
            if (w == null) {
                throw new WikittyException(String.format("No wikitty with id '%s'", id));
            } else {
                result = w.isDeleted();
            }
            return result;
        }

        @Override
        public Wikitty restore(WikittyTransaction transaction, String id, String ... fqFieldName) throws WikittyException {
            Wikitty result = wikitties.get(id);
            if (result.isDeleted()) {
                result = null;
            }
            if (result == null) {
                throw new WikittyException(String.format("No wikitty with id '%s'", id));
            }
            return result;
        }

        @Override
        public void scanWikitties(WikittyTransaction transaction, Scanner scanner) {
            Collection<Wikitty> all = wikitties.values();
            for (Wikitty wikitty : all) {
                scanner.scan(wikitty.getId());
            }
        }

        @Override
        public void clear(WikittyTransaction transaction) {
            wikitties = new LinkedHashMap<String, Wikitty>();
        }
    }

    public static class WikittyExtensionStorageInMemory implements WikittyExtensionStorage {
        protected Map<String, WikittyExtension> extensions;

        public WikittyExtensionStorageInMemory() {
            this.extensions = new HashMap<String, WikittyExtension>();
        }

        @Override
        public UpdateResponse store(WikittyTransaction transaction,
                Collection<WikittyExtension> exts) throws WikittyException {
            for (WikittyExtension ext : exts) {
                extensions.put(ext.getId(), ext);
            }
            // nothing to do in UpdateResponse
            UpdateResponse result = new UpdateResponse();
            return result;
        }

        @Override
        public boolean exists(WikittyTransaction transaction, String id) {
            boolean result = extensions.containsKey(id);
            return result;
        }

        @Override
        public List<String> getAllExtensionIds(WikittyTransaction transaction) {
            List<String> result = new ArrayList<String>(extensions.keySet());
            return result;
        }

        @Override
        public List<String> getAllExtensionsRequires(WikittyTransaction transaction,
                String extensionName) {
            ArrayList<String> ids = new ArrayList<String>();
            
            Collection<WikittyExtension> values = extensions.values();
            if (values != null) {
                for (WikittyExtension extension : values) {
                    if (extensionName.equals(extension.getRequires())) {
                        ids.add(extension.getId());
                    }
                }
            }
            
            return ids;
        }
        
        @Override
        public String getLastVersion(WikittyTransaction transaction, String extName) {
            String result = null;
            Set<String> extensionIds = extensions.keySet();
            for (String extensionId : extensionIds) {
                String extensionVersion = WikittyExtension.computeVersion(extensionId);
                if(extensionId.startsWith(extName) && (result == null ||
                            WikittyUtil.versionGreaterThan(extensionVersion, result))) {
                    result = extensionVersion;
                }
            }
            return result;
        }

        @Override
        public WikittyExtension restore(WikittyTransaction transaction, String name, String version)
                throws WikittyException {
            String id = WikittyExtension.computeId(name, version);
            WikittyExtension result = extensions.get(id);
            if (result == null) {
                throw new WikittyException(String.format("No extension with id '%s'", id));
            }
            return result;
        }

        @Override
        public void clear(WikittyTransaction transaction) {
            extensions = new HashMap<String, WikittyExtension>();
        }
    }

    public static class WikittySearchEnginInMemory implements WikittySearchEngin {

        WikittyStorageInMemory wikittyStorage;

        public WikittySearchEnginInMemory(WikittyStorageInMemory wikittyStorage) {
            this.wikittyStorage = wikittyStorage;
        }
        
        @Override
        public void clear(WikittyTransaction transaction) {
            // do nothing
        }

        @Override
        public UpdateResponse store(WikittyTransaction transaction,
                Collection<Wikitty> wikitties) {
            // do nothing
            return new UpdateResponse();
        }

        @Override
        public UpdateResponse delete(WikittyTransaction transaction,
                Collection<String> idList) throws WikittyException {
            // do nothing
            return new UpdateResponse();
        }
        

        @Override
        public void delete(Collection<String> idList) throws WikittyException {
        }
        
        public boolean checkRestriction( Restriction restriction, Wikitty w ) {
            if ( restriction instanceof BinaryOperator ) {
                BinaryOperator binOp = (BinaryOperator) restriction;

                if (  binOp.getElement().getName().equals(Element.ELT_EXTENSION) ) {
                    return w.hasExtension(binOp.getValue());
                }

                String fqfieldName = binOp.getElement().getName();
                if(Element.ELT_EXTENSION.equals(fqfieldName)) {
                    return true; 
                }
                else if (Element.ELT_ID.equals(fqfieldName)) {
                    return w.getId().equals(binOp.getValue());
                }

                // Le check restriction, ne doit pas tester les champs
                // si les wikitty n'ont meme pas l'extension concerné
                String[] extName = fqfieldName.split("\\.");
                if (!w.hasField(extName[0], extName[1])) {
                    return false;
                }

                Object o = w.getFqField( fqfieldName );
                FieldType t = w.getFieldType(fqfieldName);
                Object value = t.getValidValue(binOp.getValue());
                boolean checked = false;
                switch( restriction.getName() ) {
                case EQUALS:
                    checked = value.equals(o);
                    break;
                case LESS:
                    checked = ((Comparable)o).compareTo( value ) < 0;
                    break;
                case LESS_OR_EQUAL:
                    checked = ((Comparable)o).compareTo( value ) <= 0;
                    break;
                case GREATER:
                    checked = ((Comparable)o).compareTo( value ) > 0;
                    break;
                case GREATER_OR_EQUAL:
                    checked = ((Comparable)o).compareTo( value ) >= 0;
                    break;
                case NOT_EQUALS:
                    checked = !value.equals(o);
                    break;
                case ENDS_WITH:
                    if (t.getType() != TYPE.STRING) {
                        throw new WikittyException("Can't search for contents that 'ends with' on attribute type different of String. " +
                                "Attribute " + fqfieldName + " is " + t.getType().name() );
                    }
                    checked = ((String) o).endsWith( (String) value );
                    break;
                case STARTS_WITH:
                    if (t.getType() != TYPE.STRING) {
                        throw new WikittyException("Can't search for contents that 'starts with' on attribute type different of String. " +
                                "Attribute " + fqfieldName + " is " + t.getType().name() );
                    }
                    checked = ((String) o).startsWith( (String) value );
                    break;
                }
                return checked;
            } else if ( restriction instanceof And ) {
                And and = (And) restriction;
                for ( Restriction sub : and.getRestrictions() ) {
                    if (!checkRestriction(sub, w)) {
                        return false;
                    }
                }
                return true;
            } else {
                throw new UnsupportedOperationException( restriction.getName() + " Search Not yet implemented");
            }
        }

        @Override
        public PagedResult<String> findAllByCriteria(WikittyTransaction transaction,
                Criteria criteria) {
            // throw new UnsupportedOperationException("Not supported yet.");
            
            int firstIndex = criteria.getFirstIndex();
            int endIndex = criteria.getEndIndex();
            
            List<String> ids = new LinkedList<String>();
            int currentIndex = 0;
            
            for( Entry<String, Wikitty> entry : wikittyStorage.wikitties.entrySet() ) {
                Wikitty w = entry.getValue();
                String id = entry.getKey();
                Restriction dto = criteria.getRestriction();

                if ( !w.isDeleted() && checkRestriction(dto, w) ) {
                    currentIndex++;
                    if (currentIndex > firstIndex) {
                        ids.add(id);
                    }
                    if (endIndex >= 0 && currentIndex >= endIndex) {
                        break;
                    }
                }

            }

            return new PagedResult<String>(firstIndex, ids.size(), criteria.getRestriction().toString(), null, ids );
        }

        @Override
        public Integer findNodeCount(WikittyTransaction transaction, Wikitty w, Criteria filter) {
            throw new UnsupportedOperationException("Not supported yet.");
        }

        @Override
        public Map<String, Integer> findAllChildrenCount(WikittyTransaction transaction, Wikitty w, Criteria filter) {
            throw new UnsupportedOperationException("Not supported yet.");
        }

    }

    /**
     * if persitenceFile is not null, serialize all data to disk during store
     * operation and the file is reloaded during init
     */
    protected File persitenceFile = null;

//    protected WikittyStorage wikittyStorage;
//    protected WikittyExtensionStorage extensionStorage;
//    protected WikittySearchEngin searchEngin;

    public WikittyServiceInMemory() {
        super(new WikittyExtensionStorageInMemory(),
                new WikittyStorageInMemory(),
                null);
        searchEngin = new WikittySearchEnginInMemory(
                (WikittyStorageInMemory) wikittyStorage);
    }

    public WikittyServiceInMemory(File persitenceFile) {
        this();
        this.persitenceFile = persitenceFile;
        if (persitenceFile != null && persitenceFile.exists()) {
            try {
                ObjectInputStream in = new ObjectInputStream(new FileInputStream(
                        persitenceFile));
                ((WikittyExtensionStorageInMemory)extensionStorage).extensions = (Map) in.readObject();
                ((WikittyStorageInMemory)wikittyStorage).wikitties = (Map) in.readObject();
                in.close();
            } catch (Exception eee) {
                log.error("Can't read data file " + persitenceFile, eee);
            }
        }
    }

    @Override
    protected void finalize() throws Throwable {
        saveToPersistenceFile();

        super.finalize();
    }

    public void saveToPersistenceFile() {
        if (persitenceFile != null) {
            try {
                ObjectOutputStream out = new ObjectOutputStream(new FileOutputStream(
                        persitenceFile));
                out.writeObject(((WikittyExtensionStorageInMemory)extensionStorage).extensions);
                out.writeObject(((WikittyStorageInMemory)wikittyStorage).wikitties);
                out.close();
            } catch (IOException eee) {
                log.error("Can't write data file " + persitenceFile, eee);
            }
        }
    }

}
