/*
 * #%L
 * Wikitty :: api
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2009 - 2010 CodeLutin, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.wikitty;

import java.io.File;
import java.io.FileWriter;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.Writer;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.FutureTask;

import org.nuiton.wikitty.importexport.ExportTask;
import org.nuiton.wikitty.importexport.ImportExportCSV;
import org.nuiton.wikitty.importexport.ImportExportMethod;
import org.nuiton.wikitty.importexport.ImportExportXML;
import org.nuiton.wikitty.importexport.ImportTask;
import org.nuiton.wikitty.importexport.JobState;
import org.nuiton.wikitty.search.Search;

/**
 * Import/export service.
 * 
 * @author poussin
 * @version $Revision$
 *
 * Last update: $Date$
 * by : $Author$
 */
public class WikittyImportExportService {

    static public enum FORMAT {
        XML(new ImportExportXML()),
        CSV(new ImportExportCSV());

        /** ieport == Importer/Exporter */
        protected ImportExportMethod ieporter;
        FORMAT(ImportExportMethod ieporter) {
            this.ieporter = ieporter;
        }

        public ImportExportMethod ieporter() {
            return ieporter;
        }
    };

    // FIXME poussin 20090902 next 3 variables must be read from configuration file
    /** number of thread used to import/export task */
    protected static final int MAX_IMPORT_EXPORT_THREAD = 1;
    /** directory path where export asynchronous file are stored */
    protected static final String EXPORT_DIRECTORY = "/tmp/";
    /** url used by client to retrieve export file when job is ended */
    protected static final String EXPORT_URL = "file:///tmp/";

    /** Executor that do import export task */
    protected ExecutorService importExportExecutor =
            // TODO poussin 20090902 do thread number configurable
            Executors.newFixedThreadPool(MAX_IMPORT_EXPORT_THREAD);

    /** contains all import or export task, key is job id send to client */
    protected Map<String, Future<String>> importExportTask =
            new HashMap<String, Future<String>>();

    protected WikittyService ws;
    protected String securityToken;

    public WikittyImportExportService(String securityToken, WikittyService ws) {
        this.securityToken = securityToken;
        this.ws = ws;
    }

    public WikittyService getWikittyService() {
        return ws;
    }

    public void syncImport(FORMAT format, String s) {
        Reader reader = new StringReader(s);
        ImportTask task = new ImportTask(securityToken, ws, format, reader);
        task.run();
    }

    public void syncImportFromUri(FORMAT format, String uri) {
        try {
            URL url = new URL(uri);
            Reader reader = new InputStreamReader(url.openStream());
            ImportTask task = new ImportTask(securityToken, ws, format, reader);
            task.run();
        } catch (Exception eee) {
            throw new WikittyException(String.format(
                    "Can't import in format %s uri %s", format, uri), eee);
        }
    }

    public String asyncImportFromUri(FORMAT format, String uri) {
        try {
            URL url = new URL(uri);
            Reader reader = new InputStreamReader(url.openStream());
            ImportTask task = new ImportTask(securityToken, ws, format, reader);
            FutureTask<String> future = new FutureTask<String>(task, null);
            importExportExecutor.submit(future);

            String jobId = UUID.randomUUID().toString();
            importExportTask.put(jobId, future);
            return jobId;
        } catch (Exception eee) {
            throw new WikittyException(String.format(
                    "Can't import in format %s uri %s", format, uri), eee);
        }
    }

    /**
     * Asynchronous export by example.
     * 
     * @param format export format
     * @param e sample
     * @return job id
     */
    public String asyncExportAllByExample(FORMAT format, BusinessEntityWikitty e) {
        Criteria criteria = Search.query(e.getWikitty()).criteria();
        String result = asyncExportAllByCriteria(format, criteria);
        return result;
    }

    /**
     * Synchronous export by example.
     * 
     * @param format export format
     * @param e sample
     * @return export string
     */
    public String syncExportAllByExample(FORMAT format, BusinessEntityWikitty e) {
        Criteria criteria = Search.query(e.getWikitty()).criteria();
        String result = syncExportAllByCriteria(format, criteria);
        return result;
    }

    /**
     * Asynchronous export by criteria.
     * 
     * @param format export format
     * @param criteria criteria
     * @return export as string
     */
    public String asyncExportAllByCriteria(FORMAT format, Criteria criteria) {
        try {
            String jobId = UUID.randomUUID().toString();

            File file = new File(EXPORT_DIRECTORY, jobId);
            String url = EXPORT_URL + jobId;
            Writer result = new FileWriter(file);
            ExportTask task = new ExportTask(
                    securityToken, ws, format, criteria, result);
            FutureTask<String> future = new FutureTask<String>(task, url);
            importExportExecutor.submit(future);

            importExportTask.put(jobId, future);
            return jobId;
        } catch (Exception eee) {
            throw new WikittyException(String.format(
                    "Can't export in format %s", format), eee);
        }
    }

    /**
     * Synchronous export by criteria.
     * 
     * @param format export format
     * @param criteria criteria
     * @return export as string
     */
    public String syncExportAllByCriteria(FORMAT format, Criteria criteria) {
        StringWriter result = new StringWriter();
        ExportTask task = new ExportTask(
                securityToken, ws, format, criteria, result);
        task.run();
        return result.toString();
    }

    /**
     * Return job information.
     * 
     * @param jobId job id
     * @return job state
     */
    public JobState infoJob(String jobId) {
        try {
            Future<String> future = importExportTask.get(jobId);
            JobState result = new JobState();
            if (future.isDone()) {
                result.status = "done";
                result.resourceUri = future.get();
            } else if (future.isCancelled()) {
                result.status = "cancelled";
            } else {
                result.status = "inProgress";
            }
            return result;
        } catch (Exception eee) {
            throw new WikittyException(String.format(
                    "Can't retrieve job info for job %s", jobId), eee);
        }
    }

    public void cancelJob(String jobId) {
        Future<String> future = importExportTask.get(jobId);
        future.cancel(true); // true to kill process, perhaps to strong ?
    }

    public void freeJobResource(String jobId) {
        Future<String> future = importExportTask.remove(jobId);
        if (future != null) {
            File file = new File(EXPORT_DIRECTORY, jobId);
            file.delete();
        }
    }

}
