/*
 * #%L
 * ToPIA :: Service Migration
 * 
 * $Id: TopiaMigrationEngine.java 2840 2013-10-11 15:13:30Z athimel $
 * $HeadURL: http://svn.nuiton.org/svn/topia/tags/topia-3.0-alpha-5/topia-service-migration/src/main/java/org/nuiton/topia/migration/TopiaMigrationEngine.java $
 * %%
 * Copyright (C) 2004 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.topia.migration;

import com.google.common.base.Preconditions;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.cfg.Configuration;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.TopiaNotFoundException;
import org.nuiton.topia.event.TopiaContextAdapter;
import org.nuiton.topia.event.TopiaContextEvent;
import org.nuiton.topia.event.TopiaContextListener;
import org.nuiton.topia.event.TopiaTransactionEvent;
import org.nuiton.topia.event.TopiaTransactionVetoable;
import org.nuiton.topia.framework.AbstractTopiaContext;
import org.nuiton.topia.framework.TopiaContextImplementor;
import org.nuiton.topia.framework.TopiaUtil;
import org.nuiton.topia.migration.mappings.TMSVersion;
import org.nuiton.topia.migration.mappings.TMSVersionDAO;
import org.nuiton.util.Version;
import org.nuiton.util.VersionUtil;
import org.nuiton.util.VersionUtil.VersionComparator;

import java.util.Arrays;
import java.util.List;
import java.util.Properties;
import java.util.SortedSet;
import java.util.TreeSet;

/**
 * Le moteur de migration proposé par topia. Il est basé sur un {@link AbstractTopiaMigrationCallback}
 * qui donne la version de l'application, les version de mises à jour disponibles.
 * <p/>
 * Le call back offre aussi les commandes sql à passer pour chaque version de mise à jour.
 * <p/>
 * FIXME Finir cette documentation
 *
 * @author tchemit
 * @version $Id: TopiaMigrationEngine.java 2840 2013-10-11 15:13:30Z athimel $
 * @since 2.3.4
 */
public class TopiaMigrationEngine implements TopiaMigrationService {

    /** logger */
    private final static Log log = LogFactory.getLog(TopiaMigrationEngine.class);

    /** Configuration hibernate ne mappant que l'entite version (initialise en pre-init) */
    protected Configuration versionConfiguration;

    /** Un drapeau pour savoir si la table version existe en base (initialise en pre-init) */
    protected boolean versionTableExist;

    /** Configuration hibernate ne mappant que l'entite version de l'ancien systeme de migration (initialise en pre-init) */
    protected Configuration legacyVersionConfiguration;

    /** Un drapeau pour savoir si la table version (de l'ancien service Manual) existe en base (initialise en pre-init) */
    protected boolean legacyVersionTableExist;

    /** Version courante de la base (initialise en pre-init) */
    protected Version dbVersion;

    /** Drapeau pour savoir si la base est versionnée ou non */
    protected boolean dbNotVersioned;

    /**
     * A flag to know if none of the dealed entities tables exists in db.
     *
     * @since 2.5.3
     */
    protected boolean dbEmpty;

    /** Un drapeau pour effectuer la migration au demarrage (initialise en pre-init) */
    protected boolean migrateOnInit;

    /** CallbackHandler list (initialise en pre-init) */
    protected AbstractTopiaMigrationCallback callback;

    /** topia root context (initialise en pre-init) */
    protected TopiaContext rootContext;

    /** Un drapeau pour savoir si le service a bien ete initialise (i.e a bien fini la methode preInit) */
    protected boolean init;

    /**
     * A flag to check if version was detected in database.
     * <p/>
     * This flag is set to {@code true} at the end of method {@link #detectDbVersion()}.
     */
    protected boolean versionDetected;

    /** Un drapeau pour afficher les requetes sql executees */
    protected boolean showSql;

    /** Un drapeau pour afficher la progression des requetes sql executees */
    protected boolean showProgression;

    /** delegate context listener. */
    protected final TopiaContextListener contextListener;

    /** delgate transaction listener */
    protected final TopiaTransactionVetoable transactionVetoable;

    public TopiaMigrationEngine() {

        transactionVetoable = new TopiaTransactionVetoable() {
            @Override
            public void beginTransaction(TopiaTransactionEvent event) {

                TopiaContext context =
                        event.getSource();

                // add topia context listener
                context.addTopiaContextListener(contextListener);

            }
        };
        contextListener = new TopiaContextAdapter() {

            @Override
            public void postCreateSchema(TopiaContextEvent event) {
                if (log.isDebugEnabled()) {
                    log.debug("postCreateSchema event called : will save version in database");
                }
                saveApplicationVersion();
            }

            @Override
            public void postUpdateSchema(TopiaContextEvent event) {
                if (log.isDebugEnabled()) {
                    log.debug("postUpdateSchema event called : will save version in database");
                }
                saveApplicationVersion();
            }

            @Override
            public void postRestoreSchema(TopiaContextEvent event) {
                if (log.isDebugEnabled()) {
                    log.debug("postRestoreSchema event detected, redo, schema migration");
                }
                if (migrateOnInit) {
                    // do automatic migration
                    if (log.isDebugEnabled()) {
                        log.debug("Starts Migrate from postRestoreSchema...");
                    }
                    try {
                        doMigrateSchema();
                    } catch (Exception e) {
                        if (log.isErrorEnabled()) {
                            log.error("postRestoreSchema schema migration failed for reason " + e.getMessage(), e);
                        }
                    }
                }
            }
        };
    }

    //--------------------------------------------------------------------------
    //-- TopiaService implementation
    //--------------------------------------------------------------------------

    @Override
    public Class<?>[] getPersistenceClasses() {
        return new Class<?>[]{TMSVersion.class};
    }

    @Override
    public String getServiceName() {
        return SERVICE_NAME;
    }

    @Override
    public boolean preInit(TopiaContext context) {
        rootContext = context;

        Properties config = ((AbstractTopiaContext)context).getConfig();

        String callbackStr = getSafeParameter(config, MIGRATION_CALLBACK);
        if (log.isDebugEnabled()) {
            log.debug("Use callback            - " + callbackStr);
        }

        migrateOnInit = Boolean.valueOf(config.getProperty(MIGRATION_MIGRATE_ON_INIT, String.valueOf(Boolean.TRUE)));
        if (log.isDebugEnabled()) {
            log.debug("Migrate on init         - " + migrateOnInit);
        }

        showSql = Boolean.valueOf(config.getProperty(MIGRATION_SHOW_SQL, String.valueOf(Boolean.FALSE)));
        if (log.isDebugEnabled()) {
            log.debug("Show sql                - " + showSql);
        }

        showProgression = Boolean.valueOf(config.getProperty(MIGRATION_SHOW_PROGRESSION, String.valueOf(Boolean.FALSE)));
        if (log.isDebugEnabled()) {
            log.debug("Show progression        - " + showProgression);
        }
        // enregistrement du callback
        try {
            Class<?> clazz = Class.forName(callbackStr);
            callback = (AbstractTopiaMigrationCallback) clazz.newInstance();
        } catch (Exception e) {
            log.error("Could not instanciate CallbackHandler [" + callbackStr + "]", e);
        }

        // creation de la configuration hibernate ne concernant que l'entite Version
        // afin de pouvoir creer la table via un schemaExport si necessaire

        Configuration configuration = new Configuration();
        for (Class<?> aClass : getPersistenceClasses()) {
            configuration.addClass(aClass);
        }

        versionConfiguration = createHibernateConfiguration(configuration);

        init = true;

        // add topia context listener
        context.addTopiaContextListener(contextListener);
        context.addTopiaTransactionVetoable(transactionVetoable);

        if (log.isDebugEnabled()) {
            log.debug("Service [" + this + "] is init.");
        }

        if (migrateOnInit) {
            // do automatic migration
            try {

                if (log.isDebugEnabled()) {
                    log.debug("Starts Migrate from preInit...");
                }
                doMigrateSchema();

            } catch (MigrationServiceException e) {
                throw new TopiaException("Can't migrate schema for reason " + e.getMessage(), e);
            }
        } else {
            if (log.isDebugEnabled()) {
                log.debug("Service [" + this + "] skip migration on init as required");
            }
        }
        return true;
    }

    @Override
    public boolean postInit(TopiaContext context) {
        // nothing to do in post-init
        return true;
    }

    public void doMigrateSchema() throws MigrationServiceException {
        // migration
        boolean complete = migrateSchema();
        if (!complete) {
            if (log.isErrorEnabled()) {
                log.error("Database migration not succesfully ended!");
            }
            throw new TopiaException("Database migration not succesfully ended!");
        }
    }

    //--------------------------------------------------------------------------
    //-- TopiaMigrationService implementation
    //--------------------------------------------------------------------------

    @Override
    public boolean migrateSchema() throws MigrationServiceException {

        checkInit();

        detectDbVersion();

        Version version = callback.getApplicationVersion();

        log.info(String.format("Starting Topia Migration Service  - Application version \\: %1$s, Database version \\: %2$s",
                               version.getVersion(),
                               dbVersion.getVersion())
        );

        if (log.isDebugEnabled()) {
            log.debug("Migrate schema to version = " + dbVersion);
            log.debug("is db not versionned ?    = " + dbNotVersioned);
            log.debug("is db empty ?             = " + dbEmpty);
            log.debug("TMSVersion exists         = " + versionTableExist);
        }

        if (dbEmpty) {

            // db is empty (no table, no migration to apply)
            return true;
        }

        if (versionTableExist && dbVersion.equals(version)) {
            if (log.isInfoEnabled()) {
                log.info("Database is up to date, no migration needed.");
            }
            // la base est a jour
            return true;
        }

        // Aucune version existante, la base de données est vierge
        if (versionTableExist && dbNotVersioned && migrateOnInit) {
            log.info("Database is empty, no migration needed.");
            // la base est vierge, aucune migration nécessaire
            // mise à jour de la table tmsversion
            saveApplicationVersion();
            return true;
        }

        if (legacyVersionTableExist && dbVersion.equals(version)) {

            // on a trouvee une table depreciee tmsVersion avec la bonne version de base
            // il suffit donc d'enregister la version dans la nouvelle table
            if (log.isInfoEnabled()) {
                log.info("Database is up to date, no migration needed.");
            }
            // la base est a jour mais il faut migrer la table
            saveApplicationVersion();
            return true;
        }

        SortedSet<Version> allVersions =
                new TreeSet<Version>(new VersionComparator());
        allVersions.addAll(Arrays.asList(callback.getAvailableVersions()));
        if (log.isInfoEnabled()) {
            log.info(String.format("Available versions: %1$s", allVersions));
        }

        // tell if migration is needed
        boolean needToMigrate = false;

        if (dbVersion.before(version)) {

            // on filtre les versions a appliquer
            List<Version> versionsToApply =
                    VersionUtil.filterVersions(allVersions,
                                               dbVersion,
                                               version,
                                               false,
                                               true
                    );

            if (versionsToApply.isEmpty()) {
                if (log.isInfoEnabled()) {
                    log.info("No version to apply, no migration needed.");
                }
            } else {
                if (log.isInfoEnabled()) {
                    log.info(String.format("Versions to apply: %1$s", versionsToApply));
                }

                // perform the migration
                needToMigrate = callback.doMigration(rootContext,
                                                     dbVersion,
                                                     showSql,
                                                     showProgression,
                                                     versionsToApply);

                if (log.isDebugEnabled()) {
                    log.debug("Handler choose : " + needToMigrate);
                }
                if (!needToMigrate) {
                    // l'utilisateur a annule la migration
                    return false;
                }
            }
        }

        // on sauvegarde la version si necessaire (base non versionnee ou migration realisee)
        if (!versionTableExist || needToMigrate) {

            if (log.isDebugEnabled()) {
                log.debug("Set application version in database to " + version);
            }

            // put version in database and create table if required
            saveApplicationVersion();
        }

        // return success flag
        // - no migration needed
        // - or migration needed and accepted
        return true;
    }

    //--------------------------------------------------------------------------
    //-- Internal methods
    //--------------------------------------------------------------------------

    /**
     * Enregistre la version donnee en base avec creation de la table
     * si elle n'existe pas.
     */
    protected void saveApplicationVersion() {

        checkInit();

        Version version = callback.getApplicationVersion();

        detectDbVersion();

        if (log.isDebugEnabled()) {
            log.debug("Save version     = " + version);
            log.debug("Table exists     = " + versionTableExist);
            log.debug("Detected version = " + dbVersion);
        }

        boolean createTable = !versionTableExist;
        // update version even if database has not been migrated
        // only case that database doesn't exist match this
        if (createTable) {
            // si la base n'etait pas versionnee, la table version n'existe pas
            // creation
            if (log.isDebugEnabled()) {
                log.debug("Adding tms_version table");
            }

            // creer le schema en base
            // dans la configuration versionConfiguration, il n'y a que la table version
            TMSVersionDAO.createTable(versionConfiguration);

            if (log.isDebugEnabled()) {
                log.debug("Table for " + TMSVersion.class.getSimpleName() + " created");
            }
        }

        // Set new version in database
        TopiaContext tx = rootContext.beginTransaction();
        try {

            // delete all previous data in table
            TMSVersionDAO.deleteAll(tx);

            if (log.isInfoEnabled()) {
                log.info(String.format("Saving new database version: %1$s", version));
            }

            // create new version and store it in table
            TMSVersion tmsVersion =
                    TMSVersionDAO.create(tx, version.getVersion());
            if (log.isDebugEnabled()) {
                log.debug("Created version: " + tmsVersion.getVersion());
            }

            tx.commitTransaction();
        } catch (TopiaException e) {
            if (tx != null) {
                tx.rollbackTransaction();
            }
            throw e;
        } finally {
            if (tx != null) {
                tx.closeContext();
            }
        }

        if (legacyVersionTableExist) {

            if (log.isDebugEnabled()) {
                log.debug("Will drop legacy tmsVersion table");
            }
            // on supprime l'ancienne table
            TMSVersionDAO.dropTable(legacyVersionConfiguration);
        }

        // on change les etats internes du service
        // ainsi cela empechera le redeclanchement de la migration
        // suite a une creation de schema
        versionTableExist = true;
        dbVersion = version;
    }

    /**
     * Recupere depuis la base les états internes du service :
     * <p/>
     * <ul>
     * <li>{@link #versionTableExist}</li>
     * <li>{@link #dbVersion}</li>
     * <li>{@link #dbEmpty}</li>
     * </ul>
     */
    protected void detectDbVersion() {

        if (versionDetected) {

            // this method was already invoked
            if (log.isDebugEnabled()) {
                log.debug("version was already detected : " + dbVersion);
            }
            return;
        }

        // compute dbempty field value
        dbEmpty = detectDbEmpty();

        if (log.isDebugEnabled()) {
            log.debug("Db is empty : " + dbEmpty);
        }


        Version v = null;
        try {

            // on detecte si la table de versionning existe
            versionTableExist =
                    TopiaUtil.isSchemaExist(versionConfiguration,
                                            TMSVersion.class.getName());

            // check if at least one class exists in db


            if (log.isDebugEnabled()) {
                log.debug("Table " + TMSVersionDAO.TABLE_NAME + " exist = " + versionTableExist);
            }

            if (versionTableExist) {

                // recuperation de la version de la base
                v = getVersion(versionTableExist, TMSVersionDAO.TABLE_NAME);

                if (log.isWarnEnabled()) {
                    if (v == null) {
                        log.warn("Version not found on table " + TMSVersionDAO.TABLE_NAME);
                    }
                }
                return;
            }

            // try with legacy table tmsVersion
            Configuration conf = new Configuration();
            conf.addXML(TMSVersionDAO.LEGACY_MAPPING);

            legacyVersionConfiguration = createHibernateConfiguration(conf);
            legacyVersionTableExist =
                    TopiaUtil.isSchemaExist(legacyVersionConfiguration,
                                            TMSVersion.class.getName());

            if (legacyVersionTableExist) {

                if (log.isDebugEnabled()) {
                    log.debug("Legacy : detected " + TMSVersionDAO.LEGACY_TABLE_NAME + " table");
                }

                // recuperation de la version de la base
                v = getVersion(legacyVersionTableExist, TMSVersionDAO.LEGACY_TABLE_NAME);

                if (v != null) {

                    if (log.isDebugEnabled()) {
                        log.debug(String.format("Legacy : detected database version: %1$s", v));
                    }
                }
            }
        } finally {

            if (v == null) {
                // la base dans ce cas n'est pas versionee.
                // On dit que la version de la base est 0
                // et les schema de cette version 0 doivent
                // etre detenu en local
                v = Version.VZERO;
                dbNotVersioned = true;
                log.info("Database version not found, so database schema is considered as V0");
            } else {
                log.info(String.format("detected database version: %1$s", v));
            }
            dbVersion = v;
            versionDetected = true;
        }
    }

    /**
     * Detects if there is some schema existing for at least one of the dealed
     * entity of the underlying db context.
     *
     * @return {@code true} if there is no schema for any of the dealed entities,
     *         {@code false} otherwise.
     * @since 2.5.3
     */
    protected boolean detectDbEmpty() {


        try {
            boolean result;
            // get db real hibernate configuration
//            Configuration rootConfiguration =
//                    rootContext.getHibernateConfiguration();

            result = TopiaUtil.isSchemaEmpty(rootContext);
            return result;
        } catch (TopiaNotFoundException e) {
            throw new RuntimeException(e);
        }

    }

    protected Version getVersion(boolean versionTableExist, String tableName) {
        if (!versionTableExist) {

            // table does not exist, version is null
            return null;
        }
        try {
            TopiaContext tx = rootContext.beginTransaction();
            try {
                Version v = TMSVersionDAO.getVersion(tx, tableName);
                return v;
            } finally {
                if (tx != null) {
                    tx.closeContext();
                }
            }
        } catch (TopiaException e) {
            throw new TopiaException("Can't obtain dbVersion for reason " + e.getMessage(), e);
        }
    }

    protected String getSafeParameter(Properties config, String key) {
        String value = config.getProperty(key, null);
        Preconditions.checkState(StringUtils.isNotEmpty(value), "'" + key + "' not set.");
        return value;
    }

    protected void checkInit() {
        Preconditions.checkState(init, "Service was not initialized!");
    }

    /**
     * Creates the hibernate configuration to be used by the service.
     *
     * @param configuration the incoming hibernate configuration
     * @return the complete configuration usable by the service
     * @since 2.5.3
     */
    protected Configuration createHibernateConfiguration(Configuration configuration) {
        Properties config = ((AbstractTopiaContext)rootContext).getConfig();

        Properties prop = new Properties();
        prop.putAll(configuration.getProperties());
        prop.putAll(config);

        configuration.setProperties(prop);
        configuration.buildMappings();
        return configuration;

    }


}