/*
 * #%L
 * ToPIA :: Service Migration
 * 
 * $Id: MigrationServiceImpl.java 1894 2010-04-15 15:44:51Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/topia/tags/topia-2.3.3/topia-service-migration/src/main/java/org/nuiton/topia/migration/MigrationServiceImpl.java $
 * %%
 * Copyright (C) 2004 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.topia.migration;

import java.io.File;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.migration.callback.MigrationCallbackHandler;
import org.nuiton.topia.migration.callback.MigrationCallbackHandler.MigrationChoice;
import org.nuiton.util.Version;
import org.nuiton.topia.migration.kernel.ConfigurationAdapter;
import org.nuiton.topia.migration.kernel.ConfigurationHelper;
import org.nuiton.topia.migration.kernel.Transformer;
import org.nuiton.util.Resource;
import org.hibernate.cfg.Configuration;

/**
 * MigrationServiceImpl.java
 *
 * Classe principale du projet.
 * 
 * @author Chatellier Eric
 * @author Chevallereau Benjamin
 * @author Eon Sï¿½bastien
 * @author Trï¿½ve Vincent
 * @version $Revision: 1894 $
 *
 * Last update : $Date: 2010-04-15 17:44:51 +0200 (jeu., 15 avril 2010) $
 */
public class MigrationServiceImpl implements MigrationService {

    /**
     * Nom courant du fichier de configuration.
     */
    protected String currentHibernateConfigurationFile;
    
    /**
     * Configuration hibernate courante utilisee par l'application
     */
    protected Configuration currentApplicationConfiguration;
    
    /**
     * Chemin du dossier contenant les schema de touts les versions
     */
    protected String mappingsDirectory;
    
    /**
     * Version courante de l'application
     */
    protected Version currentApplicationVersion;
    
    /**
     * CallbackHandler list
     */
    protected List<MigrationCallbackHandler> migrationCallBackHandlers;
    
    /**
     * Logger (common-logging)
     */
    private static Log logger = LogFactory.getLog(MigrationServiceImpl.class);
    
    /**
     * Constructeur vide.
     */
    public MigrationServiceImpl() {
        
        // init the configuration file
        currentHibernateConfigurationFile = null;
        // init configuration
        currentApplicationConfiguration = null;
        
        // init callbask list
        migrationCallBackHandlers = new LinkedList<MigrationCallbackHandler>();
    }

    /* (non-Javadoc)
     * @see org.nuiton.topia.migration.TopiaMigrationService#getConfigurationFile()
     */
    @Override
    public String getConfigurationFile() {
        return currentHibernateConfigurationFile;
    }

    /* (non-Javadoc)
     * @see org.nuiton.topia.migration.TopiaMigrationService#setConfigurationFile(java.lang.String)
     */
    @Override
    public void setConfigurationFile(String hibernateConfigurationFile) {
        currentHibernateConfigurationFile = hibernateConfigurationFile;
    }

    /* (non-Javadoc)
     * @see org.nuiton.topia.migration.TopiaMigrationService#getConfiguration()
     */
    @Override
    public Configuration getConfiguration() {
        return currentApplicationConfiguration;
    }

    /* (non-Javadoc)
     * @see org.nuiton.topia.migration.TopiaMigrationService#setConfiguration(org.hibernate.cfg.Configuration)
     */
    @Override
    public void setConfiguration(Configuration configuration) {
        currentApplicationConfiguration = configuration;
    }

    /* (non-Javadoc)
     * @see org.nuiton.topia.migration.TopiaMigrationService#getMappingsDirectory()
     */
    @Override
    public String getMappingsDirectory() {
        return mappingsDirectory;
    }

    /* (non-Javadoc)
     * @see org.nuiton.topia.migration.TopiaMigrationService#setMappingsDirectory(java.lang.String)
     */
    @Override
    public void setMappingsDirectory(String mappingsDirectory) {
        this.mappingsDirectory = mappingsDirectory;
    }
    
    /* (non-Javadoc)
     * @see org.nuiton.topia.migration.TopiaMigrationService#setApplicationVersion(java.lang.String)
     */
    @Override
    public void setApplicationVersion(String version) {
        currentApplicationVersion = new Version(version);
    }
    
    /* (non-Javadoc)
     * @see org.nuiton.topia.migration.TopiaMigrationService#addMigrationCallbackHandler(org.nuiton.topia.migration.callback.MigrationCallbackHandler)
     */
    @Override
    public void addMigrationCallbackHandler(
            MigrationCallbackHandler callbackHandler) {
        migrationCallBackHandlers.add(callbackHandler);
    }
    
    /**
     * Charge la configuration locale si elle n'est pas deja ete fournit
     */
    protected void loadApplicationConfiguration() {
        
        // configuration pas deja fournit
        if(currentApplicationConfiguration == null) {
            // creation
            currentApplicationConfiguration = new Configuration();
        
            if (currentHibernateConfigurationFile != null ) {
                logger.debug("Loading configuration file : " + currentHibernateConfigurationFile);
                
                // chargement via l'objet configuration dhibernate
                currentApplicationConfiguration.configure(currentHibernateConfigurationFile);
            }
            else {
                logger.debug("Loading configuration file : default hibernate configuration file");
                
                // chargement via l'objet configuration dhibernate
                currentApplicationConfiguration.configure();
            }
        }
        else {
            // log
            logger.debug("Configuration given, nothing to load");
        }
    }
    
    /**
     * Verifie si les information indispensable à la migration ont été
     * renseignee.
     * 
     * @throws MigrationServiceException
     */
    protected void checkInformation() throws MigrationServiceException {
        // check that version is set
        if(currentApplicationVersion == null) {
            throw new MigrationServiceException("No version set");
        }
        
        // check that shema location is set
        if(mappingsDirectory == null) {
            throw new MigrationServiceException("No old mapping directory set");
        }
    }
    
    /* (non-Javadoc)
     * @see org.nuiton.topia.migration.TopiaMigrationService#migrateSchema()
     */
    @Override
    public boolean migrateSchema() throws MigrationServiceException {
        
        // log
        logger.info("Starting Topia Migration Service");
        
        // check informations
        checkInformation();

        // chargement de la configuration de l'application
        loadApplicationConfiguration();
        
        // initie un DatabaseManager
        // fournit les propietes de connection a la base (properties)
        DatabaseManager dbManager = new DatabaseManager(currentApplicationConfiguration.getProperties());
        
        // recupere la version de la base
        Version vdbVersion = dbManager.getDataBaseVersion();
        
        // si la version n'a pas ete trouvee
        if(vdbVersion == null) {
            // la base dans ce cas n'est pas versionee.
            // On dit que la version de la base est 0
            // et les schema de cette version 0 doivent
            // etre detenu en local
            vdbVersion = Version.VZERO;
            
            logger.info("Database version not found, so database schema is considered as V0");
        }
        
        logger.info("Application version : " + currentApplicationVersion.getVersion() + ", database version : " + vdbVersion.getVersion());
        
        // tel if migration is needed
        boolean bMigrationNeeded = false;
        // tel if migration is wanted
        MigrationChoice bMigrationWanted = MigrationChoice.NO_MIGRATION;
        
        // test if schema exist in database...
        // if not, the schema must be created
        // and it will be created in version this.currentApplicationConfiguration
        Configuration vdbConfiguration = getSingleConfiguration(vdbVersion);
        vdbConfiguration.setProperties(currentApplicationConfiguration.getProperties());
        bMigrationNeeded = dbManager.isSchemaExist(vdbConfiguration);
        if(logger.isDebugEnabled()) {
            if(bMigrationNeeded) {
                logger.debug("Schema for version " + vdbVersion.getVersion() + " found. Can do migration.");
            } else {
                logger.debug("Schema for version " + vdbVersion.getVersion() + " not found. No migration needed.");
            }
        }
        
        // vdbVersion < currentApplicationVersion
        if(bMigrationNeeded && vdbVersion.compareTo(currentApplicationVersion) < 0) {
            
            logger.info("Database need update");
            
            bMigrationNeeded = true;
            // ask handler for migration
            bMigrationWanted = askHandlerForMigration(dbManager, vdbVersion.getVersion(),currentApplicationVersion.getVersion());
            
            logger.info("Handler choose : " + bMigrationWanted);
        }
        else {
            bMigrationNeeded = false;
            logger.info("Database is up to date, no migration needed.");
        }
        
        // si la migration doit etre faite
        if(bMigrationNeeded && bMigrationWanted.equals(MigrationChoice.MIGRATION)) {
            
            logger.info("Beginning database migration");
            
            // ici, on charge toutes les configuration, entre > vdbVersion et < currentApplicationVersion
            Map<Version, Configuration> mVersionAndConfigurationMap = loadIntermediateConfigurations(vdbVersion);
            
            // vdbVersion mapping has been loaded earlier
            // on construit les ConfigurationAdpater
            mVersionAndConfigurationMap.put(vdbVersion, vdbConfiguration);
            
            // Les configurationAdpater pour le kernel
            SortedMap<Version,ConfigurationAdapter> smVersionAndConfigurationAdapterMap = new TreeMap<Version,ConfigurationAdapter>();
            
            // les configurations sont chargees
            // on doit :
            // - pour la version vdbVersion, on utilise les tables deja en base
            // - pour les autres, creer les tables (suffixees avec la version)
            // - creation du schema courant
            logger.debug("Set old database for old mappings");
            
            // en meme temps, on construit les ConfigurationAdapter pour le noyau
            for(Map.Entry<Version,Configuration> entry : mVersionAndConfigurationMap.entrySet()) {
                Version vVersion = entry.getKey();
                Configuration cConfiguration = entry.getValue();
                
                // la version vdbVersion a deja ses proprietes et ne doit pas etre renommee
                if(!vdbVersion.equals(vVersion)) {
                    //ConfigurationHelper.getConfigurationForVersion(v)
                    // ne positionne pas les properties parce qu'elle n'en a pas connaissance
                    // on les met ici
                    cConfiguration.setProperties(currentApplicationConfiguration.getProperties());
                    
                    // renommage des table
                    // et creation des schema intermediaires
                    cConfiguration = dbManager.setRenamedTableSchema(cConfiguration,vVersion);
                    logger.debug("Creating schema for version : " + vVersion.getVersion());
                    dbManager.setApplicationSchemaInDatabase(cConfiguration);
                }
                
                // on construit les ConfigurationAdpater
                ConfigurationAdapter cfgAdpater = new ConfigurationAdapter(cConfiguration,vVersion);
                smVersionAndConfigurationAdapterMap.put(vVersion, cfgAdpater);
            }
            
            // enfin, il reste la configuration de l'application
            // on va instancier le nouveau schema (le creer)
            
            // on renomme le nom des tables d'abord
            currentApplicationConfiguration = dbManager.setRenamedTableSchema(currentApplicationConfiguration, currentApplicationVersion);
            
            logger.debug("Creating current application schema");
            dbManager.setApplicationSchemaInDatabase(currentApplicationConfiguration);
            
            ConfigurationAdapter appCfgAdpater = new ConfigurationAdapter(currentApplicationConfiguration, currentApplicationVersion);
            smVersionAndConfigurationAdapterMap.put(currentApplicationVersion, appCfgAdpater);
            
            logger.info("Data migration");
            
            // Ici, on a l'ancien schema deja present en base
            // les schemas intermediaires creer et vides
            // et le nouveau schema cree et vide
            // on doit maintenant migrer les donnees

            // execute la transformation
            Transformer trans = new Transformer(smVersionAndConfigurationAdapterMap);
            
            // migrate data
            trans.execute();
            
            // log
            logger.info("Data migrated");
            
            logger.debug("Deleting old database");
            
            // suppresion des anciennes tables de toutes les configuration, sauf
            // currentApplicationVersion
            // (elle n'est pas dans mVersionAndConfigurationMap)
            for(Configuration cfg : mVersionAndConfigurationMap.values()) {
                dbManager.removeTablesFromOldMapping(cfg);
            }
            
            // renommage correct du schema courant
            dbManager.renameTables(currentApplicationConfiguration, currentApplicationVersion);
            
            // il faudrait ici valider les transactions et fermer les sessions
            //  vmvManager a sa propre gestion des transactions/session
            //  this.remoteConfiguration doit en avoir ouverte
            //  this.localConfiguration aussi
            
            // all done
            logger.info("All done, migration complete");
            
            // ferme la connexion a la base
            dbManager.disconnect();
        }
        else {
            // ferme la connexion a la base
            dbManager.disconnect();
        }
        
        // manage no migration, but shema version here
        if(bMigrationNeeded && 
                ( bMigrationWanted.equals(MigrationChoice.MIGRATION) ||
                        bMigrationWanted.equals(MigrationChoice.CUSTOM_MIGRATION))) {
            
            // put version
            logger.info("Set application version in database to " + currentApplicationVersion);
            
            // put version in databse
            putVersionInDatabase(currentApplicationConfiguration.getProperties(),currentApplicationVersion,vdbVersion.equals(Version.VZERO));
        }
        
        // return succes flag
        // - no migration needed
        // - or migration needed and accepted
        return !bMigrationNeeded || (bMigrationNeeded &&
                ( bMigrationWanted.equals(MigrationChoice.MIGRATION) ||
                        bMigrationWanted.equals(MigrationChoice.CUSTOM_MIGRATION)));
    }

    /**
     * Put version in database
     * 
     * Single method because, version can be created alone...
     * 
     * @param properties proprietes de connexion
     * @param version version
     * @param createTable
     */
    protected void putVersionInDatabase(Properties properties, Version version, boolean createTable) {
        
        DatabaseManager dbManager = new DatabaseManager(properties);
        
        // update version even if database has not been migrated
        // only case that database doesn't exist match this
        if(createTable) {
            // si la base n'etait pas versionnee, la table version n'existe pas
            // creation
            dbManager.createVersionTable();
        }
        
        // Changement de la version en base
        dbManager.putVersionInDatabase(version);
        
        dbManager.disconnect();
        
    }

    /**
     * Ask handler for migration.
     * 
     * Return true if all handler return true, or if there is no handler
     * 
     * @param dbManager 
     * @param databaseVersion
     * @param applicationVersion
     * @return <tt>true</tt> or <tt>false</tt>
     */
    protected MigrationChoice askHandlerForMigration(DatabaseManager dbManager, 
            String databaseVersion, String applicationVersion) {
        
        // true par defaut, s'il n'y a pas de handlers
        MigrationChoice result = MigrationChoice.MIGRATION;
        
        for(MigrationCallbackHandler callback : migrationCallBackHandlers) {
            MigrationChoice thiscallbackResult = callback.doMigration(dbManager, 
                    databaseVersion, applicationVersion);
            
            // hack , si un des callback repond CUSTOM_MIGRATION
            // ca sera CUSTOM_MIGRATION
            
            if(thiscallbackResult == MigrationChoice.NO_MIGRATION) {
                if(!result.equals(MigrationChoice.CUSTOM_MIGRATION)) {
                    result = MigrationChoice.NO_MIGRATION;
                }
            } else if(thiscallbackResult == MigrationChoice.CUSTOM_MIGRATION) {
                result = MigrationChoice.CUSTOM_MIGRATION;
            }
            else if(thiscallbackResult == MigrationChoice.MIGRATION) {
                if(!result.equals(MigrationChoice.CUSTOM_MIGRATION)) {
                    result = MigrationChoice.MIGRATION;
                }
            }
        }
        
        return result;
    }

    /**
     * Charge les configurations de version a partir de vdbVersion "non compris"
     * jusqu'a currentApplicationVersion "non compris"
     * @param vdbVersion la version de depart
     * @return
     */
    protected Map<Version, Configuration> loadIntermediateConfigurations(Version vdbVersion) {
        // schema des noms de dossier de version
        Pattern pattern = Pattern.compile(mappingsDirectory + File.separator + "([0-9]+(\\.[0-9]+)*)");
        
        // instancie la map ordonee
        Map<Version, Configuration> mVersionAndConfigurationMap = null;
        
        List<URL> urls = null;
        // Don't use File.separator, don't work on windows
        // EC-20090714 : fix class loader for maven tomcat launch
        urls = Resource.getURLs(".*" + mappingsDirectory + "/.*", (URLClassLoader)MigrationServiceImpl.class.getClassLoader());
        
        if (urls != null && urls.size() > 0) {
            
            mVersionAndConfigurationMap = new HashMap<Version, Configuration>();
            
            // ensemble ordonnee des version a charger apres
            TreeSet<Version> tsEnsembleVersionACharger = new TreeSet<Version>();
            
            for(URL url : urls) {
                Matcher matcher = pattern.matcher(url.getFile());
                    
                if(matcher.find()) {
                    // group(1) est ce qui match entre le premier niveau de parentheses
                    String sVersion = matcher.group(1); 
                    //logger.debug("Directory " + fileInIt.getName() + " matches, version = " + sVersion);
                        
                    tsEnsembleVersionACharger.add(new Version(sVersion));
                }
            }

            // charge les version qui conviennent
            for(Version v : tsEnsembleVersionACharger) {
                if(v.compareTo(vdbVersion) <= 0) {
                    logger.debug("No load needed for version " + v.getVersion());
                }
                else {
                    logger.debug("Loading mapping for version " + v.getVersion());
                    
                    Configuration cfgForVersion = getSingleConfiguration(v);
                    mVersionAndConfigurationMap.put(v, cfgForVersion);
                }
            }
        }
        else {
            logger.error("No mapping found in classpath '" + mappingsDirectory + "'; can't load old mappings");
        }
        
        return mVersionAndConfigurationMap;
    }
    
    /**
     * Recupere une configuration sur disque pour une version.
     * 
     * @param version version
     * @return une configuration hibernate
     */
    protected Configuration getSingleConfiguration(Version version) {
        // Don't use File.separator, don't work on windows
        String mappingVersionDir = mappingsDirectory + "/" + version.getVersion();
        
        ConfigurationHelper cfgHelper = ConfigurationHelper.getInstance();
        Configuration cfgForVersion = cfgHelper.getConfigurationInDirectory(mappingVersionDir);
        
        return cfgForVersion;
    }
}
