/*
 * #%L
 * ToPIA :: Persistence
 * 
 * $Id: DeleteEntityTest.java 2852 2013-10-30 17:42:49Z athimel $
 * $HeadURL: http://svn.nuiton.org/svn/topia/tags/topia-3.0-alpha-4/topia-persistence/src/test/java/org/nuiton/topiatest/deletetest/DeleteEntityTest.java $
 * %%
 * Copyright (C) 2004 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */


/**
 * DeleteEntityTest.java
 *
 * Created: 4 juin 2009
 *
 * @author Florian Desbois <fdesbois@codelutin.com>
 * @version $Revision: 2852 $
 *
 * Mise a jour: $Date: 2013-10-30 18:42:49 +0100 (Wed, 30 Oct 2013) $
 * par : $Author: athimel $
 */

package org.nuiton.topiatest.deletetest;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.junit.Assert;
import org.junit.Rule;
import org.junit.Test;
import org.nuiton.topia.TopiaDatabase;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.TopiaTestDAOHelper;
import org.nuiton.topia.TopiaTestTopiaPersistenceContext;
import org.nuiton.topiatest.Gender;
import org.nuiton.topiatest.Personne;
import org.nuiton.topiatest.PersonneDAO;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;

/**
 * Deleting tests with DAO and Entities generated with ToPIA (diagram
 * delete-test in topiatest.zargo). Different case of deleting, with inheritance
 * or NMultiplicity relationship between two entities, or both. Initiate from an
 * issue with DAOAbstractGenerator delete method generation. Tests with H2
 * Database. Configuration in src/test/resources/TopiaContextImpl.properties
 */
public class DeleteEntityTest {

    private static final Log log = LogFactory.getLog(DeleteEntityTest.class);

    @Rule
    public final TopiaDatabase db = new TopiaDatabase();

    /**
     * Test for deleting entities with inheritance. Delete from the DAO linked
     * with the subclass entity and from the DAO linked with the superclass
     * entity. In the test model, the two entities have NMultiplicity
     * relationship without association class entity.
     *
     * @throws TopiaException if any exception while manipulating db
     */
    @Test
    public void testDeleteEntityWithInheritance() throws TopiaException {
        log.debug("START TEST : testDeleteEntityWithInheritance");

        TopiaTestTopiaPersistenceContext transaction = db.beginTransaction();

        log.debug("DAO : PersonneDAO");
        PersonneDAO dao = TopiaTestDAOHelper.getPersonneDAO(transaction);

        log.debug("CREATE PERSONNE : Bob Marley");
        Personne personne = dao.create(Personne.PROPERTY_NAME, "Bob Marley");
        transaction.commitTransaction();
        String idPersonne = personne.getTopiaId();
        assertNotNull(idPersonne);
        log.debug("ENTITY PERSONNE SAVED !");

        log.debug("DELETE PERSONNE");
        dao.delete(personne);
        transaction.commitTransaction();
        Personne res = dao.forTopiaIdEquals(idPersonne).findAnyOrNull();
        assertNull(res);
        log.debug("ENTITY PERSONNE DELETED !");

        log.debug("CREATE PERSONNE : Ziggy Marley");
        Personne personne2 = dao.create(Personne.PROPERTY_NAME, "Ziggy Marley");
        transaction.commitTransaction();
        String idPersonne2 = personne2.getTopiaId();
        assertNotNull(idPersonne2);
        log.debug("ENTITY PERSONNE SAVED !");

        log.debug("DAO parent (abstract) : PartyDAO");
        Party2DAO dao2 = TopiaTestDAOHelper.getParty2DAO(transaction);

        log.debug("DELETE PERSONNE with PartyDAO");
        dao2.delete(personne2);
        transaction.commitTransaction();
        Party2 res2 = dao2.forTopiaIdEquals(idPersonne2).findAnyOrNull();
        assertNull(res2);
        log.debug("ENTITY PERSONNE DELETED !");


    }

    /**
     * Test for deleting entities with NMultiplicity relation without
     * association class entity. Test DAO generation for deleting references
     * between two entities with NMultiplicity relation. In the test model, the
     * two entities have both inheritance.
     *
     * @throws TopiaException if any exception while manipulating db
     */
    @Test
    public void testDeleteEntityWithManyToManyRelation() throws TopiaException {
        log.debug("START TEST : testDeleteEntityWithManyToManyRelation");

        TopiaTestTopiaPersistenceContext transaction = db.beginTransaction();

        PersonneDAO dao = TopiaTestDAOHelper.getPersonneDAO(transaction);

        log.debug("CREATE PERSONNE : Bob Marley");
        Personne personne = dao.create(Personne.PROPERTY_NAME, "Bob Marley");
        transaction.commitTransaction();
        String idPersonne = personne.getTopiaId();
        assertNotNull(idPersonne);
        log.debug("ENTITY PERSONNE SAVED !");

        Contact2DAO contactDAO = TopiaTestDAOHelper.getContact2DAO(transaction);

        log.debug("CREATE CONTACT : jaja@codelutin.com");
        Contact2 contact = contactDAO.create(Contact2.PROPERTY_CONTACT_VALUE, "jaja@codelutin.com");
        transaction.commitTransaction();
        String idContact = contact.getTopiaId();
        assertNotNull(idContact);
        log.debug("ENTITY CONTACT SAVED !");

        log.debug("ADD CONTACT TO PERSONNE");
        personne.addContacts(contact);
        transaction.commitTransaction();
        assertEquals(1, personne.getContacts().size());
        log.debug("CONTACT ADDED !");

        log.debug("DELETE PERSONNE");
        dao.delete(personne);
        transaction.commitTransaction();
        Personne res = dao.forTopiaIdEquals(idPersonne).findAnyOrNull();
        assertNull(res);
        log.debug("ENTITY PERSONNE DELETED !");

        assertEquals(0, contact.getParty2().size());

        log.debug("DELETE CONTACT");
        contactDAO.delete(contact);
        transaction.commitTransaction();
        Contact2 res2 = contactDAO.forTopiaIdEquals(idContact).findAnyOrNull();
        assertNull(res2);
        log.debug("ENTITY PERSONNE DELETED !");

    }

    /**
     * Test than deleting entities will modify isPersisted() result
     */
    @Test
    public void testIsPersisted() {
        log.debug("START TEST : testIsPersisted");

        TopiaTestTopiaPersistenceContext transaction = db.beginTransaction();

        PersonneDAO dao = TopiaTestDAOHelper.getPersonneDAO(transaction);

        Personne person = dao.newInstance();
        Assert.assertNull(person.getTopiaId());

        person.setName("Arno");
        person.setGender(Gender.MALE);
        Assert.assertFalse(person.isPersisted());

        Personne person2 = dao.create(person);
        Assert.assertTrue(person.isPersisted());
        Assert.assertTrue(person2.isPersisted());

        dao.delete(person2);
        Assert.assertFalse(person.isPersisted());
        Assert.assertFalse(person2.isPersisted());
    }

}
