/*
 * #%L
 * ToPIA :: Persistence
 * 
 * $Id: DAOHelperTransformer.java 2432 2012-04-06 07:26:42Z sletellier $
 * $HeadURL: http://svn.nuiton.org/svn/topia/tags/topia-2.6.11/topia-persistence/src/main/java/org/nuiton/topia/generator/DAOHelperTransformer.java $
 * %%
 * Copyright (C) 2004 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.topia.generator;

import java.util.Set;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.eugene.models.object.ObjectModelType;
import org.nuiton.eugene.java.ObjectModelTransformerToJava;
import org.nuiton.eugene.models.object.*;
import org.nuiton.eugene.models.object.xml.ObjectModelAttributeImpl;
import org.nuiton.eugene.models.object.xml.ObjectModelEnumerationImpl;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.TopiaRuntimeException;
import org.nuiton.topia.framework.TopiaContextImplementor;
import org.nuiton.topia.persistence.TopiaDAO;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.topia.persistence.TopiaEntityEnum;
import org.nuiton.topia.persistence.util.EntityOperator;
import org.nuiton.topia.persistence.util.EntityOperatorStore;

import java.lang.reflect.Array;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;


/*{generator option: parentheses = false}*/

/*{generator option: writeString = +}*/

/**
 * Created: 13 nov. 2009 09:05:17
 *
 * @author tchemit <chemit@codelutin.com>
 * @version $Id: DAOHelperTransformer.java 2432 2012-04-06 07:26:42Z sletellier $
 * @since 2.3.0
 * @plexus.component role="org.nuiton.eugene.Template" role-hint="org.nuiton.topia.generator.DAOHelperTransformer"
 */
public class DAOHelperTransformer extends ObjectModelTransformerToJava {

    private static final Log log =
            LogFactory.getLog(DAOHelperTransformer.class);

    @Override
    public void transformFromModel(ObjectModel model) {
        String packageName =
                getOutputProperties().getProperty(PROP_DEFAULT_PACKAGE);
        String modelName = model.getName();
        String daoHelperClazzName = modelName + "DAOHelper";
        String entityEnumName = modelName + "EntityEnum";

        List<ObjectModelClass> classes =
                TopiaGeneratorUtil.getEntityClasses(model, true);

        boolean generateOperator =
                TopiaGeneratorUtil.shouldGenerateOperatorForDAOHelper(model);

        boolean generateStandaloneEnum =
                TopiaGeneratorUtil.shouldGenerateStandaloneEnumForDAOHelper(model);

        ObjectModelClass daoHelper = createClass(daoHelperClazzName,
                                                 packageName);

        ObjectModelEnumeration entityEnum;

        if (generateStandaloneEnum) {
            if (log.isDebugEnabled()) {
                log.debug("Will generate standalone " + entityEnumName +
                     " in package " + packageName);
            }
            entityEnum = createEnumeration(entityEnumName, packageName);
            addImport(entityEnum, TopiaEntity.class);
            addImport(entityEnum, EntityOperatorStore.class);
            addImport(entityEnum, Arrays.class);
            addImport(entityEnum, TopiaRuntimeException.class);
            addImport(entityEnum, ArrayUtils.class);

        } else {
            entityEnum = (ObjectModelEnumerationImpl)
                    addInnerClassifier(daoHelper,
                                       ObjectModelType.OBJECT_MODEL_ENUMERATION,
                                       entityEnumName
                    );
            addImport(daoHelper, TopiaRuntimeException.class);
            addImport(daoHelper, TopiaEntityEnum.class);
            addImport(daoHelper, EntityOperatorStore.class);
            addImport(daoHelper, Arrays.class);
            addImport(daoHelper, ArrayUtils.class);
        }

        // generate DAOHelper
        createDAOHelper(model,
                        daoHelper,
                        daoHelperClazzName,
                        entityEnumName,
                        generateOperator,
                        classes
        );

        // generate TopiaEntityEnum
        createEntityEnum(entityEnum,
                         daoHelperClazzName,
                         entityEnumName,
                         generateOperator,
                         generateStandaloneEnum,
                         classes
        );
    }

    protected void createDAOHelper(ObjectModel model,
                                   ObjectModelClass daoHelper,
                                   String daoHelperClazzName,
                                   String entityEnumName,
                                   boolean generateOperator,
                                   List<ObjectModelClass> classes) {

        String modelName = model.getName();
        String modelVersion = model.getVersion();

        addImport(daoHelper, TopiaContextImplementor.class);
        addImport(daoHelper, TopiaDAO.class);
        addImport(daoHelper, TopiaEntity.class);
        addImport(daoHelper, TopiaContext.class);
        addImport(daoHelper, Array.class);

        if (generateOperator) {
            addImport(daoHelper,EntityOperator.class);
            addImport(daoHelper, EntityOperatorStore.class);
        }

        // add non public constructor
        ObjectModelOperation constructor =
                addConstructor(daoHelper, ObjectModelModifier.PROTECTED);
        setOperationBody(constructor," ");

        ObjectModelOperation op;

        // getModelVersion method
        op = addOperation(daoHelper, "getModelVersion", "String", ObjectModelModifier.PUBLIC, ObjectModelModifier.STATIC);
        setOperationBody(op, ""
/*{
        return "<%=modelVersion%>";
    }*/
        );

        // getModelName method
        op = addOperation(daoHelper, "getModelName", "String", ObjectModelModifier.PUBLIC, ObjectModelModifier.STATIC);
        setOperationBody(op, ""
/*{
        return "<%=modelName%>";
    }*/
        );


        for (ObjectModelClass clazz : classes) {
            String clazzName = clazz.getName();
            String daoClazzName = clazzName + "DAO";

            // specialized getXXXDao method
            op = addOperation(daoHelper, "get" + daoClazzName, clazz.getPackageName() + '.' + daoClazzName, ObjectModelModifier.PUBLIC, ObjectModelModifier.STATIC);
            addParameter(op, TopiaContext.class, "context");
            addImport(daoHelper, clazz);
            addException(op, TopiaException.class);
            setOperationBody(op, ""
/*{
        TopiaContextImplementor ci = (TopiaContextImplementor) context;
        <%=daoClazzName%> result = ci.getDAO(<%=clazzName%>.class, <%=daoClazzName%>.class);
        return result;
    }*/
            );

        }

        // generic getDao method
        op = addOperation(daoHelper, "getDAO", "<T extends TopiaEntity, D extends TopiaDAO<? super T>> D", ObjectModelModifier.PUBLIC, ObjectModelModifier.STATIC);
        addParameter(op, TopiaContext.class, "context");
        addParameter(op, "Class<T>", "klass");
        addException(op, TopiaException.class);
        setOperationBody(op, ""
/*{
        TopiaContextImplementor ci = (TopiaContextImplementor) context;
        <%=entityEnumName%> constant = <%=entityEnumName%>.valueOf(klass);
        D dao = (D) ci.getDAO(constant.getContract());
        return dao;
    }*/
        );

        op = addOperation(daoHelper, "getDAO", "<T extends TopiaEntity, D extends TopiaDAO<? super T>> D", ObjectModelModifier.PUBLIC, ObjectModelModifier.STATIC);
        addParameter(op, TopiaContext.class, "context");
        addParameter(op, "T", "entity");
        addException(op, TopiaException.class);
        setOperationBody(op, ""
/*{
        TopiaContextImplementor ci = (TopiaContextImplementor) context;
        <%=entityEnumName%> constant = <%=entityEnumName%>.valueOf(entity);
        D dao = (D) ci.getDAO(constant.getContract());
        return dao;
    }*/
        );

        // getContractClass method
        op = addOperation(daoHelper, "getContractClass", "<T extends TopiaEntity> Class<T>", ObjectModelModifier.PUBLIC, ObjectModelModifier.STATIC);
        addParameter(op, "Class<T>", "klass");
        setOperationBody(op, ""
/*{
        <%=entityEnumName%> constant = <%=entityEnumName%>.valueOf(klass);
        return (Class<T>) constant.getContract();
    }*/
        );

        // getImplementationClass method
        op = addOperation(daoHelper, "getImplementationClass", "<T extends TopiaEntity> Class<T>", ObjectModelModifier.PUBLIC, ObjectModelModifier.STATIC);
        addParameter(op, "Class<T>", "klass");
        setOperationBody(op, ""
/*{
        <%=entityEnumName%> constant = <%=entityEnumName%>.valueOf(klass);
        return (Class<T>) constant.getImplementation();
    }*/
        );

        // getContractClasses method
        op = addOperation(daoHelper, "getContractClasses", "Class<? extends TopiaEntity>[]", ObjectModelModifier.PUBLIC, ObjectModelModifier.STATIC);
        setOperationBody(op, ""
/*{
        <%=entityEnumName%>[] values = <%=entityEnumName%>.values();
        Class<? extends TopiaEntity>[] result = (Class<? extends TopiaEntity>[]) Array.newInstance(Class.class, values.length);
        for (int i = 0; i < values.length; i++) {
            result[i] = values[i].getContract();
        }
        return result;
    }*/
        );

        // getImplementationClasses method
        op = addOperation(daoHelper, "getImplementationClasses", "Class<? extends TopiaEntity>[]", ObjectModelModifier.PUBLIC, ObjectModelModifier.STATIC);
        setOperationBody(op, ""
/*{
        <%=entityEnumName%>[] values = <%=entityEnumName%>.values();
        Class<? extends TopiaEntity>[] result = (Class<? extends TopiaEntity>[]) Array.newInstance(Class.class, values.length);
        for (int i = 0; i < values.length; i++) {
            result[i] = values[i].getImplementation();
        }
        return result;
    }*/
        );

        // getImplementationClassesAsString method
        op = addOperation(daoHelper, "getImplementationClassesAsString", "String", ObjectModelModifier.PUBLIC, ObjectModelModifier.STATIC);
        setOperationBody(op, ""
/*{
        StringBuilder buffer = new StringBuilder();
        for (Class<? extends TopiaEntity> aClass : getImplementationClasses()) {
            buffer.append(',').append(aClass.getName());
        }
        return buffer.substring(1);
    }*/
        );

        // getContracts method
        op = addOperation(daoHelper, "getContracts", entityEnumName+"[]", ObjectModelModifier.PUBLIC, ObjectModelModifier.STATIC);
        setOperationBody(op, ""
/*{
        return <%=entityEnumName%>.values();
    }*/
        );

        if (generateOperator) {
            // getOperator method
            op = addOperation(daoHelper, "getOperator", "<T extends TopiaEntity> EntityOperator<T>", ObjectModelModifier.PUBLIC, ObjectModelModifier.STATIC);
            addParameter(op,"Class<T>","klass");
            setOperationBody(op, ""
/*{
        <%=entityEnumName%> constant = <%=entityEnumName%>.valueOf(klass);
        return EntityOperatorStore.getOperator(constant);
    }*/
            );
        }
    }

    protected void createEntityEnum(ObjectModelEnumeration entityEnum,
                                    String daoHelperClazzName,
                                    String entityEnumName,
                                    boolean generateOperator,
                                    boolean generateStandaloneEnum,
                                    List<ObjectModelClass> classes) {

        ObjectModelAttributeImpl attr;
        ObjectModelOperation op;

        addInterface(entityEnum, TopiaEntityEnum.class);

        for (ObjectModelClass clazz : classes) {
            String clazzName = clazz.getName();

            boolean withNatural = false;
            boolean withNotNull = false;
            StringBuilder naturalIdsParams = new StringBuilder();
            StringBuilder notNullParams = new StringBuilder();

            Set<ObjectModelAttribute> naturalIdsAttributes = TopiaGeneratorUtil.getNaturalIdAttributes(clazz);
            for (ObjectModelAttribute naturalIdAttributes : naturalIdsAttributes) {
                withNatural = true;
                // attribut metier
                naturalIdsParams.append(", \"").append(naturalIdAttributes.getName()).append("\"");
                if (TopiaGeneratorUtil.isNotNull(naturalIdAttributes)) {
                    withNotNull = true;
                    // attribut not-null
                    notNullParams.append(", \"").append(naturalIdAttributes.getName()).append("\"");
                }
            }

            StringBuilder params = new StringBuilder(clazzName + ".class");
            if (withNotNull) {
                params.append(", new String[]{ " + notNullParams.substring(2) + " }");
            } else {
                params.append(", ArrayUtils.EMPTY_STRING_ARRAY");
            }
            if (withNatural) {
                params.append(", ").append(naturalIdsParams.substring(2));
            }
            addLiteral(entityEnum, clazzName + '(' + params.toString() + ')');

            if (generateStandaloneEnum) {
                addImport(entityEnum, clazz);
            }
        }

        attr = (ObjectModelAttributeImpl) addAttribute(entityEnum, "contract", "Class<? extends TopiaEntity>");
        attr.setDocumentation("The contract of the entity.");

        attr = (ObjectModelAttributeImpl) addAttribute(entityEnum, "implementationFQN", "String");
        attr.setDocumentation("The fully qualified name of the implementation of the entity.");

        attr = (ObjectModelAttributeImpl) addAttribute(entityEnum, "implementation", "Class<? extends TopiaEntity>");
        attr.setDocumentation("The implementation class of the entity (will be lazy computed at runtime).");

        attr = (ObjectModelAttributeImpl) addAttribute(entityEnum, "naturalIds", "String[]");
        attr.setDocumentation("The array of property involved in the natural key of the entity.");

        attr = (ObjectModelAttributeImpl) addAttribute(entityEnum, "notNulls", "String[]");
        attr.setDocumentation("The array of not null properties of the entity.");

        // constructor
        op = addConstructor(entityEnum, ObjectModelModifier.PACKAGE);
        addParameter(op,"Class<? extends TopiaEntity >","contract");
        addParameter(op,"String[]","notNulls");
        addParameter(op,"String...","naturalIds");
        setOperationBody(op, ""
/*{
        this.contract = contract;
        this.notNulls = notNulls;
        this.naturalIds = naturalIds;
        implementationFQN = contract.getName() + "Impl";
    }*/
        );

        // getContract method
        op = addOperation(entityEnum, "getContract", "Class<? extends TopiaEntity>", ObjectModelModifier.PUBLIC);
        addAnnotation(entityEnum,op,Override.class.getSimpleName());
        setOperationBody(op, ""
/*{
        return contract;
    }*/
        );

        // getNaturalIds method
        op = addOperation(entityEnum, "getNaturalIds", "String[]", ObjectModelModifier.PUBLIC);
        addAnnotation(entityEnum,op,Override.class.getSimpleName());
        setOperationBody(op, ""
/*{
        return naturalIds;
    }*/
        );

        // isUseNaturalIds method
        op = addOperation(entityEnum, "isUseNaturalIds", "boolean", ObjectModelModifier.PUBLIC);
        addAnnotation(entityEnum,op,Override.class.getSimpleName());
        setOperationBody(op, ""
/*{
        return naturalIds.length > 0;
    }*/
        );

        // getNotNulls method
        op = addOperation(entityEnum, "getNotNulls", "String[]", ObjectModelModifier.PUBLIC);
        addAnnotation(entityEnum,op,Override.class.getSimpleName());
        setOperationBody(op, ""
/*{
        return notNulls;
    }*/
        );

        // isUseNotNulls method
        op = addOperation(entityEnum, "isUseNotNulls", "boolean", ObjectModelModifier.PUBLIC);
        addAnnotation(entityEnum,op,Override.class.getSimpleName());
        setOperationBody(op, ""
/*{
        return notNulls.length > 0;
    }*/
        );

        // getImplementationFQN method
        op = addOperation(entityEnum, "getImplementationFQN","String",ObjectModelModifier.PUBLIC);
        addAnnotation(entityEnum,op,Override.class.getSimpleName());
        setOperationBody(op, ""
/*{
        return implementationFQN;
    }*/
        );

        // setImplementationFQN method
        op = addOperation(entityEnum, "setImplementationFQN","void",ObjectModelModifier.PUBLIC);
        addAnnotation(entityEnum,op,Override.class.getSimpleName());
        addParameter(op,"String","implementationFQN");
        if (generateOperator) {
               setOperationBody(op, ""
/*{
        this.implementationFQN = implementationFQN;
        implementation = null;
        // reinit the operators store
        EntityOperatorStore.clear();
    }*/
            );
        } else {
            setOperationBody(op, ""
/*{
        this.implementationFQN = implementationFQN;
        this.implementation = null;
    }*/
            );
        }

        // accept method
        op = addOperation(entityEnum, "accept","boolean",ObjectModelModifier.PUBLIC);
        addAnnotation(entityEnum,op,Override.class.getSimpleName());
        addParameter(op,"Class<? extends TopiaEntity>","klass");
        setOperationBody(op, ""
/*{
        return <%=daoHelperClazzName%>.getContractClass(klass) == contract;
    }*/
        );

        // getImplementation method
        op = addOperation(entityEnum, "getImplementation","Class<? extends TopiaEntity>",ObjectModelModifier.PUBLIC);
        addAnnotation(entityEnum,op,Override.class.getSimpleName());
        setOperationBody(op, ""
/*{
        if (implementation == null) {
        try {
                implementation = (Class<? extends TopiaEntity>) Class.forName(implementationFQN);
            } catch (ClassNotFoundException e) {
                throw new TopiaRuntimeException("could not find class " + implementationFQN, e);
            }
        }
        return implementation;
    }*/
        );

        // valueOf method
        op = addOperation(entityEnum, "valueOf", entityEnumName, ObjectModelModifier.PUBLIC,ObjectModelModifier.STATIC);
        addParameter(op,"TopiaEntity","entity");
        setOperationBody(op, ""
/*{
        return valueOf(entity.getClass());
    }*/
        );

        // valueOf method
        op = addOperation(entityEnum, "valueOf", entityEnumName, ObjectModelModifier.PUBLIC,ObjectModelModifier.STATIC);
        addParameter(op,"Class<?>","klass");
        setOperationBody(op, ""
/*{
        if (klass.isInterface()) {
           return valueOf(klass.getSimpleName());
        }
        for (<%=entityEnumName%> entityEnum : <%=entityEnumName%>.values()) {
            if (entityEnum.getContract().isAssignableFrom(klass)) {
                //todo check it works for inheritance
                return entityEnum;
            }
        }
        throw new IllegalArgumentException("no entity defined for the class " + klass + " in : " + Arrays.toString(<%=entityEnumName%>.values()));
    }*/
        );
    }
}
