/*
 * *##% 
 * ToPIA :: Persistence
 * Copyright (C) 2004 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */

package org.nuiton.topia.framework;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.RandomStringUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.persistence.TopiaDAO;
import org.nuiton.topia.persistence.TopiaEntity;

/**
 * Query HQL managment to simplify usage of
 * {@link TopiaContext#find(java.lang.String, java.lang.Object[]) }.
 *
 * TODO-FD20091224 Complete documentation of this class + JUnit Tests
 *
 * <pre>
 * This class is used to construct a HQL query and then execute it from a
 * TopiaContext. The TopiaQuery is linked to a TopiaEntity which
 * is the main element manipulated in the query. There is two parts in using
 * this class :
 * - construction of the query, using add, addFrom, addOrder, addSelect,
 *   addGroup, ...
 * - execution of the query, using executeToEntityList, executeToEntity,
 *   executeToInteger, ...
 *
 * Construction
 * ============
 *
 * This class make easier the way to construct a HQL query.
 * 
 * Example 1 :
 * -----------
 *
 * SQL :
 *       "SELECT * FROM PersonImpl WHERE firstName LIKE 'M%' AND year > 1980"
 *
 * HQL using {@link org.nuiton.topia.TopiaContext#find(java.lang.String, java.lang.Object[]) } :
 *       TopiaContext context = rootContext.beginTransaction();
 *       context.find("FROM " + Person.class.getName() + " WHERE firstName LIKE :firstName AND year > :year",
 *              "firstName", "M%", year, 1980);
 *
 * TopiaQuery : 
 *       TopiaQuery query = TopiaQuery.createQuery(Person.class).add(
 *           Person.FIRST_NAME, Op.LIKE, "M%").add(Person.YEAR, Op.GT, 1980);
 *
 * But the real advantage is when you have some parameters to test before adding
 * them to the query. With the older method, it was tidious to construct
 * and add parameters to finally use the find method from TopiaContext.
 *
 * Example 2 :
 * -----------
 *
 * HQL using {@link org.nuiton.topia.TopiaContext#find(java.lang.String, java.lang.Object[]) } :
 *       TopiaContext context = rootContext.beginTransaction();
 *
 *       String query = "FROM " + Person.class.getName();
 *       List<Object> params = new ArrayList<Object>();
 *       boolean filtered = false;
 *       // company parameter can be null
 *       if (company != null) {
 *           query += " WHERE company = :company";
 *           params.add("company");
 *           params.add(company);
 *           filtered = true;
 *       }
 *
 *       // contact paramater can be null
 *       if (contact != null) {
 *           query += filtered ? " AND " : " WHERE "
 *           query += " contact = :contact";
 *           params.add("contact");
 *           params.add(contact);
 *           filtered = true;
 *       }
 *
 *       context.find(query, params.toArray());
 *
 * Here we have only two non obligatory params, but imagine if we must have
 * almost 6 or 7 parameters like this !
 *
 * TopiaQuery :
 *       TopiaQuery query = TopiaQuery.createQuery(Person.class);
 *
 *       if (company != null) {
 *           query.add(Person.COMPANY, company);
 *       }
 *
 *       if (contact != null) {
 *           query.add(Person.CONTACT, contact);
 *       }
 *
 * Many ways to create the same query :
 * ------------------------------------
 *
 * You can use multiple different manners to create a query, it depends on the
 * complexicity. More complex is the query, more easier is to construct it.
 *
 * HQL : "FROM PersonImpl AS P WHERE (P.company IS NULL OR P.company = :company) AND P.firstName LIKE :firstName"
 *
 * Using TopiaQuery and an Alias :
 * query = TopiaQuery.createQuery(Person.class, "P");
 * 1- query.add("(P.company IS NULL OR P.company = :company") AND P.firstName LIKE :firstName")
 *         .addParam("company", company).addParam("firstName",firstName + "%");
 * 2- query.add("P.company IS NULL OR P.company = :company")
 *         .add("P.firstName LIKE :firstName").addParam("company", company)
 *         .addParam("firstName",firstName + "%");
 * 3- query.add("P.company IS NULL OR P.company = :company")
 *         .add("P.firstName", Op.LIKE, firstName + "%")
 *         .addParam("company", company);
 *
 * You can use TopiaQuery to create a subquery in an other TopiaQuery, you have
 * to use the method {@link #fullQuery() } to get the full query in HQL and give
 * it as a string in the other TopiaQuery.
 *
 * Execution
 * =========
 *
 * After construction, you can execute the query in different ways.
 *
 * Default method :
 * ----------------
 *
 * - execute : as the same result as
 * {@link org.nuiton.topia.TopiaContext#find(java.lang.String, java.lang.Object[]) }
 *
 * Depends on entity type ;
 * ------------------------
 *
 * - executeToEntity : only one result, the first one
 * - executeToEntityList : all results returned in a List
 * - executeToEntityMap : all results returned in a Map with key defined by user
 * or topiaId by default
 *
 * For aggregate :
 * ---------------
 *
 * These methods have in argument the SELECT to execute the query. The previous
 * SELECT (if defined) will not be deleted, but temporarly not used.
 *
 * - executeToInteger : for example for "SUM", "COUNT"
 * - executeToString : for example for "MAX"
 * - executeCount : directly a "count(*)"
 * - executeToObject : for other type of possible result (Long, Boolean, Double, ...)
 *
 * Property loading
 * ================
 *
 * When using Hibernate, some times, Entities linked to the main one will be
 * lazy initialized, but you want them directly when the query will be executed
 * to avoid problems when closing context. You can use the method
 * {@link #addLoad(java.lang.String[]) } to tell the TopiaQuery to load some
 * properties when executing the query. After that, you don't need to call them
 * for loading them in Hibernate.
 *
 * The syntax is the same as a property in HQL query using delegation :
 * "person.company" where person and company are entities.
 *
 * Note : loading only available on collection or entities but not property
 * on a collection of entities which must be made manually.
 *
 * For a Contact which is linked to a person (entity) and the person linked to
 * company (entity) you can add to a TopiaQuery<Contact> :
 *       query.addLoad("person.company")
 *
 * For a list of addresses (entity) in the contact you can do :
 *       query.addLoad("addresses")
 *
 * But it's not possible to do for example with meeting (entity) linked to the
 * contact and responsible (entity) linked to a meeting :
 *       query.addLoad("meetings.responsible")
 *
 * </pre>
 *
 * Created: 21 déc. 2009
 *
 * @author fdesbois
 * @version $Revision: 1814 $
 * @since 2.3.0
 *
 * Mise a jour: $Date: 2010-02-25 14:58:22 +0100 (jeu., 25 févr. 2010) $
 * par : $Author$
 */
public class TopiaQuery {

    private static final Log log = LogFactory.getLog(TopiaQuery.class);

    /** Params for HQL query **/
    protected List<Object> params;

    /** Select part of the query **/
    protected StringBuilder select;

    protected boolean distinct;

    /** From part of the query **/
    protected StringBuilder from;

    /** Where part of the query **/
    protected StringBuilder where;

    /** Order By part of the query **/
    protected StringBuilder orderBy;

    /** Group By part of the query **/
    protected StringBuilder groupBy;

    protected Integer startIndex;

    protected Integer endIndex;

    /** Used to determine if parentheses are needed for Where statement **/
    protected boolean parentheses;

    protected List<String> propertiesToLoad;

    protected String mainAlias;

    protected TopiaDAO<? extends TopiaEntity> dao;

    /**
     * Enum to simmplify using operation in query
     */
    public static enum Op {
        /** EQUALS **/
        EQ("="),
        /** GREATER THAN **/
        GT(">"),
        /** GREATER OR EQUALS **/
        GE(">="),
        /** LIKE for String manipulation **/
        LIKE("LIKE"),
        /** LESS THAN **/
        LT("<"),
        /** LESS OR EQUALS **/
        LE("<="),
        /** IS NOT NULL **/
        NOT_NULL("IS NOT NULL"),
        /** IS NULL **/
        NULL("IS NULL");

        protected String value;

        /**
         * Constructor of the Op Enum.
         *
         * @param value corresponding to the String for the query
         */
        Op(String value) {
            this.value = value;
        }

        @Override
        public String toString() {
            return this.value;
        }
    }

    public TopiaQuery() {
        parentheses = true;
    }

    /**
     * Create a TopiaQuery based on the {@code entityClass}. The from statement
     * is automatically set.
     * 
     * @param mainEntityClass
     */
    protected TopiaQuery(Class<? extends TopiaEntity> mainEntityClass) {
        this();
        setFrom(mainEntityClass);
    }

    /**
     * Create a TopiaQuery based on the {@code entityClass}. The from statement
     * is automatically set, the select statement must be necessary in some
     * case, the query will manage this case using the mainAlias by default.
     *
     * @param mainEntityClass
     * @param alias for the mainEntityClass
     */
    public TopiaQuery(Class<? extends TopiaEntity> mainEntityClass, String alias) {
        this();
        setFrom(mainEntityClass, alias);
    }

    /**
     * Create a TopiaQuery from a DAO. The main entity will be automatically
     * added to the select part of the query if it is needed.
     *
     * @param dao DAO linked to the entity to threat
     */
    public TopiaQuery(TopiaDAO<? extends TopiaEntity> dao) {
        this();
        setFrom(dao.getEntityClass());
        this.dao = dao;
    }

    /**
     * Create a TopiaQuery from a DAO with an Alias. The main entity will be 
     * automatically added to the select part of the query if it is needed.
     *
     * @param dao DAO linked to the entity to threat
     * @param alias of the main entity in the query
     */
    public TopiaQuery(TopiaDAO<? extends TopiaEntity> dao, String alias) {
        this();
        setFrom(dao.getEntityClass(), alias);
        this.dao = dao;
    }

    /**
     * Set the mainEntity in the from part of the query.
     *
     * @param mainEntityClass type of the mainEntity
     * @return the TopiaQuery
     */
    public TopiaQuery setFrom(Class<? extends TopiaEntity> mainEntityClass) {
        from = new StringBuilder(" FROM ").append(mainEntityClass.getName());
        return this;
    }

    /**
     * Set the mainEntity in the from part of the query and use an alias for
     * this mainEntity.
     *
     * @param mainEntityClass type of the mainEntity
     * @param alias for the entity in the query
     * @return the TopiaQuery
     */
    public TopiaQuery setFrom(Class<? extends TopiaEntity> mainEntityClass, String alias) {
        setFrom(mainEntityClass);
        mainAlias = alias;
        from.append(' ').append(alias);
        return this;
    }

    /**
     * Add an element to the from in the query. Used to add some other data in
     * the query or for join.
     *
     * @param str the element to add
     * @return the TopiaQuery
     */
    public TopiaQuery addFrom(String str) {
        from.append(", ").append(str);
        return this;
    }

    /**
     * Add an other entity type to the from in the query.
     *
     * @param entityClass different from the mainEntity in the DAO
     * @return the TopiaQuery
     */
    public TopiaQuery addFrom(Class<? extends TopiaEntity> entityClass) {
        return addFrom(entityClass.getName());
    }

    /**
     * Add an other entity type to the from in the query with an alias.
     *
     * @param entityClass different from the mainEntity in the DAO
     * @param alias of the entity in the query
     * @return the TopiaQuery
     */
    public TopiaQuery addFrom(Class<? extends TopiaEntity> entityClass, String alias) {
        return addFrom(entityClass.getName() + " " + alias);
    }

    /**
     * Create a TopiaQuery with entityClass initialization.
     *
     * @param <T> entity type extends TopiaEntity
     * @param entityClass Class for an entity Query
     * @return the new TopiaQuery
     * @deprecated use constructor instead :
     * {@link #TopiaQuery(org.nuiton.topia.persistence.TopiaDAO)  }
     */
    @Deprecated
    public static <T extends TopiaEntity> TopiaQuery createQuery(Class<T> entityClass) {
        return new TopiaQuery(entityClass);
    }

    /**
     * Create a TopiaQuery from a DAO.
     *
     * @param <T>  entity type in the dao extends TopiaEntity
     * @param dao DAO linked to the entity to threat
     * @return the new TopiaQuery
     * @deprecated use constructor instead :
     * {@link #TopiaQuery(org.nuiton.topia.persistence.TopiaDAO)  }
     */
    @Deprecated
    public static <T extends TopiaEntity> TopiaQuery createQuery(TopiaDAO<T> dao) {
        return new TopiaQuery(dao);
    }

    /**
     * Create a TopiaQuery with entityClass initialization and its Alias. The main entity will be automatically added to the select part of
     * the query if it is needed.
     *
     * @param <T> entity type in the dao extends TopiaEntity
     * @param entityClass Class for an entity Query
     * @param alias of the main entity in the query
     * @return the new TopiaQuery
     * @deprecated use constructor instead :
     * {@link #TopiaQuery(java.lang.Class, java.lang.String)  }
     */
    @Deprecated
    public static <T extends TopiaEntity> TopiaQuery createQuery(Class<T> entityClass, String alias) {
        return new TopiaQuery(entityClass, alias);
    }

    /**
     * Create a TopiaQuery from a DAO with an Alias. The main entity will be automatically added to the select part of
     * the query if it is needed.
     *
     * @param <T> entity type in the dao extends TopiaEntity
     * @param dao DAO linked to the entity to threat
     * @param alias of the main entity in the query
     * @return the new TopiaQuery
     * @deprecated use constructor instead :
     * {@link #TopiaQuery(org.nuiton.topia.persistence.TopiaDAO, String) java.lang.Class) } 
     */
    @Deprecated
    public static <T extends TopiaEntity> TopiaQuery createQuery(TopiaDAO<T> dao, String alias) {
        return new TopiaQuery(dao, alias);
    }

    @Override
    public String toString() {
        StringBuilder result = new StringBuilder(fullQuery()).
                append("; (PARAMS : ").
                append(getParams()).
                append("); (LIMIT : ").
                append(startIndex).
                append(", ").
                append(endIndex).
                append(')');
        return result.toString();
//        return fullQuery() + "; (PARAMS : " + getParams() + "); (LIMIT : " + startIndex + ", " + endIndex + ")";
    }

    /**
     * Get the full query.
     *
     * @return a String corresponding to the full query.
     */
    public String fullQuery() {
        StringBuilder result = new StringBuilder();
        StringBuilder selectStatement = new StringBuilder("SELECT ");
        if (distinct) {
            selectStatement.append("DISTINCT ");
        }
        if (select != null) {
            result.append(selectStatement).append(select);
        // Set default select if there is more than one table in from
        // part and main alias is defined
        } else if (StringUtils.contains(from.toString(), ',') &&
                   StringUtils.isNotEmpty(mainAlias)) {
            result.append(selectStatement).append(mainAlias);
        }
        result.append(from);
        if (where != null) {
            result.append(where);
        }
        if (groupBy != null) {
            result.append(groupBy);
        }
        if (orderBy != null) {
            result.append(orderBy);
        }
        return StringUtils.trim(result.toString());


//        String result = "";
//        String selectKeyWord = "SELECT ";
//        selectKeyWord += distinct ? " DISTINCT " : "";
//        if (select != null) {
//            result = selectKeyWord + select;
//        // Set default select if there is more than one table in from part and main alias is defined
//        } else if (from.contains(",") && !StringUtils.isEmpty(mainAlias)) {
//            result = selectKeyWord + mainAlias;
//        }
//        result += from;
//        if (where != null) {
//            result += where;
//        }
//        if (groupBy != null) {
//            result += groupBy;
//        }
//        if (orderBy != null) {
//            result += orderBy;
//        }
//        return result.trim();
    }

    /**
     * Add a HQL parameter to the Query.
     *
     * @param id identification of the param in the query
     * @param paramValue value of the param
     * @return the TopiaQuery
     */
    public TopiaQuery addParam(String id, Object paramValue) {
        getParams().add(id);
        getParams().add(paramValue);
        return this;
    }

    /**
     * Add muliple paramaters to the Query. The key of each param will be tested
     * if not already exist in the existing params list and will be renamed
     * in this case.
     *
     * @param params a list of HQL params with key and value in order.
     * @return the TopiaQuery
     * @see TopiaQuery#getValueName(java.lang.String)
     */
    public TopiaQuery addParams(List<Object> params) {
        for (int i = 0; i < params.size(); i += 2) {
            String paramName = (String)params.get(i);
            addParam(getValueName(paramName), params.get(i+1));
        }
        return this;
    }

    public List<Object> getParams() {
        if (params == null) {
            params = new ArrayList<Object>();
        }
        return params;
    }

    /**
     * Return the mainAlias set from constructor.
     *
     * @return a String or null if no alias is set
     */
    public String getMainAlias() {
        return mainAlias;
    }

    /**
     * Add a property to load when query is executed.
     * Used to avoid LazyInitializationException for property needed after
     * closing context.
     * The property is a string like those in HQL query.
     * <pre>
     * Exemples :
     * - "person.company" (Property TopiaEntity person linked to the result
     *   entity in query and company linked to person)
     *   --> calling myEntity.getPerson().getCompany();
     * - "partyRoles" (Property Collection partyRoles linked to the result
     *   entity in query)
     *   --> calling myEntity.getPartyRoles().size();
     * </pre>
     *
     * @param properties
     * @return the TopiaQuery
     */
    public TopiaQuery addLoad(String... properties) {
        getPropertiesToLoad().addAll(Arrays.asList(properties));
        return this;
    }

    protected List<String> getPropertiesToLoad() {
        if (propertiesToLoad == null) {
            propertiesToLoad = new ArrayList<String>();
        }
        return propertiesToLoad;
    }

    /**
     * Add a where element to the Query. Could be anything.
     * Parentheses are added automatically (even if there are not needed).
     *
     * @param where element to add
     * @return the TopiaQuery
     */
    public TopiaQuery add(String where) {
        if (StringUtils.isEmpty(where)) {
            return this;
        }
        if (this.where == null) {
            this.where = new StringBuilder(" WHERE ");
        } else {
            this.where.append(" AND ");
        }
        if (parentheses) {
            this.where.append('(');
        }
        this.where.append(where);
        if (parentheses) {
            this.where.append(')');
        }
        // Reinitialize parentheses boolean for next add call
        parentheses = true;
        return this;
    }

    /**
     * Add an element to the query. The parameter will be automatically added.
     * The constraint is needed to determine what type of operation it is.
     * Ex : add("boat", Op.EQ, boat) means -> boat = :boat.
     * If the paramValue is Null, the paramName will be added to the query
     * with the constraint null (IS NULL).
     *
     * @param paramName the name of the parameter in the query (attribute of
     *        the entity)
     * @param constraint the operation concerned
     * @param paramValue the value of the parameter (an other entity, a String,
     * ...)
     * @return the TopiaQuery
     */
    public TopiaQuery add(String paramName, Op constraint, Object paramValue) {
        StringBuilder result = new StringBuilder(paramName).append(' ');
        if (paramValue == null) {
            result.append(Op.NULL);            
        } else {
            String valueName = getValueName(paramName);
            result.append(constraint).append(" :").append(valueName);
            addParam(valueName, paramValue);
        }
        parentheses = false;
        return add(result.toString());
    }

    /**
     * Add an element to the query. The nullity is tested or a constraint is
     * added for that element. Ex : addNullOr("begin", Op.GT, new Date()) means
     * begin IS NULL OR begin > :begin (where :begin = new Date()).
     *
     * @param paramName the name of the parameter in the query (attribute of
     *        the entity)
     * @param constraint the operation concerned by the or
     * @param paramValue the value of the parameter (an other entity, a String,
     * ...)
     * @return the TopiaQuery
     */
    public TopiaQuery addNullOr(String paramName, Op constraint, Object paramValue) {
        String valueName = getValueName(paramName);
        StringBuilder result = 
                new StringBuilder(paramName).append(' ').append(Op.NULL).
                append(" OR ").append(paramName).append(constraint).
                append(" :").append(valueName);
        addParam(valueName, paramValue);
        return add(result.toString());
    }

    protected String getValueName(String paramName) {
        int dot = paramName.lastIndexOf('.');
        String valueName = paramName;
        if (dot != -1) {
            valueName = paramName.substring(dot+1);
        }
        if (getParams().contains(valueName)) {
            valueName = valueName + "_" + RandomStringUtils.randomAlphanumeric(4);
        }
        return valueName;
    }

    /**
     * Add an element to the query with the constraint Not null.
     *
     * @param paramName name of the parameter in the query
     * @return the TopiaQuery
     */
    public TopiaQuery addNotNull(String paramName) {
        StringBuilder result =
                new StringBuilder(paramName).append(' ').append(Op.NOT_NULL);
        return add(result.toString());
    }

    /**
     * Add an element to the query. The parameter will be automatically added.
     * The default constrainst operation is Op.EQ for EQUALS.
     * Ex : add("boat", boat) means -> boat = :boat.
     *
     * @param paramName name of the parameter in the query
     * @param paramValue value of the parameter
     * @return the TopiaQuery
     * @see TopiaQuery#add(String, TopiaQuery.Op, Object)
     */
    public TopiaQuery add(String paramName, Object paramValue) {
        return add(paramName, Op.EQ, paramValue);
    }

    /**
     * Add an element to the query with a list of different possible values.
     *
     * @param paramName name of the parameter in the query
     * @param values different values for this parameter
     * @return the TopiaQuery
     * @see #add(java.lang.String, java.util.Collection, boolean)
     */
    public TopiaQuery add(String paramName, Collection<Object> values) {
        return add(paramName, values, false);
    }

    /**
     * Add an element to the query with a list of different values. The IN key
     * word will be used to set the different values. An other constraint on
     * nullity can be set with isNull argument.
     * The element can have one of the value from the collection or can be null
     * if the isNull argument is true.
     *
     * @param paramName name of the parameter in the query
     * @param values different values for this parameter
     * @param isNull use it to test the nullity of parameter
     * @return the TopiaQuery
     * @see #add(java.lang.String)
     */
    public TopiaQuery add(String paramName, Collection<Object> values, boolean isNull) {        
        StringBuilder queryIn = new StringBuilder();
        if (!values.isEmpty()) {
            queryIn.append(paramName).append(" IN (");
            int count = 1;
            for (Object value : values) {
                String valueName = getValueName(paramName + count);
                if (count != 1) {
                    queryIn.append(", ");
                }
                queryIn.append(':').append(valueName);
                addParam(valueName, value);
                count++;
            }
            queryIn.append(')');
        }
        if (isNull) {
            if (!values.isEmpty()) {
                queryIn.append(" OR ");
            }
            queryIn.append(paramName).append(' ').append(Op.NULL);
            //queryIn += values.isNotEmpty() ? " OR " : "";
            //queryIn += paramName + " IS NULL";
        }
        return add(queryIn.toString());
    }

    /**
     * Add a map of properties to the where clause of the query. Each property
     * will be added to the query with Op.EQ operation, the key in the map is
     * the property name, and the value is the value of the parameter in the
     * query.
     *
     * @param properties
     * @return the TopiaQuery
     */
    public TopiaQuery add(Map<String, Object> properties) {
        for (String key : properties.keySet()) {
            add(key, properties.get(key));
        }
        return this;
    }

    /**
     * Add an element to the select in the query. Depends on the result wanted
     * in execute methods. The main entity will be automatically added only if
     * an alias is initialize from constructor. If you want only this select
     * element, use {@link #setSelect(java.lang.String) }
     * method instead.
     *
     * @param select element to add
     * @return the TopiaQuery
     */
    public TopiaQuery addSelect(String... select) {
        String str = convertStringArray(select);
        // if select is the mainAlias, do nothing
        if (mainAlias != null && str.equals(mainAlias)) {
            return this;
        }
        // if select is not null, add the new element to the select
        if (this.select != null) {
            this.select.append(", ");
        // if mainAlias is not null, add it before adding the select in argument
        } else if (mainAlias != null) {
            this.select = new StringBuilder(mainAlias).append(", ");
        } else {
            this.select = new StringBuilder();
        }
        this.select.append(convertStringArray(select));
        return this;
    }

    /**
     * Set the select in the query. Depends on the result wanted in execute
     * methods.
     *
     * @param select element to set
     * @return the TopiaQuery
     */
    public TopiaQuery setSelect(String... select) {
        this.select = new StringBuilder(convertStringArray(select));
        return this;
    }

    /**
     * Add the distinct key word in the query. The result will not have multiple
     * same values.
     *
     * @return the TopiaQuery
     */
    public TopiaQuery addDistinct() {
        this.distinct = true;
        return this;
    }

    /**
     * Add an element to the order in the query. Used to add some parameters to
     * order by.
     *
     * @param order element to add
     * @return the TopiaQuery
     */
    public TopiaQuery addOrder(String... order) {
        if (orderBy == null) {
            orderBy = new StringBuilder(" ORDER BY ");
        } else {
            orderBy.append(", ");
        }
        orderBy.append(convertStringArray(order));
        return this;
    }

    public TopiaQuery addOrderDesc(String order) {
        return addOrder(order + " DESC");
    }

    /**
     * Add an element to the group of the query. Used to add some paramters to
     * group by.
     *
     * @param group element to add
     * @return the TopiaQuery
     */
    public TopiaQuery addGroup(String... group) {
        if (groupBy == null) {
            groupBy = new StringBuilder(" GROUP BY ");
        } else {
            groupBy.append(", ");
        }
        groupBy.append(convertStringArray(group));
        return this;
    }

    /**
     * Helper method for array type. Each value will be separated by a comma.
     * @param array of String
     * @return a String with values of the array separated by a comma
     */
    protected String convertStringArray(String... array) {
        StringBuilder result = new StringBuilder();
        for (String value : array) {
            result.append(", ").append(value);
        }
        String str = "";
        if (result.length() > 0) {
            str = result.substring(2);
        }
        return str;
    }

    /**
     * Limit the result of the query with startIndex and endIndex.
     *
     * @param start first index to get from the results
     * @param end last index to get from the results
     * @return the TopiaQuery
     */
    public TopiaQuery setLimit(int start, int end) {
        this.startIndex = start;
        this.endIndex = end;
        return this;
    }

    /**
     * Remove limits previously set
     * @return the TopiaQuery
     */
    public TopiaQuery resetLimit() {
        startIndex = null;
        endIndex = null;
        return this;
    }

    /**
     * Set the max results wanted for the query.
     *
     * @param max the number of elements wanted
     * @return the TopiaQuery
     */
    public TopiaQuery setMaxResults(int max) {
        return setLimit(0,max-1);
    }

    /**
     * Simple execution of the query. This method use directly the find method
     * in TopiaContext interface.
     *
     * @param transaction the TopiaContext to use for execution
     * @return a List of results
     * @throws TopiaException
     * @see org.nuiton.topia.TopiaContext#find(java.lang.String, java.lang.Object[])
     */
    public List execute(TopiaContext transaction) throws TopiaException {
        String query = fullQuery();
        if (log.isDebugEnabled()) {
            log.debug("# QUERY : " + query);
            log.debug("# PARAMS : " + Arrays.toString(params.toArray()));
        }        
        if (startIndex != null && endIndex != null) {
            return transaction.find(query, startIndex, endIndex, getParams().toArray());
        }
        return transaction.find(query, getParams().toArray());
    }

    /**
     * DAO must be defined to use this method.
     *
     * @return a List of results
     * @throws TopiaException
     * @see #execute(org.nuiton.topia.TopiaContext)
     */
    public List execute() throws TopiaException {
        validateDAO();
        return execute(dao.getContext());
    }

    /**
     * Execute the query and get a List of entity. Some properties will be
     * loaded if they are
     * prealably set using ${@link #addLoad(java.lang.String[]) }.
     *
     * @param <E>
     * @param transaction the TopiaContext to use for execution
     * @param entityClass 
     * @return a List of TopiaEntity corresponding to the entityClass in argument
     * @throws TopiaException
     * @throws ClassCastException
     */
    public <E extends TopiaEntity> List<E> executeToEntityList(TopiaContext transaction, Class<E> entityClass)
            throws TopiaException, ClassCastException {
        List res = execute(transaction);
        if (log.isTraceEnabled()) {
            log.trace("Properties to load : " + getPropertiesToLoad());
        }
        List<E> results = new ArrayList<E>();
        for (Object o : res) {
            if (o == null) {
                continue;
            }
            if (o instanceof Object[]) {
                // If it's an array, we want only the first element wich is the
                // entity wanted
                // We know that the array have at least one element
                o = ((Object[])o)[0];
            }
            if (!entityClass.isAssignableFrom(o.getClass())) {
                throw new ClassCastException(o.getClass().getName() + " can't be cast to " + entityClass.getName() + " o : " + o);
            }
            E entity = (E)o;
            // Check distinct constraint for complex query where o is firstly an
            // Object[] (potentially distinct results with existing entity to add)
            if (! (distinct && results.contains(entity)) ) {
                if (!getPropertiesToLoad().isEmpty()) {
                    loadProperties(entity);
                }
                results.add(entity);
            }
        }
        return results;
    }

    /**
     * DAO must be defined to use this method.
     *
     * @param <E>
     * @return a List of TopiaEntity corresponding to the entityClass in argument
     * @throws TopiaException
     * @throws ClassCastException
     * @see #executeToEntityList(org.nuiton.topia.TopiaContext)
     * @deprecated use dao method instead :
     * {@link TopiaDAO#findAllByQuery(TopiaQuery) }
     */
    @Deprecated
    public <E extends TopiaEntity> List<E> executeToEntityList()
            throws TopiaException, ClassCastException {
        validateDAO();
        return (List<E>)dao.findAllByQuery(this);
    }

    /**
     * Execute the query and get a Map of entity with key type in argument. Some
     * properties will be loaded if they are
     * prealably set using ${@link #addLoad(java.lang.String[]) }.
     *
     * @param <E>
     * @param <K> the type of the map key
     * @param transaction the TopiaContext to use for execution
     * @param entityClass 
     * @param keyName the property name of the key in the entity
     * @param keyClass the key class for the result map
     * @return a Map with the key type defined and the entity in value
     * @throws TopiaException
     * @throws ClassCastException
     */
    public <E extends TopiaEntity, K> Map<K, E> executeToEntityMap(TopiaContext transaction, Class<E> entityClass, String keyName, Class<K> keyClass)
            throws TopiaException, ClassCastException {

        // Use LinkedHashMap to keep insert order from list results which can be ordered
        Map<K, E> results = new LinkedHashMap<K, E>();
        List<E> list = executeToEntityList(transaction, entityClass);
        for (E elmt : list) {
            Object value = loadProperty(elmt, keyName);
            if (value != null && !keyClass.isAssignableFrom(value.getClass())) {
                throw new ClassCastException(value.getClass().getName() + " can't be cast to " + keyClass.getName());
            }
            results.put((K)value, elmt);
        }
        return results;
    }

    /**
     * DAO must be defined to use this method.
     *
     * @param <E>
     * @param <K>
     * @param keyName
     * @param keyClass
     * @return a Map with the key type defined and the entity in value
     * @throws TopiaException
     * @throws ClassCastException
     * @see #executeToEntityMap(org.nuiton.topia.TopiaContext, java.lang.String, java.lang.Class)
     * @deprecated use dao method instead :
     * {@link TopiaDAO#findAllMappedByQuery(TopiaQuery, String, Class) }
     */
    @Deprecated
    public <E extends TopiaEntity, K> Map<K, E> executeToEntityMap(String keyName, Class<K> keyClass)
            throws TopiaException, ClassCastException {
        validateDAO();
        return (Map<K, E>)dao.findAllMappedByQuery(this, keyName, keyClass);
    }

    /**
     * Execute the query and get a Map of entity with topiaId in key. Some
     * properties will be loaded if they are
     * prealably set using ${@link #addLoad(java.lang.String[]) }.
     *
     * @param <E> 
     * @param transaction the TopiaContext to use for execution
     * @param entityClass
     * @return a Map with the key type defined and the entity in value
     * @throws TopiaException
     * @throws ClassCastException
     */
    public <E extends TopiaEntity> Map<String, E> executeToEntityMap(TopiaContext transaction, Class<E> entityClass)
            throws TopiaException, ClassCastException {
        return executeToEntityMap(transaction, entityClass, TopiaEntity.TOPIA_ID, String.class);
    }

    /**
     * DAO must be defined to use this method.
     *
     * @param <E>
     * @return a Map with the key type defined and the entity in value
     * @throws TopiaException
     * @throws ClassCastException
     * @see #executeToEntityMap(org.nuiton.topia.TopiaContext)
     * @deprecated use dao method instead :
     * {@link TopiaDAO#findAllMappedByQuery(TopiaQuery) }
     */
    @Deprecated
    public <E extends TopiaEntity> Map<String, E> executeToEntityMap()
            throws TopiaException, ClassCastException {
        validateDAO();
        return (Map<String, E>)dao.findAllMappedByQuery(this);
    }

    /**
     * Execute the query and get the first result entity. Some properties will
     * be loaded if they are
     * prealably set using ${@link #addLoad(java.lang.String[]) }.
     *
     * @param <E> 
     * @param transaction the TopiaContext to use for execution
     * @param entityClass 
     * @return a TopiaEntity corresponding  to the entityClass in argument
     * @throws TopiaException
     * @throws ClassCastException
     */
    public <E extends TopiaEntity> E executeToEntity(TopiaContext transaction, Class<E> entityClass)
            throws TopiaException, ClassCastException {
        setMaxResults(1);
        List<E> results = executeToEntityList(transaction, entityClass);
        resetLimit();
        return !results.isEmpty() ? results.get(0) : null;
    }

    /**
     * DAO must be defined to use this method.
     *
     * @param <E> 
     * @return a TopiaEntity corresponding  to the entityClass in argument
     * @throws TopiaException
     * @throws ClassCastException
     * @see #executeToEntity(org.nuiton.topia.TopiaContext)
     * @deprecated use dao method instead :
     * {@link TopiaDAO#findByQuery(TopiaQuery) }
     */
    @Deprecated
    public <E extends TopiaEntity> E executeToEntity()
            throws TopiaException, ClassCastException {
        validateDAO();
        //return executeToEntity(dao.getContext());
        return (E)dao.findByQuery(this);
    }

    /**
     * Execute the query and get an Object for result.
     * The select is overriden to get only the right value for return.
     *
     * @param transaction the TopiaContext to use for execution
     * @param select the Select overriden
     * @return an Object
     * @throws TopiaException
     */
    public Object executeToObject(TopiaContext transaction, String select) throws TopiaException {
        StringBuilder oldValue = this.select;
        if (!StringUtils.isEmpty(select)) {
            setSelect(select);
        }
        Object result = null;
        setMaxResults(1);
        List results = execute(transaction);
        if (!results.isEmpty()) {
            result = results.get(0);
        }
        this.select = oldValue;
        resetLimit();
        return result;
    }

    /**
     * Execute the query and get an Integer for result. Used only for query with
     * aggration select which return a Long : COUNT, SUM ...
     * The select is overriden to get only the right value for return.
     *
     * @param transaction the TopiaContext to use for execution
     * @param select the Select overriden (ex : SUM(myParam))
     * @return an Integer
     * @throws TopiaException
     */
    public int executeToInteger(TopiaContext transaction, String select) throws TopiaException {
        Long res = (Long)executeToObject(transaction, select);
        return res != null ? res.intValue() : 0;
    }

    /**
     * DAO must be defined to use this method.
     *
     * @param select
     * @return an Integer
     * @throws TopiaException
     * @see #executeToInteger(org.nuiton.topia.TopiaContext, java.lang.String)
     */
    public int executeToInteger(String select) throws TopiaException {
        validateDAO();
        return executeToInteger(dao.getContext(), select);
    }

    /**
     * Execute the query and get a String for result. Used for query with MAX, ...
     * The select is overriden to get only the right value for return.
     *
     * @param transaction the TopiaContext to use for execution
     * @param select the Select overriden (ex : MAX(myParam))
     * @return a String
     * @throws TopiaException
     */
    public String executeToString(TopiaContext transaction, String select) throws TopiaException {
        Object res = executeToObject(transaction, select);
        return res != null ? (String)res : "";
    }

    /**
     * DAO must be defined to use this method.
     *
     * @param select
     * @return a String result
     * @throws TopiaException
     * @see #executeToString(org.nuiton.topia.TopiaContext, java.lang.String)
     */
    public String executeToString(String select) throws TopiaException {
        validateDAO();
        return executeToString(dao.getContext(), select);
    }

    /**
     * DAO must be defined to use this method.
     *
     * @param select
     * @return an Object
     * @throws TopiaException
     * @see #executeToObject(org.nuiton.topia.TopiaContext, java.lang.String)
     */
    public Object executeToObject(String select) throws TopiaException {
        validateDAO();
        return executeToObject(dao.getContext(), select);
    }

    /**
     * Execute a simple count on the query, i.e. the number of results get from
     * the query.
     *
     * @param transaction the TopiaContext to use for execution
     * @return an int corresponding to the number of result in the query
     * @throws TopiaException
     */
    public int executeCount(TopiaContext transaction) throws TopiaException {
        return executeToInteger(transaction, "COUNT(*)");
    }

    /**
     * DAO must be defined to use this method.
     *
     * @return an int corresponding to the number of result in the query
     * @throws TopiaException
     * @see #executeCount(org.nuiton.topia.TopiaContext)
     */
    public int executeCount() throws TopiaException {
        validateDAO();
        return executeCount(dao.getContext());
    }

    protected boolean validateDAO() throws TopiaException {
        if (dao == null) {
            throw new TopiaException("DAO not defined in TopiaQuery, " +
                    "can't execute it without TopiaContext");
        }
        return true;
    }

    /**
     * Load all properties for the entity.
     *
     * @param <T> type of the entity extends TopiaEntity
     * @param entity used to load properties
     * @throws TopiaException
     */
    protected <T extends TopiaEntity> void loadProperties(T entity) throws TopiaException {
        for (String prop : getPropertiesToLoad()) {
            if (log.isTraceEnabled()) {
                log.trace("load property " + prop + " ...");
            }
            List<String> str = Arrays.asList(prop.split("\\."));
            Iterator<String> it = str.iterator();
            TopiaEntity currEntity = entity;
            while (it.hasNext()) {
                String s = it.next();
                if (log.isTraceEnabled()) {
                    log.trace("Current entity : " + currEntity.getClass().getSimpleName());
                    log.trace("Current loading : " + s);
                }
                if (it.hasNext()) {
                    currEntity = loadEntityProperty(currEntity, s);
                } else {
                    loadProperty(currEntity, s);
                }
            }
        }
    }

    /**
     * Load a property of type TopiaEntity from an other entity.
     *
     * @param <T> type of the entity extends TopiaEntity
     * @param entity used to load the property
     * @param property name of the property in the entity
     * @return a TopiaEntity corresponding to the property loaded
     * @throws TopiaException
     */
    protected <T extends TopiaEntity> TopiaEntity loadEntityProperty(T entity, String property) throws TopiaException {
        return (TopiaEntity)loadProperty(entity, property);
    }

    /**
     * Load a property from an entity.
     *
     * @param <T> type of the entity extends TopiaEntity
     * @param entity used to load the property
     * @param property name of the property in the entity
     * @return an Object corresponding to the property loaded
     * @throws TopiaException
     */
    protected <T extends TopiaEntity> Object loadProperty(T entity, String property) throws TopiaException {
        try {
            Object res = PropertyUtils.getProperty(entity, property);
            if (log.isTraceEnabled()) {
                log.trace("load property '" + property + "' for '" + entity.getClass().getSimpleName() + "'");
            }
            if (res != null && Collection.class.isAssignableFrom(res.getClass())) {
                Collection list = (Collection) res;
                list.size();
            }
            return res;
        } catch (IllegalAccessException eee) {
            throw new TopiaException("Illegal access on property " + property + " from entity " + entity.getClass().getName(), eee);
        } catch (InvocationTargetException eee) {
            throw new TopiaException("Invocation error on entity " + entity.getClass().getName() + " for property " + property, eee);
        } catch (NoSuchMethodException eee) {
            throw new TopiaException("Getter method does not exist for property " + property + " from entity " + entity.getClass().getName(), eee);
        }
    }

    @Override
    protected void finalize() {
        // Clean StringBuilder statements
        select = null;
        from = null;
        where = null;
        orderBy = null;
        groupBy = null;
    }

}
