/*
 * *##% 
 * Maven helper plugin
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package org.nuiton.plugin;

import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.project.MavenProject;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.model.Resource;
import org.codehaus.plexus.util.DirectoryScanner;
import org.nuiton.io.MirroredFileUpdater;

/**
 * Base mojo with usefull methods and implementing {@link Plugin} contract.
 *
 * @author chemit
 */
public abstract class AbstractPlugin extends AbstractMojo implements Plugin {

    /**
     * Method to initialize the mojo before doing any concrete actions.
     * <p/>
     * <b>Note:</b> The method is invoked before the {@link #doAction()} method.
     * <p/>
     * @return <code>true</code> if mojo is well initialize and something has 
     * to be done in the {@link #doAction()} method, <code>false</code>
     * otherwise.
     * @throws Exception if any
     */
    protected abstract boolean init() throws Exception;

    /**
     * Do plugin action.
     * <p/>
     * The method {@link #execute()} invoke this method only and only if :
     * <ul>
     * <li>{@link #checkPackaging()} returns <code>true</code>.</li>
     * <li>method {@link #init()} returns <code>true</code>.</li>
     * </ul>
     *
     * @throws Exception if any
     */
    protected abstract void doAction() throws Exception;
    /**
     * the message to display when will skip the goal
     */
    protected String skipAfterInitMessage;

    protected AbstractPlugin() {
        this("The goal could not be initialized, will skip the goal");
    }

    protected AbstractPlugin(String skipAfterInitMessage) {
        this.skipAfterInitMessage = skipAfterInitMessage;
    }

    @Override
    public void execute() throws MojoExecutionException, MojoFailureException {
        try {

            boolean canContinue = checkPackaging();
            if (!canContinue) {
                getLog().warn("The goal is skip due to packaging '" + getProject().getPackaging() + "'");
                return;
            }

            canContinue = init();

            if (!canContinue) {
                getLog().warn(skipAfterInitMessage);
                return;
            }

            doAction();

        } catch (Exception e) {
            throw new MojoExecutionException("could not init goal " + getClass().getSimpleName() + " for reason : " + e.getMessage(), e);
        }
    }

    /**
     * Check if the project packaging is acceptable for the mojo.
     * <p/>
     * By default, accept all packaging types.
     * <p/>
     * <b>Note:</b> This method is the first instruction to be executed in
     * the {@link #execute()}.
     * <p/>
     * <b>Tip:</b> There is two method to simplify the packaging check :
     *
     * {@link #acceptPackaging(org.nuiton.plugin.Plugin.Packaging[])}
     *
     * and
     *
     * {@link #rejectPackaging(org.nuiton.plugin.Plugin.Packaging[])}
     *
     *
     * @return {@code true} if can execute the goal for the packaging of the
     * project, {@code false} otherwise.
     */
    protected boolean checkPackaging() {
        return true;
    }

    /**
     * Accept the project's packaging between some given.
     * 
     * @param packages the accepted packaging
     * @return {@code true} if the project's packagin is one of the given ones.
     */
    protected boolean acceptPackaging(Packaging... packages) {
        String projectPackaging = getProject().getPackaging();

        for (Packaging p : packages) {
            if (p.name().equals(projectPackaging)) {
                // accept packaging
                return true;
            }
        }
        // reject packaging
        return false;
    }

    /**
     * Accept the project's packaging if not in given one.
     *
     * @param packages the rejecting packagings
     * @return {@code true} if the project's packaging is not in the given ones.
     */
    protected boolean rejectPackaging(Packaging... packages) {
        String projectPackaging = getProject().getPackaging();

        for (Packaging p : packages) {
            if (p.name().equals(projectPackaging)) {
                // reject this packaging
                return false;
            }
        }
        // accept packaging
        return true;
    }

    /**
     * Copy a file to a given locationand logging.
     *
     * @param srcFile represents the file to copy.
     * @param destFile file name of destination file.
     *
     * @throws MojoExecutionException with a message if an
     *             error occurs.
     */
    public void copyFile(File srcFile, File destFile)
            throws MojoExecutionException {
        try {
            getLog().info("Copying " + srcFile.getName() + " to " + destFile);

            PluginHelper.copy(srcFile, destFile);

        } catch (Exception e) {
            throw new MojoExecutionException("Error copying from " + srcFile + " to " + destFile, e);
        }
    }

    /**
     * Write a {@code content} into the given destination file for the given
     * {@code encoding}.
     *
     * @param destFile location where to write the content
     * @param content content ot write in the file
     * @param encoding the enconding of the file
     * @throws IOException if any pb while writing the content into the file
     */
    public void writeFile(File destFile, String content, String encoding) throws IOException {
        PluginHelper.writeString(destFile, content, encoding);
    }

    /**
     * Test if a file exists and is newer than the pom file.
     *
     * @param f the file to test
     * @return <code>true</code> if file exists and is newer than the pom file,
     *         <code>false</code> otherwise.
     */
    protected boolean isFileNewerThanPomFile(File f) {
        File pomFile = getProject().getFile();
        return f.exists() && f.lastModified() > pomFile.lastModified();
    }

    /**
     * Collects some file.
     * 
     * @param includes includes
     * @param excludes excludes
     * @param roots root directories to treate
     * @param files cache of file detected indexed by their root directory
     * @param updater
     */
    protected void getFilesToTreateForRoots(String[] includes, String[] excludes, List<String> roots, Map<File, String[]> files, MirroredFileUpdater updater) {

        DirectoryScanner ds = new DirectoryScanner();
        ds.setIncludes(includes);
        if (excludes != null) {
            ds.setExcludes(excludes);

        }
        for (String src : roots) {

            File f = new File(src);
            if (!f.exists()) {
                // do nothing on a non-existent
                continue;
            }

            if (isVerbose()) {
                getLog().info("discovering java source files in root " + src);
            }

            ds.setBasedir(f);
            // scan
            ds.scan();

            // get files
            String[] tmp = ds.getIncludedFiles();

            if (tmp.length < 1) {
                // no files found
                continue;
            }

            List<String> toTreate = new ArrayList<String>();

            if (updater != null) {
                updater.setSourceDirectory(f);
            }


            for (String filePath : tmp) {
                File srcFile = new File(f, filePath);
                // check file is up-to-date
                if (updater == null || !updater.isFileUpToDate(srcFile)) {
                    toTreate.add(filePath);
                }
            }


            if (toTreate.isEmpty()) {
                // no file or all are up-to-date
                continue;
            }

            // register files
            files.put(f, toTreate.toArray(new String[toTreate.size()]));

        }
    }

    /**
     * Collect to some files with a mirror.
     *
     * @param includes includes
     * @param excludes excludes
     * @param srcDir the directory to treate
     * @param updater an updater (will give the mirrored files)
     * @return the map of mirrored files associated to their files in srcDir
     */
    protected Map<String, String> getFilesToTreate(String[] includes, String[] excludes, File srcDir, MirroredFileUpdater updater) {

        Map<String, String> result = new java.util.TreeMap<String, String>();

        DirectoryScanner ds = new DirectoryScanner();
        ds.setIncludes(includes);
        if (excludes != null) {
            ds.setExcludes(excludes);

        }

        if (!srcDir.exists()) {
            // do nothing on a non-existent
            return result;
        }

        if (isVerbose()) {
            getLog().info("discovering files for " + srcDir);
        }

        ds.setBasedir(srcDir);
        // scan
        ds.scan();

        // get files
        String[] tmp = ds.getIncludedFiles();

        if (tmp.length < 1) {
            // no files found
            return result;
        }

        List<String> toTreate = new ArrayList<String>();

        if (updater != null) {
            updater.setSourceDirectory(srcDir);
        }

        for (String filePath : tmp) {
            File srcFile = new File(srcDir, filePath);
            File mirrorFile = updater.getMirrorFile(srcFile);
            // check file is up-to-date
            if (updater == null || !updater.isFileUpToDate(srcFile)) {
                result.put(filePath, mirrorFile.getAbsolutePath());
                toTreate.add(filePath);
            }
        }


        if (toTreate.isEmpty()) {
            // no file or all are up-to-date
            return result;
        }

        // register files
        return result;
    }

    /**
     * Add a given directory in maven project's compile source roots (if not
     * already present).
     *
     * @param srcDir the location to include in compile source roots
     */
    protected void addCompileSourceRoots(File srcDir) {
        if (!getProject().getCompileSourceRoots().contains(srcDir.getPath())) {
            if (isVerbose()) {
                getLog().info("adding source roots : " + srcDir.getPath());
            }
            getProject().addCompileSourceRoot(srcDir.getPath());
        }
    }

    /**
     * Remove a given directory in maven project's compile source roots (if
     * present).
     *
     * @param srcDir the location to remove from compile source roots
     */
    protected void removeCompileSourceRoots(File srcDir) {
        if (getProject().getCompileSourceRoots().contains(srcDir.getPath())) {
            if (isVerbose()) {
                getLog().info("removing source roots : " + srcDir.getPath());
            }
            getProject().getCompileSourceRoots().remove(srcDir.getPath());
        }
    }

    /**
     * Add a given directory in maven project's test compile source roots (if not
     * already present).
     *
     * @param srcDir the location to include in test compile source roots
     */
    protected void addTestCompileSourceRoots(File srcDir) {
        if (!getProject().getTestCompileSourceRoots().contains(srcDir.getPath())) {
            if (isVerbose()) {
                getLog().info("adding test source roots : " + srcDir.getPath());
            }
            getProject().addTestCompileSourceRoot(srcDir.getPath());
        }
    }

    /**
     * Remove a given directory in maven project's test compile source roots (if
     * present).
     *
     * @param srcDir the location to remove from test compile source roots
     */
    protected void removeTestCompileSourceRoots(File srcDir) {
        if (getProject().getTestCompileSourceRoots().contains(srcDir.getPath())) {
            if (isVerbose()) {
                getLog().info("removing test source roots : " + srcDir.getPath());
            }
            getProject().getTestCompileSourceRoots().remove(srcDir.getPath());
        }
    }

    /**
     * Add a new resource location to the maven project (in not already present).
     *
     * @param dir the new resource location to add
     */
    protected void addResourceDir(String dir) {
        boolean added = PluginHelper.addResourceDir(dir, getProject());
        if (added) {
            getLog().info("add resource " + dir);
        }
    }

    /**
     * Add a new test resource location to the maven project (in not already present).
     *
     * @param dir the new resource location to add
     */
    protected void addTestResourceDir(String dir) {
        boolean added = PluginHelper.addTestResourceDir(dir, getProject());
        if (added) {
            getLog().info("add test resource " + dir);
        }
    }

    /**
     * Init mojo classLoader.
     *
     * @param project the maven project
     * @param src the source directory
     * @param addSourcesToClassPath a flag to a maven sources to classLoader
     * @param testPhase a flag to specify if we are in a test phase (changes the classLoader)
     * @param addResourcesToClassPath flag to add maven's resources to classLoader
     * @param addCompileClassPath flag to add maven's project compile classPath to classLoader
     * @param addProjectClassPath flag to add maven'es project dependecies to classLoader
     * @return the new classLoader
     * @throws MalformedURLException
     */
    @SuppressWarnings({"unchecked"})
    protected URLClassLoader initClassLoader(MavenProject project, File src, boolean addSourcesToClassPath, boolean testPhase, boolean addResourcesToClassPath, boolean addCompileClassPath, boolean addProjectClassPath) throws MalformedURLException {
        URLClassLoader loader = null;
        if (project != null) {

            URLClassLoader result;
            try {

                List<URL> lUrls = new ArrayList<URL>();
                List<String> sources = project.getCompileSourceRoots();

                if (addSourcesToClassPath) {
                    for (String source : sources) {
                        lUrls.add(new File(source).toURI().toURL());
                    }
                    if (testPhase) {
                        for (Object source : project.getTestCompileSourceRoots()) {
                            lUrls.add(new File(source.toString()).toURI().toURL());
                        }
                    }
                }
                if (addResourcesToClassPath) {
                    for (Object source : project.getResources()) {
                        Resource r = (Resource) source;
                        lUrls.add(new File(r.getDirectory()).toURI().toURL());
                    }
                }
                if (testPhase && addCompileClassPath) {
                    if (project != null) {
                        lUrls.add(new File(project.getBuild().getOutputDirectory()).toURI().toURL());
                    }
                }
                if (src != null) {
                    lUrls.add(src.toURI().toURL());
                }
                if (addProjectClassPath) {
                    getLog().info("use project compile scope class-path");
                    // add also all dependencies of the project in compile scope
                    Set<?> artifacts = project.getArtifacts();
                    for (Object o : artifacts) {
                        Artifact a = (Artifact) o;
                        lUrls.add(a.getFile().toURI().toURL());
                    }
                }

                result = new URLClassLoader(lUrls.toArray(new URL[lUrls.size()]), getClass().getClassLoader());

            } catch (IOException e) {
                throw new RuntimeException("Can't create ClassLoader for reason " + e.getMessage(), e);
            }
            loader = result;
        } else {
            List<URL> lUrls = new ArrayList<URL>();
            if (addSourcesToClassPath) {
                lUrls.add(src.toURI().toURL());
            }
            loader = new URLClassLoader(lUrls.toArray(new URL[lUrls.size()]), getClass().getClassLoader());
        }
        if (isVerbose()) {
            for (URL entry : loader.getURLs()) {
                getLog().info("classpath : " + entry);
            }
        }
        return loader;
    }

    /**
     * Obtain the url of a file, if file does not exist, try in the classPath.
     *
     * @param f the required resource file.
     * @return the url of the resource
     * @throws IOException for any error while looking up for the url of the resources
     */
    protected URL getTemplate(File f) throws IOException {
        URL r = null;
        if (f.exists()) {
            r = f.toURI().toURL();
        } else {
            r = getClass().getResource(f.toString());
        }
        return r;
    }

    /**
     * Check that the given resource exists in a simple fs file or in the classPath.
     *
     * @param f the required resource file.
     * @throws IOException for any error while looking up for the resources content.
     */
    protected void checkResource(File f) throws IOException {
        if (!f.exists()) {
            // test in classPath
            InputStream r = null;
            try {
                r = getClass().getResourceAsStream(f.toString());
                if (r == null) {
                    throw new IOException("could not find ressource " + f);
                }
            } finally {
                if (r != null) {
                    r.close();
                }
            }
        }
    }
}
