package org.nuiton.eugene.plugin.writer;

import org.nuiton.eugene.models.Model;
import org.nuiton.eugene.models.object.ObjectModel;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import java.io.File;
import java.io.IOException;

/**
 * This {@link org.nuiton.eugene.ModelFileWriter} transform xmi files to object model file.
 *
 * @author tchemit
 * @plexus.component role="org.nuiton.eugene.ModelFileWriter" role-hint="xmi2objectmodel" description="xmi to object model"
 * @since 2.0.0
 */
public class XmiToObjectModelFileWriter extends BaseXmiToModelFileWriter {


    public static final String[] XMI_FILE_FILTER = new String[]{"*.xmi", "**/*.xmi"};

    @Override
    public <M extends Model> boolean acceptModel(Class<M> modelType) {
        // accept only state models
        return ObjectModel.class.isAssignableFrom(modelType);
    }

    @Override
    public String getExtension() {
        return "objectmodel";
    }

    @Override
    protected String getStyleSheet(File model) {
        String styleSheet = null;

        String version = getXmiVersion(model);
        if (version.startsWith("1.")) {
            styleSheet = "xmi1.2ToObjectModel.xsl";
        } else if (version.startsWith("2.")) {
            styleSheet = "xmi2.1ToObjectModel.xsl";
        } else {
            getLog().error("Unsupported xmi version [" + version + "]");
        }

        return styleSheet;
    }

    /**
     * Try to find xmi version on a file.
     *
     * @param xmiFile file to inspect
     * @return version or null if version can't have been found
     */
    protected String getXmiVersion(File xmiFile) {
        String version = null;

        SAXParserFactory factory = SAXParserFactory.newInstance();

        try {
            SAXParser parser = factory.newSAXParser();

            XmiVersionHandler handler = new XmiVersionHandler();
            parser.parse(xmiFile, handler);

            version = handler.getVersion();
        } catch (ParserConfigurationException e) {
            getLog().debug("Can't parse file as xmi", e);
        } catch (SAXException e) {
            getLog().debug("Can't parse file as xmi", e);
        } catch (IOException e) {
            getLog().debug("Can't parse file as xmi", e);
        }

        return version;
    }

    /**
     * Sax handler to find xmi version into xmi document.
     */
    protected class XmiVersionHandler extends DefaultHandler {

        public String version = null;

        public XmiVersionHandler() {
            super();
        }

        public String getVersion() {
            return version;
        }

        @Override
        public void startElement(String uri, String localName, String qName,
                                 Attributes attributes) throws SAXException {

            if (qName.equals("XMI")) {
                version = attributes.getValue("xmi.version");
                getLog().debug("XMI version found : " + version);
            }

            if (version == null) {
                version = attributes.getValue("xmi:version");
                getLog().debug("XMI version found : " + version);
            }

        }
    }


}