/*
 * #%L
 * Lima :: Swing
 * %%
 * Copyright (C) 2008 - 2012 CodeLutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package org.chorem.lima.ui.importexport;

import com.google.common.base.Charsets;
import com.google.common.base.Strings;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.lima.LimaSwingConfig;
import org.chorem.lima.LimaTechnicalException;
import org.chorem.lima.business.ExportResult;
import org.chorem.lima.business.ImportExportResults;
import org.chorem.lima.business.ImportResult;
import org.chorem.lima.business.api.ExportService;
import org.chorem.lima.business.api.ImportService;
import org.chorem.lima.business.exceptions.AlreadyExistFinancialStatement;
import org.chorem.lima.business.exceptions.AlreadyExistVatStatementException;
import org.chorem.lima.business.exceptions.BeginAfterEndFiscalPeriodException;
import org.chorem.lima.business.exceptions.ImportFileException;
import org.chorem.lima.business.exceptions.InvalidAccountNumberException;
import org.chorem.lima.business.exceptions.LimaException;
import org.chorem.lima.business.exceptions.LockedEntryBookException;
import org.chorem.lima.business.exceptions.LockedFinancialPeriodException;
import org.chorem.lima.business.exceptions.MoreOneUnlockFiscalPeriodException;
import org.chorem.lima.business.exceptions.NoDataToImportException;
import org.chorem.lima.business.exceptions.NoFiscalPeriodFoundException;
import org.chorem.lima.business.exceptions.NotAllowedLabelException;
import org.chorem.lima.business.exceptions.NotBeginNextDayOfLastFiscalPeriodException;
import org.chorem.lima.business.exceptions.NotNumberAccountNumberException;
import org.chorem.lima.entity.Account;
import org.chorem.lima.entity.Entry;
import org.chorem.lima.entity.EntryBook;
import org.chorem.lima.entity.FinancialStatement;
import org.chorem.lima.entity.FinancialTransaction;
import org.chorem.lima.entity.FiscalPeriod;
import org.chorem.lima.entity.Identity;
import org.chorem.lima.entity.VatStatement;
import org.chorem.lima.enums.EncodingEnum;
import org.chorem.lima.enums.ImportExportEnum;
import org.chorem.lima.service.LimaServiceFactory;
import org.chorem.lima.util.ErrorHelper;
import org.jdesktop.swingx.painter.BusyPainter;

import javax.swing.*;
import java.awt.*;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStreamWriter;
import java.nio.charset.Charset;
import java.rmi.server.ExportException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

import static org.nuiton.i18n.I18n.t;

/**
 * Import export helper.
 * Calling service with specified file to import and displaying wait view to user.
 *
 * @author echatellier
 * Date : 24 avr. 2012
 */
public class ImportExport {

    private static final Log log = LogFactory.getLog(ImportExport.class);
    public static final int BUFFER_SIZE = 1024;

    /** Parent view. */
    protected Component viewComponent;

    protected ImportService importService;

    protected ExportService exportService;

    private ImportExportWaitView waitView;

    protected ErrorHelper errorHelper;

    public static final String JAVA_IO_TMPDIR = System.getProperty("java.io.tmpdir")+"/";

    public ImportExport(Component view) {
        viewComponent = view;

        //services
        importService = LimaServiceFactory.getService(ImportService.class);
        exportService = LimaServiceFactory.getService(ExportService.class);
        errorHelper = new ErrorHelper(LimaSwingConfig.getInstance());

        //create the wait dialog panel
        waitView = new ImportExportWaitView();
        waitView.setLocationRelativeTo(view);
        BusyPainter busyPainter = waitView.getBusylabel().getBusyPainter();
        busyPainter.setHighlightColor(new Color(44, 61, 146).darker());
        busyPainter.setBaseColor(new Color(168, 204, 241).brighter());
    }

    /**
     * Call the appropriate method in business service.
     *
     * @param importExportChoice determine what to import/export and export type (CSV, EBP, PDF)
     * @param path file path for import/export file, if null a dialog will ask for it
     * @param verbose if true a dialog show result
     */
    public void importExport(final ImportExportEnum importExportChoice, String path, boolean verbose) {
        final Charset defaultCharset = Charsets.UTF_8;
        String filePath = path;
        if (Strings.isNullOrEmpty(filePath)) {
            filePath = chooseFile(importExportChoice.getImportMode(), importExportChoice);
        }
        processImport(verbose, importExportChoice, defaultCharset, filePath);

    }

    protected void processImport(final boolean verboseMode, final ImportExportEnum importExportMethod, final Charset defaultCharset, final String filePath) {
        //if export cancel
        if (!Strings.isNullOrEmpty(filePath)) {
            final Boolean importMode = importExportMethod.getImportMode();

            new SwingWorker<ImportExportResults, Void>() {

                @Override
                protected ImportExportResults doInBackground() {
                    ImportExportResults results = new ImportExportResults();
                    switch (importExportMethod) {
                        //####################################### CSV ##############################################

                        case CSV_ACCOUNTCHARTS_EXPORT:
                            results.pushExportResults(exportService.exportAccountsAsCSV(defaultCharset.name()));
                            createFile(filePath, defaultCharset.name(), results.getExportResults().get(0).getExportData());
                            break;

                        case CSV_ACCOUNTCHARTS_IMPORT:
                            String content = loadFile(filePath, defaultCharset.name());
                            results.pushImportResults(importService.importAccountAsCSV(content));
                            break;
                        case CSV_ENTRYBOOKS_EXPORT:
                            results.pushExportResults(exportService.exportEntryBooksAsCSV(defaultCharset.name()));
                            createFile(filePath, defaultCharset.name(), results.getExportResults().get(0).getExportData());
                            break;
                        case CSV_ENTRYBOOKS_IMPORT:
                            content = loadFile(filePath, defaultCharset.name());
                            results.pushImportResults(importService.importEntryBooksAsCSV(content));
                            break;
                        case CSV_ENTRIES_EXPORT:
                            results.pushExportResults(exportService.exportEntriesAsCSV(defaultCharset.name(), true));
                            createFile(filePath, defaultCharset.name(), results.getExportResults().get(0).getExportData());
                            break;
                        case CSV_ENTRIES_IMPORT:
                            content = loadFile(filePath, defaultCharset.name());
                            results.pushImportResults(importService.importEntriesAsCSV(content));
                            break;

                        case CSV_ALL_EXPORT:
                            results = exportBackup(filePath, defaultCharset.name());
                            break;
                        case CSV_ALL_IMPORT:
                            results = importAllFromZipFile(filePath);
                            break;

                        case CSV_VAT_EXPORT:
                            results.pushExportResults(exportService.exportVatStatements(defaultCharset.name()));
                            createFile(filePath, defaultCharset.name(), results.getExportResults().get(0).getExportData());
                            break;
                        case CSV_VAT_IMPORT:
                            content = loadFile(filePath, defaultCharset.name());
                            results.pushImportResults(importService.importVATStatementsAsCSV(content));
                            break;

                        case CSV_FINANCIALSTATEMENTS_EXPORT:
                            results.pushExportResults(exportService.exportFinancialStatements(defaultCharset.name()));
                            createFile(filePath, defaultCharset.name(), results.getExportResults().get(0).getExportData());
                            break;
                        case CSV_FINANCIALSTATEMENTS_IMPORT:
                            content = loadFile(filePath, defaultCharset.name());
                            results.pushImportResults(importService.importFinancialStatementsAsCSV(content));
                            break;

                        //####################################### EBP ##############################################
                        //For windows ebp so using encoding ISOLATIN1

                        case EBP_ACCOUNTCHARTS_EXPORT:
                            results.pushExportResults(exportService.exportAccountAsEbp(defaultCharset.name()));
                            createFile(filePath, EncodingEnum.ISOLATIN1.getEncoding(), results.getExportResults().get(0).getExportData());
                            break;
                        case EBP_ACCOUNTCHARTS_IMPORT:
                            content = loadFile(filePath, EncodingEnum.ISOLATIN1.getEncoding());
                            results.pushImportResults(importService.importAccountFromEbp(content));
                            break;

                        case EBP_ENTRYBOOKS_EXPORT:
                            results.pushExportResults(exportService.exportEntryBookAsEbp(defaultCharset.name()));
                            createFile(filePath, EncodingEnum.ISOLATIN1.getEncoding(), results.getExportResults().get(0).getExportData());
                            break;
                        case EBP_ENTRYBOOKS_IMPORT:
                            content = loadFile(filePath, EncodingEnum.ISOLATIN1.getEncoding());
                            results.pushImportResults(importService.importEntryBookFromEbp(content));
                            break;

                        case EBP_ENTRIES_EXPORT:
                            results.pushExportResults(exportService.exportEntriesAsEbp(defaultCharset.name()));
                            createFile(filePath, EncodingEnum.ISOLATIN1.getEncoding(), results.getExportResults().get(0).getExportData());
                            break;
                        case EBP_ENTRIES_IMPORT:
                            content = loadFile(filePath, EncodingEnum.ISOLATIN1.getEncoding());
                            results.pushImportResults(importService.importEntriesFromEbp(content));
                            break;
                        default:
                            break;
                    }
                    return results;
                }

                @Override
                protected void done() {
                    try {

                        if (log.isDebugEnabled()) {
                            log.debug("importMode : " + importMode);
                        }

                        //hidde wait dialog panel
                        waitView.setVisible(false);

                        // display result dialog
                        if (verboseMode) {
                            ImportExportResults globalResult = get();
                            if (importMode && globalResult == null) {
                                JOptionPane.showMessageDialog(viewComponent, t("lima.import.error"),
                                        t("lima.import.title"), JOptionPane.ERROR_MESSAGE);

                            } else {
                                if (importMode) {
                                    List<ImportResult> importResults = globalResult.getImportResults();
                                    if (importResults != null) {
                                        ComputeImportResultMessage(importResults);
                                    }
                                } else {
                                    List<ExportResult> exportResults = globalResult.getExportResults();
                                    if (exportResults != null) {
                                        ComputeExportResultMessage(exportResults);
                                    }
                                }
                            }
                        }
                    } catch (InterruptedException | ExecutionException ex) {
                        if (log.isErrorEnabled()) {
                            log.error("Can't get result message", ex);
                        }
                    }
                }
            }.execute();
            waitView.setVisible(true);
        }
    }

    protected void ComputeExportResultMessage(List<ExportResult> exportResults) {
        StringBuilder message = new StringBuilder();
        for (ExportResult result : exportResults) {
            Class fromSource = result.getFromSource();
            String messageFromSource = getFromSourceMessage(fromSource);
            message.append("Export ");
            message.append(messageFromSource);

            List<ExportException> exportExceptions = result.getExportExceptions();
            if (exportExceptions != null && !exportExceptions.isEmpty()) {
                message.append(t("lima.export.exceptions") + "\n");
            }
        }

        DisplayImportExportResultMessage(message, false);
    }

    protected void ComputeImportResultMessage(List<ImportResult> resultList) {
        StringBuilder message = new StringBuilder();
        message.append(t("lima.import.terminated") + "\n");
        StringBuilder errorMessage = null;
        for (ImportResult result : resultList) {
            Class fromSource = result.getFromSource();
            message.append("Import ");
            message.append(getFromSourceMessage(fromSource));
            message.append(t("lima.import.report", result.getNbCreated(), result.getNbUpdated(), result.getNbIgnored()) + "\n");
            Map<Integer, LimaException> exceptionsByLine = result.getAllExceptionsByLine();
            errorMessage = displayErrorMessage(exceptionsByLine);
            // message is null if import failed
            if (errorMessage == null) {
                break;
            } else {
                message.append(errorMessage.toString());
            }
        }

        if (errorMessage != null) {
            DisplayImportExportResultMessage(message, true);
        }
    }

    protected void DisplayImportExportResultMessage(StringBuilder message, Boolean importMode) {
        if (log.isDebugEnabled()){
            log.debug(message.toString());
        }

        String title = importMode ? t("lima.import.terminated") : t("lima.export.terminated");
        JOptionPane.showMessageDialog(
                waitView,
                message.toString(),
                title,
                JOptionPane.INFORMATION_MESSAGE);
    }

    protected StringBuilder displayErrorMessage(Map<Integer, LimaException> exceptionsByLine) {
        StringBuilder result = new StringBuilder();
        if (exceptionsByLine != null) {
            for (Map.Entry<Integer, LimaException> e:exceptionsByLine.entrySet()) {
                result.append(t("lima.import.line", e.getKey()) + "\n");
                    LimaException importException = e.getValue();
                if (importException instanceof InvalidAccountNumberException) {
                    result.append(t("lima.account.error.invalidAccountNumber", ((InvalidAccountNumberException) importException).getAccountNumber()) + "\n");
                } else if (importException instanceof NotNumberAccountNumberException) {
                    result.append(t("lima.account.error.notNumberAccountNumber", ((NotNumberAccountNumberException) importException).getAccountNumber()) + "\n");
                } else if (importException instanceof NotAllowedLabelException) {
                    result.append(t("lima.error.notAllowedLabel", ((NotAllowedLabelException) importException).getLabel()) + "\n");
                } else if (importException instanceof MoreOneUnlockFiscalPeriodException) {
                    result.append(t("lima.fiscalPeriod.error.moreOneUnlockFiscalPeriod", ((MoreOneUnlockFiscalPeriodException) importException).getBeginDate(), ((MoreOneUnlockFiscalPeriodException) importException).getEndDate()) + "\n");
                } else if (importException instanceof BeginAfterEndFiscalPeriodException) {
                    result.append(t("lima.fiscalPeriod.error.beginAfterEndFiscalPeriod", ((BeginAfterEndFiscalPeriodException) importException).getFiscalPeriod().getBeginDate(), ((BeginAfterEndFiscalPeriodException) importException).getFiscalPeriod().getEndDate()) + "\n");
                } else if (importException instanceof NotBeginNextDayOfLastFiscalPeriodException) {
                    result.append(t("lima.fiscalPeriod.error.notBeginNextDayOfLastFiscalPeriod", ((NotBeginNextDayOfLastFiscalPeriodException) importException).getFiscalPeriod().getBeginDate(), ((NotBeginNextDayOfLastFiscalPeriodException) importException).getFiscalPeriod().getEndDate()) + "\n");
                } else if (importException instanceof LockedFinancialPeriodException) {
                    result.append(t("lima.fiscalPeriod.error.lockedFinancialPeriod") + "\n");
                } else if (importException instanceof LockedEntryBookException) {
                    result.append(t("lima.entryBook.error.lockedEntryBook", ((LockedEntryBookException) importException).getClosedPeriodicEntryBook().getFinancialPeriod().getBeginDate(), ((LockedEntryBookException) importException).getClosedPeriodicEntryBook().getFinancialPeriod().getEndDate()) + "\n");
                } else if (importException instanceof AlreadyExistFinancialStatement) {
                    result.append(t("lima.financialStatement.error.alreadyExistFinancialStatement", ((AlreadyExistFinancialStatement) importException).getFinancialStatementLabel()) + "\n");
                } else if (importException instanceof AlreadyExistVatStatementException) {
                    result.append(t("lima.vatStatement.error.alreadyExistVatStatement", ((AlreadyExistVatStatementException) importException).getVatStatementLabel()) + "\n");
                } else if (importException instanceof NoFiscalPeriodFoundException) {
                    result.append(t("lima.error.noFiscalPeriodFound") + "\n");
                } else if (importException instanceof NoDataToImportException) {
                    errorHelper.showErrorMessage(t("lima.import.error.noDataToImport"));
                    log.warn(t("lima.import.error.noDataToImport"));
                    result = null;
                    break;
                } else if (importException instanceof ImportFileException){
                    errorHelper.showErrorMessage(((ImportFileException) importException).getDetailMessage());
                    log.warn(((ImportFileException) importException).getDetailMessage());
                    result = null;
                    break;
                } else {
                    result.append(t("lima.import.error.unknown") + "\n");
                }
                try {
                    throw importException;
                } catch (LimaException ex) {
                    log.error(ex);
                }

            }
        }
        return result;
    }

    /**
     * Return with entities have been imported/exported.
     * @param fromSource The imported target class
     * @return the name of the imported/exported entities class.
     */
    protected String getFromSourceMessage(Class fromSource) {
        String message;
        if (fromSource == null) {
            message = "BACKUP"+"\n";
        } else if (fromSource.equals(Account.class)){
            message = t("lima.importExport.account")+"\n";
        } else if (fromSource.equals(EntryBook.class)) {
            message = t("lima.importExport.entryBook")+"\n";
        } else if (fromSource.equals(Entry.class)) {
            message = t("lima.importExport.entry")+"\n";
        } else if (fromSource.equals(FinancialStatement.class)) {
            message = t("lima.importExport.financialStatement")+"\n";
        } else if (fromSource.equals(FinancialTransaction.class)) {
            message = t("lima.importExport.financialTransaction")+"\n";
        } else if (fromSource.equals(FiscalPeriod.class)) {
            message = t("lima.importExport.fiscalPeriod")+"\n";
        } else if (fromSource.equals(Identity.class)) {
            message = t("lima.importExport.identity")+"\n";
        } else if (fromSource.equals(VatStatement.class)) {
            message = t("lima.importExport.vatStatement")+"\n";
        } else {
            throw new LimaTechnicalException("Source not know");
        }
        return message;
    }


    /**
     * open choose file dialog with appropriate file mode view
     * folders for export or folders+files for import
     *
     * @param importMode true = import mode false = export
     * @param importExportMethod Inport/export to process
     * @return the targeted file path
     */
    public String chooseFile(Boolean importMode, ImportExportEnum importExportMethod) {
        String filePath = "";
        JFileChooser chooser = new JFileChooser();
        //Encoding option

        JComboBox comboBox = new JComboBox<String>(EncodingEnum.descriptions());

        if (importExportMethod.getEncodingOption()) {
            JPanel panel = new JPanel();
            panel.setLayout(new BorderLayout());
            panel.add(new JLabel(t("lima.importExport.encoding.choice")), BorderLayout.WEST);
            panel.add(comboBox, BorderLayout.CENTER);
            ((Container) chooser.getComponent(2)).add(panel, BorderLayout.SOUTH);
        }

        String approveButtonText;
        if (importMode) {
            chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
            approveButtonText = t("lima.importExport.import");
            chooser.setDialogTitle(approveButtonText);
            chooser.setApproveButtonText(approveButtonText);
        } else if (importExportMethod.equals(ImportExportEnum.CSV_ALL_EXPORT)) {
            approveButtonText = t("lima.importExport.export");
            chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
            chooser.setDialogTitle(approveButtonText);
            chooser.setApproveButtonText(approveButtonText);
        } else {
            approveButtonText = t("lima.importExport.export");
            chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
            chooser.setDialogTitle(approveButtonText);
            chooser.setApproveButtonText(approveButtonText);
        }
        if (chooser.showOpenDialog(viewComponent) == JFileChooser.APPROVE_OPTION) {
            filePath = chooser.getSelectedFile().getAbsolutePath();
        }

        return filePath;
    }


    /**
     * Get csv datas in string and write file.
     *
     * @param filePath path to file
     * @param charset charset to use
     * @param data data to write on file
     */
    protected File createFile(String filePath, String charset, String data) {
        File file = new File(filePath);
        BufferedWriter out = null;
        try {
            out = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(file), charset));
            out.write(data);
            out.flush();
            out.close();
        } catch (IOException eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't write file " + filePath, eee);
            }
        } finally {
            IOUtils.closeQuietly(out);
        }
        return file;
    }

    /**
     * Open csv file and get his data on a string.
     *
     * @param filePath path to targeted file
     * @param charset charset to use for import
     * @return file contents
     */
    protected String loadFile(String filePath, String charset) {
        String result = null;
        InputStream is = null;
        try {
            is = new FileInputStream(filePath);
            result = IOUtils.toString(is, charset);

        } catch (IOException eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't read file " + filePath, eee);
            }
        } finally {
            IOUtils.closeQuietly(is);
        }

        return result;
    }

    protected ImportExportResults exportBackup(String path, String charset){
        ImportExportResults streamData = exportService.exportBackup(charset);

        ZipOutputStream export = null;
        FileOutputStream result = null;
        try {
            SimpleDateFormat dateFormat = new SimpleDateFormat("dd-MM-yyyy HH:mm:ss");
            result = new FileOutputStream(path + "/LIMA-BACKUP-"+ dateFormat.format(new Date()));
            export = new ZipOutputStream(result);
            for (ExportResult exportedData : streamData.getExportResults()) {
                if (exportedData != null && StringUtils.isNotBlank(exportedData.getExportData())) {
                    String data = exportedData.getExportData();
                    File file = createFile(JAVA_IO_TMPDIR + exportedData.getFromSource().getSimpleName(), charset, data);
                    if (file != null) {
                        FileInputStream stream = null;
                        try {
                            ZipEntry ze = new ZipEntry(file.getName());
                            export.putNextEntry(ze);
                            int len;
                            byte[] buffer = new byte[BUFFER_SIZE];
                            stream = new FileInputStream(file);
                            while ((len = stream.read(buffer)) > 0) {
                                export.write(buffer, 0, len);
                            }
                        } finally {
                            IOUtils.closeQuietly(stream);
                            FileUtils.forceDelete(file);
                        }
                    }
                } else {
                    // export failed
                    break;
                }
            }
            export.flush();

        } catch (IOException e) {
            errorHelper.showErrorMessage(t("lima.export.failed"));
            log.error(t("lima.export.failed"), e);
        } finally {
            IOUtils.closeQuietly(export);
            IOUtils.closeQuietly(result);
        }
        return streamData;
    }

    protected ImportExportResults importAllFromZipFile(String filePath) {

        String tmpDir = System.getProperty("java.io.tmpdir")+"/";
        String determinant = String.valueOf(new Date().getTime());
        ImportResult result = extractZipFile(filePath, tmpDir, determinant);
        ImportExportResults results = processBackup(result, tmpDir, determinant);
        return results;
    }

    protected ImportResult extractZipFile(String filePath, String tmpDir, String determinant) {
        ImportExportResults results = new ImportExportResults();
        ImportResult result = results.createAddAndGetImportResult(null);

        ZipInputStream zipInputStream = null;
        FileInputStream inputStream = null;
        // use to be sure to not load old streams.
        try {
            inputStream = new FileInputStream(filePath);

            zipInputStream = new ZipInputStream(inputStream);

            ZipEntry entry;

            while ((entry = zipInputStream.getNextEntry()) != null) {
                FileOutputStream fileoutputstream = null;
                try {
                    byte[] buffer = new byte[BUFFER_SIZE];
                    String targetFileName= tmpDir + entry.getName() + "-" + determinant +".csv";
                    fileoutputstream = new FileOutputStream(targetFileName);

                    int n;
                    while ((n = zipInputStream.read(buffer, 0, BUFFER_SIZE)) > -1) {
                        fileoutputstream.write(buffer, 0, n);
                    }
                    zipInputStream.closeEntry();
                } finally {
                    IOUtils.closeQuietly(fileoutputstream);
                }
            }
        } catch (Exception e) {
            result.addInitException(new ImportFileException(t("lima.import.error.extractFile")));
            log.error(e.getStackTrace());
        } finally {
            IOUtils.closeQuietly(zipInputStream);
            IOUtils.closeQuietly(inputStream);
        }
        return result;
    }

    protected ImportExportResults processBackup(ImportResult result, String tmpDir, String determinant) {
        ImportExportResults results = null;
        InputStream entryBooksStream = null, transactionsStream = null, fiscalPeriodsStream = null, accountsStream = null, entriesStream = null, identityStream = null;
        if (result.getAllExceptionsByLine() == null || result.getAllExceptionsByLine().isEmpty()) {
            try {
                entryBooksStream = new FileInputStream(tmpDir + EntryBook.class.getSimpleName() + "-" + determinant + ".csv");
                String entryBooksStreamString = IOUtils.toString(entryBooksStream);
                IOUtils.closeQuietly(entryBooksStream);

                // import
                transactionsStream = new FileInputStream(tmpDir + FinancialTransaction.class.getSimpleName() + "-" + determinant + ".csv");
                String transactionsStreamString = IOUtils.toString(transactionsStream);
                IOUtils.closeQuietly(transactionsStream);

                accountsStream = new FileInputStream(tmpDir + Account.class.getSimpleName() + "-" + determinant + ".csv");
                String accountsStreamString = IOUtils.toString(accountsStream);
                IOUtils.closeQuietly(accountsStream);

                entriesStream = new FileInputStream(tmpDir + Entry.class.getSimpleName() + "-" + determinant + ".csv");
                String entriesStreamString = IOUtils.toString(entriesStream);
                IOUtils.closeQuietly(entriesStream);

                fiscalPeriodsStream = new FileInputStream(tmpDir + FiscalPeriod.class.getSimpleName() + "-" + determinant + ".csv");
                String fiscalPeriodsStreamString = IOUtils.toString(fiscalPeriodsStream);
                IOUtils.closeQuietly(fiscalPeriodsStream);

                identityStream = new FileInputStream(tmpDir + Identity.class.getSimpleName() + "-" + determinant + ".csv");
                String identityStreamString = IOUtils.toString(identityStream);
                IOUtils.closeQuietly(identityStream);

                results = importService.importBackup(entryBooksStreamString, transactionsStreamString, fiscalPeriodsStreamString, accountsStreamString, entriesStreamString, identityStreamString);
            } catch (Exception ex) {
                if(log.isInfoEnabled()) {
                    log.info(ex);
                }
                result.addInitException(new ImportFileException(t("lima.import.error.extractFile")));
            } finally {
                IOUtils.closeQuietly(entryBooksStream);
                IOUtils.closeQuietly(transactionsStream);
                IOUtils.closeQuietly(fiscalPeriodsStream);
                IOUtils.closeQuietly(accountsStream);
                IOUtils.closeQuietly(entriesStream);
                IOUtils.closeQuietly(identityStream);
            }
        }
        return results;
    }
}
