/*
 * #%L
 * Lima Swing
 * 
 * $Id: FinancialTransactionUnbalancedViewHandler.java 3777 2014-04-14 07:29:08Z sbavencoff $
 * $HeadURL: https://svn.chorem.org/lima/tags/lima-0.7.3/lima-swing/src/main/java/org/chorem/lima/ui/financialtransactionunbalanced/FinancialTransactionUnbalancedViewHandler.java $
 * %%
 * Copyright (C) 2008 - 2012 CodeLutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package org.chorem.lima.ui.financialtransactionunbalanced;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.lima.entity.Entry;
import org.chorem.lima.entity.EntryImpl;
import org.chorem.lima.entity.FinancialTransaction;
import org.chorem.lima.ui.combobox.FiscalPeriodComboBoxModel;
import org.chorem.lima.ui.financialtransaction.FinancialTransactionDefaultTable;

import javax.swing.*;
import javax.swing.table.TableCellEditor;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.math.BigDecimal;

import static org.nuiton.i18n.I18n.t;

/**
 * Handler associated with financial transaction view.
 *
 * @author chatellier
 * @version $Revision: 3777 $
 *          <p/>
 *          Last update : $Date: 2014-04-14 09:29:08 +0200 (Mon, 14 Apr 2014) $
 *          By : $Author: sbavencoff $
 */
public class FinancialTransactionUnbalancedViewHandler {

    /** log. */
    private static final Log log =
            LogFactory.getLog(FinancialTransactionUnbalancedViewHandler.class);

    protected FinancialTransactionUnbalancedView view;

    protected FinancialTransactionUnbalancedViewHandler(FinancialTransactionUnbalancedView view) {
        this.view = view;
        initShortCuts();
    }


    /**
     * Init all combo box in view.
     */
    public void init() {

        if (view.getFiscalPeriodComboBox().getModel().getSize() > 0) {
            view.getFiscalPeriodComboBox().setSelectedIndex(0);
        }
        refresh();
    }

    protected void initShortCuts() {

        InputMap inputMap= view.getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);
        ActionMap actionMap = view.getActionMap();
        Object binding;

        // add action on Ctrl + Shift + Delete
        binding = "delete-transaction";
        inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_DELETE, KeyEvent.CTRL_DOWN_MASK + KeyEvent.SHIFT_DOWN_MASK), binding);
        actionMap.put(binding, new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                deleteSelectedTransaction();
            }
        });

        // add action on Ctrl + N
        binding = "new-entry";
        inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_N, KeyEvent.CTRL_DOWN_MASK), binding);
        actionMap.put(binding, new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                addEntry();
            }
        });

        // add action on Ctrl + Delete
        binding = "delete-entry";
        inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_DELETE, KeyEvent.CTRL_DOWN_MASK), binding);
        actionMap.put(binding, new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                deleteSelectedEntry();
            }
        });

        // add action on Ctrl + B
        binding = "balance";
        inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_B, KeyEvent.CTRL_DOWN_MASK), binding);
        actionMap.put(binding, new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                balanceTransaction();
            }
        });

        // add action on F5
        binding = "refresh";
        inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_F5, 0) , binding);
        actionMap.put(binding, new AbstractAction() {
            @Override
            public void actionPerformed(ActionEvent e) {
                refresh();
            }
        });
    }

    //implement new transaction button
    //add a new entry to the selected transaction
    public void addEntry() {

        FinancialTransactionDefaultTable table = view.getFinancialTransactionUnbalancedTable();
        FinancialTransactionUnbalancedTableModel tableModel = view.getFinancialTransactionUnbalancedTableModel();

        int indexSelectedRow = table.getSelectedRow();
        if (indexSelectedRow != -1) {
            TableCellEditor cellEditor = table.getCellEditor();
            if (cellEditor != null) {
                cellEditor.cancelCellEditing();
            }

            Entry entry = tableModel.get(indexSelectedRow);
            //copy + paste the description
            Entry newEntry = new EntryImpl();
            newEntry.setFinancialTransaction(entry.getFinancialTransaction());
            newEntry.setVoucher(entry.getVoucher());
            newEntry.setDescription(entry.getDescription());

            tableModel.addEntry(newEntry);
            //select the new line
            int row = tableModel.indexOf(newEntry.getFinancialTransaction()) + newEntry.getFinancialTransaction().sizeEntry() - 1;
                    ListSelectionModel selectionModel = table.getSelectionModel();
            selectionModel.setSelectionInterval(
                    row, row);
            table.changeSelection(row, 1, false, false);
            table.editCellAt(row, 1);
        } else {
            if (log.isWarnEnabled()) {
                log.warn("Call addentry selected transaction without selection");
            }
        }
    }

    /**
     * Delete selected row in table (could be transaction or entry).
     * <p/>
     * Called by tableModel.
     */
    public void deleteSelectedEntry() {

        FinancialTransactionDefaultTable table = view.getFinancialTransactionUnbalancedTable();
        FinancialTransactionUnbalancedTableModel tableModel = view.getFinancialTransactionUnbalancedTableModel();

        int indexSelectedRow = table.getSelectedRow();
        if (indexSelectedRow != -1) {

            TableCellEditor cellEditor = table.getCellEditor();
            if (cellEditor != null) {
                cellEditor.cancelCellEditing();
            }

            String message = t("lima.charts.financialtransaction.question.removeentry");

            int response = JOptionPane.showConfirmDialog(view, message,
                                                         t("lima.common.confirmation"), JOptionPane.YES_NO_OPTION);

            if (response == JOptionPane.YES_OPTION) {

                tableModel.removeEntry(indexSelectedRow);

                if (tableModel.size() > 0) {
                    if (indexSelectedRow >= tableModel.size()) {
                        indexSelectedRow =  tableModel.size() - 1;
                    }
                    ListSelectionModel selectionModel =
                            table.getSelectionModel();
                    selectionModel.setSelectionInterval(
                            indexSelectedRow, indexSelectedRow);
                    table.changeSelection(indexSelectedRow, 1, false, false);
                    table.editCellAt(indexSelectedRow, 1);
                }
            }
        } else {
            if (log.isWarnEnabled()) {
                log.warn("Call delete selected row without selection");
            }
        }
    }

    /**
     * Delete selected row in table (could be transaction or entry).
     * <p/>
     * Called by tableModel.
     */
    public void deleteSelectedTransaction() {

        FinancialTransactionDefaultTable table = view.getFinancialTransactionUnbalancedTable();
        FinancialTransactionUnbalancedTableModel tableModel = view.getFinancialTransactionUnbalancedTableModel();

        int indexSelectedRow = table.getSelectedRow();
        if (indexSelectedRow != -1) {

            TableCellEditor cellEditor = table.getCellEditor();
            if (cellEditor != null) {
                cellEditor.cancelCellEditing();
            }

            String message = t("lima.charts.financialtransaction.question.removetransaction");

            int response = JOptionPane.showConfirmDialog(view, message,
                    t("lima.common.confirmation"), JOptionPane.YES_NO_OPTION);

            if (response == JOptionPane.YES_OPTION) {
                tableModel.removeTransaction(indexSelectedRow);
                //select the upper line
                if (tableModel.size() > 0) {
                    if (indexSelectedRow >= tableModel.size()) {
                        indexSelectedRow =  tableModel.size() - 1;
                    }
                    ListSelectionModel selectionModel =
                            table.getSelectionModel();
                    selectionModel.setSelectionInterval(
                            indexSelectedRow, indexSelectedRow);
                    table.changeSelection(indexSelectedRow, 1, false, false);
                    table.editCellAt(indexSelectedRow, 1);
                }
            }
        } else {
            if (log.isWarnEnabled()) {
                log.warn("Call delete selected row without selection");
            }
        }
    }

    public void selectionChanged() {
        FinancialTransactionDefaultTable table = view.getFinancialTransactionUnbalancedTable();
        FinancialTransactionUnbalancedTableModel tableModel = view.getFinancialTransactionUnbalancedTableModel();

        int selectedRow = table.getSelectedRow();
        view.setSelectedRow(selectedRow >= 0 && selectedRow < tableModel.size());
        if (selectedRow >= 0 && selectedRow < tableModel.size())  {
            FinancialTransaction transaction = tableModel.getTransactionAt(selectedRow);
            BigDecimal credit = transaction.getAmountCredit();
            BigDecimal debit = transaction.getAmountDebit();
            view.setBalance(credit.compareTo(debit) ==  0);
        } else {
            view.setBalance(true);
        }
    }

    public void balanceTransaction()  {
        FinancialTransactionDefaultTable table = view.getFinancialTransactionUnbalancedTable();
        FinancialTransactionUnbalancedTableModel tableModel = view.getFinancialTransactionUnbalancedTableModel();

        int rowSelected = table.getSelectedRow();
        if (rowSelected != -1) {

            if (table.isEditing()) {
                TableCellEditor editor = table.getCellEditor();
                editor.stopCellEditing();
            }

            Entry entry = tableModel.get(rowSelected);
            FinancialTransaction transaction = entry.getFinancialTransaction();
            BigDecimal credit = transaction.getAmountCredit();
            BigDecimal debit = transaction.getAmountDebit();
            if (entry.isDebit()) {
                debit = debit.subtract(entry.getAmount());
            } else {
                credit = credit.subtract(entry.getAmount());
            }

            BigDecimal balance = credit.subtract(debit);
            entry.setAmount(balance.abs());
            entry.setDebit(balance.signum() > 0);
            tableModel.updateEntry(entry);
            int firstRow = tableModel.indexOf(transaction);
            int lastRow = firstRow + transaction.sizeEntry() - 1;
            tableModel.fireTableRowsUpdated(firstRow, lastRow);
            view.setBalance(true);

            ListSelectionModel selectionModel = table.getSelectionModel();
            selectionModel.setSelectionInterval(rowSelected, rowSelected);
            table.changeSelection(rowSelected, 1, false, false);
            table.editCellAt(rowSelected, 1);

        }
    }

    public void refresh() {
        FinancialTransactionUnbalancedTableModel tableModel = view.getFinancialTransactionUnbalancedTableModel();
        FinancialTransactionDefaultTable table = view.getFinancialTransactionUnbalancedTable();
        table.exit();

        tableModel.refresh();
        table.clearSelection();

        FiscalPeriodComboBoxModel comboBoxModel = view.getModelFiscalPeriod();
        comboBoxModel.refresh();
    }
}
