/*
 * #%L
 * Lima Swing
 * 
 * $Id: FinancialStatementReportTableModel.java 3051 2010-11-29 14:57:16Z echatellier $
 * $HeadURL: http://svn.chorem.org/svn/lima/tags/lima-0.4.3/lima-swing/src/main/java/org/chorem/lima/ui/financialstatementreport/FinancialStatementReportTableModel.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package org.chorem.lima.ui.financialstatementreport;

import static org.nuiton.i18n.I18n._;

import java.awt.Desktop;
import java.io.IOException;
import java.math.BigDecimal;
import java.net.URI;
import java.net.URISyntaxException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import javax.swing.table.AbstractTableModel;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.lima.LimaConfig;
import org.chorem.lima.beans.FinancialStatementAmounts;
import org.chorem.lima.business.DocumentServiceMonitorable;
import org.chorem.lima.business.FinancialStatementServiceMonitorable;
import org.chorem.lima.business.FinancialTransactionServiceMonitorable;
import org.chorem.lima.business.HttpServerServiceMonitorable;
import org.chorem.lima.business.ImportServiceMonitorable;
import org.chorem.lima.business.LimaException;
import org.chorem.lima.business.ServiceListener;
import org.chorem.lima.business.utils.DocumentsEnum;
import org.chorem.lima.business.utils.FormatsEnum;
import org.chorem.lima.service.LimaServiceFactory;
import org.chorem.lima.util.ErrorHelper;

/**
 * Entry book table model.
 * 
 * @author ore
 * @author chatellier
 * @version $Revision: 3051 $
 * 
 * Last update : $Date: 2010-11-29 15:57:16 +0100 (lun., 29 nov. 2010) $
 * By : $Author: echatellier $
 */
public class FinancialStatementReportTableModel extends AbstractTableModel implements ServiceListener {

    /** serialVersionUID. */
    private static final long serialVersionUID = 1L;

    /** log. */
    private static final Log log = LogFactory.getLog(FinancialStatementReportTableModel.class);

    /** Services. */
    protected FinancialStatementServiceMonitorable financialStatementService;
    protected DocumentServiceMonitorable documentService;
    
    /** DatePicker Begin Date. */
    protected Date selectedBeginDate;
    
    /** DatePicker EndDate. */
    protected Date selectedEndDate;
    
    protected int port;
    
    private static SimpleDateFormat dateFormat =
        new SimpleDateFormat("yyyy-MM-dd");
    
    /** data cache */
    protected List<FinancialStatementAmounts> cacheDataList;
    
    /**
     * Constructor.
     */
    public FinancialStatementReportTableModel() {
        financialStatementService =
            LimaServiceFactory.getInstance().getService(
                    FinancialStatementServiceMonitorable.class);
        
        financialStatementService.addListener(this);
        port = LimaServiceFactory.getInstance().getService(
                HttpServerServiceMonitorable.class).getHttpPort();
        LimaServiceFactory.getInstance().getService(
                ImportServiceMonitorable.class).addListener(this);
        LimaServiceFactory.getInstance().getService(
                FinancialTransactionServiceMonitorable.class).addListener(this);
        documentService = LimaServiceFactory.getInstance().getService(DocumentServiceMonitorable.class);
    }

    @Override
    public int getRowCount() {
        int result = 0;
        if (cacheDataList != null){
            result = cacheDataList.size();    
        }
        return result;
    }

    @Override
    public int getColumnCount() {
        return 4;
    }

    @Override
    public String getColumnName(int column) {
        String res = "n/a";
        switch (column) {
            case 0:
                res = _("lima.table.label");
                break;
            case 1:
                res = _("lima.table.grossamount");
                break;
            case 2:
                res = _("lima.table.provisiondeprecationamount");
                break;
            case 3:
                res = _("lima.table.netamount");
                break;
        }
        return res;
    }

    public Object getElementAt(int row){
        
        Object currentRow = cacheDataList.get(row);
        return currentRow;
    }
    
    @Override
    public Object getValueAt(int row, int column) {
        Object result = cacheDataList.get(row);
        //get entries for the period for the current row 
        if (result instanceof FinancialStatementAmounts) {
            FinancialStatementAmounts currentRow = (FinancialStatementAmounts) result;
            BigDecimal grossAmount = currentRow.getGrossAmount();
            if (grossAmount == null ){
                grossAmount = new BigDecimal(0);
            }
            
            BigDecimal provisionDeprecationAmount = currentRow.getProvisionDeprecationAmount();
            if (provisionDeprecationAmount == null ){
                provisionDeprecationAmount = new BigDecimal(0);
            }
            switch (column) {
            case 0:
                result = "";
                String label = currentRow.getLabel();
                if ( label != null){
                    for (int i = 0; i <= currentRow.getLevel(); i++) {
                        result = result+"\t";
                    }
                        result = result+label;
                }
                break;
            case 1:
                if (grossAmount.doubleValue() == 0){
                    result = null;
                }
                else {
                    result = grossAmount;
                }
                break;
            case 2:
                if (provisionDeprecationAmount.doubleValue() == 0){
                    result = null;
                }
                else {
                    result = provisionDeprecationAmount;
                }
                break;
            case 3:
                BigDecimal solde = grossAmount.subtract(provisionDeprecationAmount);
                if (solde.doubleValue() == 0){
                    result = null;
                }
                else {
                    result = solde;
                }
                break;
            }
        }

        return result;
    }


    @Override
    public boolean isCellEditable(int rowIndex, int columnIndex) {
        // Just read, no write
        return false;
    }

    public void setBeginDate(Date date){
        selectedBeginDate = date;
    }
    
    public void setEndDate(Date date){
        selectedEndDate = date;
    }
    
    /**
     * get all account fot the selected period
     * @return
     */
    public List<FinancialStatementAmounts> getDataList(){
        List<FinancialStatementAmounts> results = null;
        if (selectedBeginDate != null && selectedEndDate != null){
            try {
                results = financialStatementService.financialStatementReport(selectedBeginDate, selectedEndDate);
            }
            catch (LimaException eee) {
                if (log.isErrorEnabled()) {
                    log.debug("Can't update model", eee);
                }
                ErrorHelper.showErrorDialog("Can't get entries list", eee);
            }
        }
        return results;
    }
    
    public void refresh(){
        cacheDataList = getDataList();
        fireTableDataChanged();
    }
    

    public void createDocument(FormatsEnum formatsEnum) {
        
        if (selectedBeginDate != null && selectedEndDate != null){
            String address = LimaConfig.getInstance().getHostAdress();
            try {
                String url = "http://"+address+":"+port+"/?beginDate="
                +dateFormat.format(selectedBeginDate)
                +"&endDate="+dateFormat.format(selectedEndDate)
                +"&format="+formatsEnum.getExtension()+"&model="
                + DocumentsEnum.FINANCIALSTATEMENT.getFileName();
                Desktop.getDesktop().browse(new URI(url));            
            } catch (IOException e) {
                log.error("Can't open browser", e);
            } catch (URISyntaxException e) {
                log.error("Can't create news URI", e);
            }
        }
    }
    
    @Override
    public void notifyMethod(String serviceName, String methodeName) {
        if (serviceName.contains("FinancialTransaction") || methodeName.contains("FinancialStatement") || methodeName.contains("importAll")){
            refresh();
        }
    }
    
}
