/* *##% Lima Swing
 * Copyright (C) 2008 - 2010 CodeLutin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * ##%*/

package org.chorem.lima.ui.lettering;

import static org.nuiton.i18n.I18n._;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import javax.swing.table.AbstractTableModel;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.lima.business.FinancialTransactionServiceMonitorable;
import org.chorem.lima.business.ImportServiceMonitorable;
import org.chorem.lima.business.LimaException;
import org.chorem.lima.business.ServiceListener;
import org.chorem.lima.business.utils.EntryComparator;
import org.chorem.lima.entity.Account;
import org.chorem.lima.entity.Entry;
import org.chorem.lima.entity.EntryBook;
import org.chorem.lima.entity.FinancialPeriod;
import org.chorem.lima.entity.FinancialTransaction;
import org.chorem.lima.entity.Letter;
import org.chorem.lima.service.LimaServiceFactory;
import org.chorem.lima.util.DialogHelper;
import org.chorem.lima.util.ErrorHelper;


/**
 * Basic transaction table model.
 * 
 * Le modele est filtré sur {@link #selectedEntryBook} et
 * {@link #selectedFinancialPeriod} (montée en charge !).
 * 
 * @author ore
 * @author chatellier
 * @version $Revision: 2897 $
 * 
 * Last update : $Date: 2010-05-14 13:22:26 +0200 (ven. 14 mai 2010) $
 * By : $Author: jpepin $
 */
public class LetteringTableModel extends AbstractTableModel implements ServiceListener{

    /** serialVersionUID. */
    private static final long serialVersionUID = 1L;

    /** log. */
    private static final Log log = LogFactory
            .getLog(LetteringTableModel.class);

    /** Transaction service. */
    protected final FinancialTransactionServiceMonitorable financialTransactionService;
    
    /** selected financial period */
    protected FinancialPeriod selectedFinancialPeriod;
    
    /** data cache */
    protected List<Object> cacheDataList;
        
    /** collection

    
    /**
     * Model constructor.
     * 
     * Just init service proxies.
     */
    public LetteringTableModel() {
        /* Services */
    	financialTransactionService =
        	LimaServiceFactory.getInstance().getService(
        			FinancialTransactionServiceMonitorable.class);
    	financialTransactionService.addListener(this);
    	LimaServiceFactory.getInstance().getService(
    			ImportServiceMonitorable.class).addListener(this);
    }

    /**
     * Le model est une combinaison de Transaction/Entries.
     * 
     * 
     * @return
     */
    protected List<Object> getDataList() {
        List<Object> results = new ArrayList<Object>();
        if (selectedFinancialPeriod != null){
            try {
                List<FinancialTransaction> financialtransactions =
                    financialTransactionService.getAllFinancialTransactionsForFinancialPeriod(
                            selectedFinancialPeriod);
                for (FinancialTransaction financialtransaction : financialtransactions) {
                    results.add(financialtransaction);
                    List<Entry> entries = (List<Entry>) financialtransaction.getEntry();
                    Collections.sort(entries, new EntryComparator());
                    results.addAll(entries);
                }
            }
            catch (LimaException eee) {
                if (log.isErrorEnabled()) {
                    log.error("Can't update model", eee);
                }
               ErrorHelper.showErrorDialog("Can't get transaction list", eee);
            }	
        }      
        return results;
    }
    
    public void setLetter(int row, Letter letter){
		try {
	    	Object currentRow = cacheDataList.get(row);
	    	if (currentRow instanceof Entry){
	    		Entry entry = (Entry) currentRow;
					financialTransactionService.setEntryLetter(entry, letter);
	    	}
		} catch (LimaException eee) {
			log.error("Can't set letter", eee);
		}
    }
    
    public void removeLetter(int row){
		try {
	    	Object currentRow = cacheDataList.get(row);
	    	if (currentRow instanceof Entry){
	    		Entry entry = (Entry) currentRow;
					financialTransactionService.removeEntryLetter(entry);
	    	}
		} catch (LimaException eee) {
			log.error("Can't remove letter", eee);
		}
    }
    
    public void refresh(){
        cacheDataList = getDataList();
        fireTableDataChanged();
    }

    @Override
    public int getColumnCount() {
        return 9;
    }

    @Override
    public Class<?> getColumnClass(int column) {

        Class<?> result = null;

        switch (column) {
        case 0:
            result = Date.class;
            break;
        case 1:
            result = EntryBook.class;
            break;
        case 2:
            result = String.class;
            break;
        case 3:
            result = Account.class;
            break;
        case 4:
            result = String.class;
            break;
        case 5:
            result = Letter.class;
            break;
        case 6:
            result = BigDecimal.class;
            break;
        case 7:
            result = BigDecimal.class;
            break;
        case 8:
            result = BigDecimal.class;
            break;
        }

        return result;
    }

    
    
    @Override
    public String getColumnName(int column) {
        String result = "n/a";

        switch (column) {
        case 0:
            result = _("lima.table.date");
            break;
        case 1:
            result = _("lima.table.entrybook");
            break;
        case 2:
            result = _("lima.table.voucher");
            break;
        case 3:
            result = _("lima.table.account");
            break;
        case 4:
            result = _("lima.table.description");
            break;
        case 5:
        	result = _("lima.table.letter");
            break;
        case 6:
            result = _("lima.table.debit");
            break;
        case 7:
            result = _("lima.table.credit");
            break;
        case 8:
            result = _("lima.table.balance");
            break;
        }

        return result;
    }

    @Override
    public int getRowCount() {
        int result = 0;
        
        if (cacheDataList != null) {
            result = cacheDataList.size();
        }

        return result;
    }

    @Override
    public Object getValueAt(int row, int column) {
        Object result = null;
        
        // just prevent too much result
        if (cacheDataList != null) {
            result = cacheDataList.get(row);
            
            if (result instanceof FinancialTransaction) {
            	FinancialTransaction currentRow = (FinancialTransaction)result; 
            	BigDecimal amountDebit = currentRow.getAmountDebit();
            	BigDecimal amountCredit = currentRow.getAmountCredit();
            	
                switch (column) {
                case 0:
                    result = currentRow.getTransactionDate();
                    break;
                case 1:
                    if (currentRow.getEntryBook() != null){
                        result = currentRow.getEntryBook().getCode();
                    }
                    else {
                        result = null;
                    }
                    break;
                case 2:
                    result = null; //voucher
                    break;
                case 3:
                    result = null; // account
                    break;
                case 4:
                    result = null; // description
                    break;
                case 5 :
                    result = null; // letter
                    break;
                case 6:
                    result = amountDebit;
                    break;
                case 7:
                    result = amountCredit;
                    break;
                case 8:
                    result = amountDebit.subtract(amountCredit);
                    break;
                }
            }
            else if (result instanceof Entry) {
                Entry currentEntry = (Entry)result;
                switch (column) {
                case 0:
                    result = null; // date
                    break;
                case 1 : // entry book   
                	result = null;
                    break;
                case 2:
                    result = currentEntry.getVoucher();
                    break;
                case 3: // account
                    if (currentEntry.getAccount() != null){
                        result = currentEntry.getAccount().getAccountNumber();
                    }
                    else {
                        result = null;
                    }
                    break;
                case 4:
                    result = currentEntry.getDescription();
                    break;
                case 5 :
                	if (currentEntry.getLetter() != null){
                		result = currentEntry.getLetter().getCode();
                	}
                	else {
                		result = null;
                	}
                    break;
                case 6:
                    result = currentEntry.getDebit() ? currentEntry.getAmount() : 0;
                    break;
                case 7:
                    result = currentEntry.getDebit() ? 0 : currentEntry.getAmount();
                    break;
                case 8:
                	result = null;
                    break;
                }
                
            }
        }
        return result;
    }
    
    
    public void setFinancialPeriod(FinancialPeriod financialPeriod){
        selectedFinancialPeriod = financialPeriod;
        refresh();
    }
    
    /**
     * To set cells editable or not
     * different condition for entry or financial transaction
     */
    @Override
    public boolean isCellEditable(int rowIndex, int columnIndex) {
        boolean editableCell=false;
        Object currentRow = cacheDataList.get(rowIndex);
        // cells editable for the entry row, all cells exclude the date 
        if ((currentRow instanceof Entry) && ((columnIndex==5))) {
            editableCell=true;
        }
        return editableCell;    }
    
    
    
    public Object getElementAt(int row){
    	Object result = null;
    	if (cacheDataList != null){
            result = cacheDataList.get(row);
    	}
        return result;
    }
   
    
    /**
     * to modifiy letter entry
     */
    @Override
    public void setValueAt(Object value, int row, int column) {
        int financialTransactionRow = 0;
        if (cacheDataList != null) {
            Object currentRow = cacheDataList.get(row);
            if (currentRow instanceof Entry) {
                Entry currentEntry = (Entry)currentRow;
                switch (column) {
                case 5 :
                    currentEntry.setLetter((Letter)value);
                    break;
                }
                try {
                    financialTransactionService.updateEntry(currentEntry);

                } catch (LimaException eee) {
                    if (log.isDebugEnabled()){
                        log.error("Can't update entry", eee);
                    }
                    DialogHelper.showMessageDialog(eee.getMessage());
                }
                //update the financial transaction in entire
                financialTransactionRow = 
                    getDataList().indexOf(((Entry) currentRow).
                            getFinancialTransaction());
            }
            //on recharge la liste
            cacheDataList = getDataList();
            fireTableRowsUpdated(financialTransactionRow, getRowCount()-1);
        }
    }

	@Override
	public void notifyMethod(String serviceName, String methodeName) {
		if (serviceName.contains("FinancialTransaction") || methodeName.contains("importEntries") || methodeName.contains("importAll")){
			refresh();
		}
	}
     
}
