/* *##% Lima Swing
 * Copyright (C) 2008 - 2010 CodeLutin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * ##%*/


package org.chorem.lima.ui.importexport;

import static org.nuiton.i18n.I18n._;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.util.concurrent.ExecutionException;
import javax.swing.JComboBox;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingWorker;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.lima.business.ExportServiceMonitorable;
import org.chorem.lima.business.ImportServiceMonitorable;
import org.chorem.lima.business.LimaException;
import org.chorem.lima.business.utils.ImportExportEntityEnum;
import org.chorem.lima.enums.EncodingEnum;
import org.chorem.lima.enums.ImportExportEnum;
import org.chorem.lima.service.LimaServiceFactory;
import org.chorem.lima.ui.account.AccountViewHandler;
import org.chorem.lima.util.DialogHelper;
import org.jdesktop.swingx.painter.BusyPainter;

public class ImportExport {
    
	private static ImportExport exchanger;
	
    private static final Log log = LogFactory.getLog(AccountViewHandler.class);
    
	protected static Component viewComponent;
	
	protected EncodingEnum encodingEnum;

    protected ImportServiceMonitorable importService;
    
    protected ExportServiceMonitorable exportService;
    
    private ImportExportWaitView waitView;

    public ImportExport(Component view) {
    	viewComponent=view;
    	
    	//services
    	importService =
    		LimaServiceFactory.getInstance().getService(
    				ImportServiceMonitorable.class);
        exportService =
        	LimaServiceFactory.getInstance().getService(
        			ExportServiceMonitorable.class);
        
        //create the wait dialog panel
        waitView = new ImportExportWaitView();
        waitView.waitDialog.setSize(400, 200);
        waitView.setLocationRelativeTo(view);
        BusyPainter busyPainter = waitView.getBusylabel().getBusyPainter();
        busyPainter.setHighlightColor(new Color(44, 61, 146).darker());
        busyPainter.setBaseColor(new Color(168, 204, 241).brighter());
	}

    /**
     * Call the appropriate methode in business service
     * @param importExportMethode
     */
    public void importExport(ImportExportEnum importExportMethode, String file, Boolean verbose){	
    	final ImportExportEnum importExportMethodeF = importExportMethode;
    	encodingEnum = EncodingEnum.UTF8;
    	if (file.equals("")){
    		file = chooseFile(importExportMethode.getImportMode(), importExportMethode);
    	}
    	//if export cancel
    	if (!file.equals("")){
	    	final EncodingEnum charset = encodingEnum;
	    	final String filePath = file;
	    	final Boolean verboseMode = verbose;
			new SwingWorker<String,Void>() {
	            @Override protected String doInBackground() throws LimaException {
	        		String datas = "";
	            	String result = "";
					switch (importExportMethodeF) {
						case CSV_ALL_EXPORT:
							datas = exportService.exportAsCSV();
					    	createFile(filePath, charset.getEncoding(), datas);
							break;
						case CSV_ACCOUNTCHARTS_EXPORT:
							datas = exportService.exportAccountsChartAsCSV();
							createFile(filePath, charset.getEncoding(), datas);
							break;
						case CSV_ENTRYBOOKS_EXPORT:
							datas = exportService.exportEntryBookChartAsCSV();
							createFile(filePath, charset.getEncoding(), datas);
							break;
						case CSV_FINANCIALSTATEMENTS_EXPORT:
							datas = exportService.exportFinancialStatementChartAsCSV();
							createFile(filePath, charset.getEncoding(), datas);
							break;
						case EBP_ACCOUNTCHARTS_EXPORT:
							//For windows ebp
							datas = exportService.exportAccountsAsEBP();
							createFile(filePath, EncodingEnum.ISOLATIN1.getEncoding(), datas);
							break;
						case EBP_ENTRIES_EXPORT:
							//For windows ebp
							datas = exportService.exportEntriesAsEBP();
							createFile(filePath, EncodingEnum.ISOLATIN1.getEncoding(), datas);
							break;
						case CSV_ALL_IMPORT:
							datas = extractFile(filePath, charset.getEncoding());
					    	result = importService.importAllAsCSV(datas);
							break;
						case CSV_ACCOUNTCHARTS_IMPORT:
							datas = extractFile(filePath, charset.getEncoding());
					    	result = importService.importAsCSV(datas, ImportExportEntityEnum.ACCOUNT);
							break;
						case CSV_ENTRYBOOKS_IMPORT:
							datas = extractFile(filePath, charset.getEncoding());
					    	result = importService.importAsCSV(datas, ImportExportEntityEnum.ENTRYBOOK);
							break;
						case CSV_FINANCIALSTATEMENTS_IMPORT:
							datas = extractFile(filePath, charset.getEncoding());
					    	result = importService.importAsCSV(datas, ImportExportEntityEnum.FINANCIALSTATEMENT);
							break;
						case EBP_ACCOUNTCHARTS_IMPORT:
							//For windows ebp
							datas = extractFile(filePath, EncodingEnum.ISOLATIN1.getEncoding());
							result = importService.importAccountsChartFromEbp(datas);
							break;
			            case EBP_ENTRIES_IMPORT:
							//For windows ebp
			            	datas = extractFile(filePath, EncodingEnum.ISOLATIN1.getEncoding());
			            	result = importService.importEntriesFromEbp(datas);
							break;
						}
					return result;
	            }
	            @Override protected void done() {
	            	try {
	            		//hidde wait dialog panel
		            	waitView.setVisible(false);
		            	// display result dialog
		            	if (verboseMode){
			            	String result = get();
			            	if (!result.equals("")){
			        			DialogHelper.showReportDialog(result, _("lima.importexport.import"), viewComponent);
			        		}
			        		else {
			        			DialogHelper.showMessageDialog(_("lima.importexport.import.terminated"));
			        		}	
		            	}
					} catch (InterruptedException eee) {
				    	if (log.isDebugEnabled()){
							log.error("Can't get result message", eee);
				    	}
					} catch (ExecutionException eee) {
				    	DialogHelper.showMessageDialog(eee.getCause().getMessage());
				    	log.error("Error on import", eee);
					}
	            }
	        }.execute();
	        waitView.setVisible(true);
    	}
    }
    
    
    /**
     * open choose file dialog with appropriate file mode view
     * folders for export or folders+files for import
     * @param importMode
     * @return
     */
    public String chooseFile(Boolean importMode, ImportExportEnum importExportMethode){
    	String filePath = "";
   		JFileChooser chooser = new JFileChooser();
   		//Encoding option
   		
	   	JComboBox comboBox = new JComboBox(EncodingEnum.descriptions());
	   	if (importExportMethode.getEncodingOption()){
	   		JPanel panel = new JPanel();
	   		panel.setLayout(new BorderLayout());
	   		panel.add(new JLabel(_("lima.importexport.choiceencoding")), BorderLayout.WEST);
	   		panel.add(comboBox, BorderLayout.CENTER);
	        ((Container)chooser.getComponent(2)).add(panel, BorderLayout.SOUTH);
   		}
   		
		if (importMode){
			chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
		}
		else {
			chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
		}
        if (chooser.showOpenDialog(viewComponent) == JFileChooser.APPROVE_OPTION) {
            filePath = chooser.getSelectedFile().getAbsolutePath();

            if (importExportMethode.getEncodingOption()){
                encodingEnum = EncodingEnum.valueOfDescription((String) comboBox.getSelectedItem());
            }
        }

		return filePath;
    }
    
    
    /**
     * Get csv datas in string and write file
     * 
     */
	public void createFile(String filePath, String charset, String datas){
		try {
			File file = new File(filePath);
			BufferedWriter out = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(file), charset));
			out.write(datas);
			out.flush();
			out.close();
		}
	    catch (IOException eee) {
	    	if (log.isDebugEnabled()){
	    		log.error("Can't write file " + filePath, eee);
	    	}
	    }
	}
	
	
	/**
	 * Open csv file and get his datas on a string
	 * 
	 */
	public String extractFile(String filePath, String charset){
	    char[] datas = new char[(int) new File(filePath).length()];
	    try {
	    	BufferedReader in = new BufferedReader(new InputStreamReader(new FileInputStream(filePath), charset));
			in.read(datas);
		    in.close();
	    }
	    catch (IOException eee) {
	    	if (log.isDebugEnabled()){
	    		log.error("Can't read file "+filePath, eee);
	    	}
	    }
	    return new String(datas);
	}
	
    public static ImportExport getInstance(Component view) {
        if (exchanger == null) {
        	exchanger = new ImportExport(view);
        }
        else {
        	viewComponent=view;
        }
        return exchanger;
    }
	
}
