/*
 * #%L
 * Wao :: Web Interface
 * 
 * $Id: Boats.java 1593 2012-04-02 13:04:49Z bleny $
 * $HeadURL: http://svn.forge.codelutin.com/svn/wao/tags/wao-3.3.5/wao-ui/src/main/java/fr/ifremer/wao/ui/pages/Boats.java $
 * %%
 * Copyright (C) 2009 - 2010 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

package fr.ifremer.wao.ui.pages;

import fr.ifremer.wao.WaoBusinessException;
import fr.ifremer.wao.WaoException;
import fr.ifremer.wao.bean.BoatFilter;
import fr.ifremer.wao.bean.CompanyBoatInfos;
import fr.ifremer.wao.bean.ConnectedUser;
import fr.ifremer.wao.bean.ObsProgram;
import fr.ifremer.wao.bean.UserRole;
import fr.ifremer.wao.entity.Boat;
import fr.ifremer.wao.entity.BoatInfos;
import fr.ifremer.wao.entity.Company;
import fr.ifremer.wao.entity.Contact;
import fr.ifremer.wao.entity.ElligibleBoat;
import fr.ifremer.wao.entity.SampleRow;
import fr.ifremer.wao.entity.ShipOwner;
import fr.ifremer.wao.service.ServiceBoat;
import fr.ifremer.wao.service.ServiceContact;
import fr.ifremer.wao.service.ServiceReferential;
import fr.ifremer.wao.service.ServiceSampling;
import fr.ifremer.wao.service.ServiceUser;
import fr.ifremer.wao.ui.components.BoatFilterComponent;
import fr.ifremer.wao.ui.components.Layout;
import fr.ifremer.wao.ui.data.BoatDataSource;
import fr.ifremer.wao.ui.data.ExportStreamResponse;
import fr.ifremer.wao.ui.data.RequiresAuthentication;
import fr.ifremer.wao.ui.data.WaoActivationContext;
import fr.ifremer.wao.ui.services.WaoManager;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.tapestry5.Block;
import org.apache.tapestry5.EventContext;
import org.apache.tapestry5.StreamResponse;
import org.apache.tapestry5.annotations.Import;
import org.apache.tapestry5.annotations.InjectComponent;
import org.apache.tapestry5.annotations.InjectPage;
import org.apache.tapestry5.annotations.Log;
import org.apache.tapestry5.annotations.Persist;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.annotations.SessionState;
import org.apache.tapestry5.corelib.components.Zone;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.ioc.services.PropertyAccess;
import org.slf4j.Logger;

import java.io.IOException;
import java.io.InputStream;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;

/**
 * Boats
 *
 * Created: 9 nov. 2009
 *
 * @author fdesbois <fdesbois@codelutin.com>
 */
@RequiresAuthentication(allowedRoles = {UserRole.ADMIN, UserRole.COORDINATOR, UserRole.OBSERVER, UserRole.PROFESSIONAL})
@Import(stylesheet = "context:css/boats.css")
public class Boats {

    @Inject
    private Logger log;

    @InjectComponent
    private Layout layout;

    @SessionState
    @Property
    private ConnectedUser user;

    @Inject
    private ServiceReferential serviceReferential;

    @Inject
    private ServiceSampling serviceSampling;

    @Inject
    private ServiceBoat serviceBoat;

    @Inject
    private ServiceContact serviceContact;

    @Inject
    private PropertyAccess propertyAccess;

    private WaoActivationContext waoActivationContext;

    void onActivate(EventContext ec) {
        waoActivationContext = WaoActivationContext.newContextFromTapestry(ec);
    }

//    String[] onPassivate() {
//        WaoActivationContext waoActivationContext = WaoActivationContext.newEmptyContext();
//        if (sampleRowContext != null) {
//            waoActivationContext.setSampleRowCode(sampleRowContext.getCode());
//        }
//        return waoActivationContext.toStrings();
//    }

    @Property
    @Persist
    private SampleRow sampleRowContext;

    public Object onActionFromCancelSelection() {
        if (log.isInfoEnabled()) {
            log.info("user " + user + " has cancelled boat selection for row " + sampleRowContext);
        }
        sampleRowContext = null;
        if (user.isObsDeb()) {
            return ObsDebSamplingPlan.class;
        } else {
            return SamplingPlan.class;
        }
    }

    public String getSummaryForSampleRowContext() {
        String summary = null;
        if (sampleRowContext != null) {
            if (sampleRowContext.getObsProgram() == ObsProgram.OBSDEB) {
                summary = manager.getTooltipSampleRow(sampleRowContext);
            }
        }
        return summary;
    }

    void setupRender() throws WaoException, WaoBusinessException {

        // XXX bleny 20110511 resetting on page load may solve an issue when
        // creating contact and coming back to Boats. But this may break the page
        // filterComponent.resetFilter();

        boats = null;
        boatSelectedImmatriculation = null;
        companyBoatInfos = null;

        if (log.isTraceEnabled()) {
            log.trace("waoActivationContext " + waoActivationContext);
        }
        if (waoActivationContext.getBoatImmatriculation() != null) {
            getBoatFilter().setBoatImmatriculation(
                    waoActivationContext.getBoatImmatriculation());
        }
        if (waoActivationContext.getShipOwnerName() != null) {
            getBoatFilter().setShipOwnerName(
                    waoActivationContext.getShipOwnerName());
        }

        String sampleRowContextId = waoActivationContext.getSampleRowCode();
        if (sampleRowContextId != null) {
            SampleRow sampleRow = serviceSampling.getSampleRowByCode(user, sampleRowContextId);
            filterComponent.setSampleRow(sampleRow);
            if (user.isCoordinatorOrObserver()) {
                sampleRowContext = sampleRow;
            } else {
                sampleRowContext = null;
            }
        }
    }

    StreamResponse onActionFromExportShowBoats() {
        if (log.isInfoEnabled()) {
            log.info("exporting boats with filter=" + getBoatFilter());
        }
        return new ExportStreamResponse("wao-navires") {

            @Override
            public InputStream getStream() throws IOException {
                InputStream result = null;
                try {
                    result = serviceBoat.exportBoatCsv(getBoatFilter());
                } catch (WaoException eee) {
                    throw new IOException(eee);
                }
                return result;
            }
        };
    }

    /******************** FILTERS *******************/

    @InjectComponent
    private BoatFilterComponent filterComponent;

    /** Selected boat immatriculation for showing boatInfos */
    @Persist
    private Integer boatSelectedImmatriculation;

    /**
     * Filters to apply on boats list.
     *
     * @return the current BoatFilter to apply
     * @throws WaoException
     */
    public BoatFilter getFilter() throws WaoException {
        return getBoatFilter();
    }

    public BoatFilter getBoatFilter() throws WaoException {
        return filterComponent.getFilter();
    }

    public boolean isFiltersVisible() {
        return filterComponent.getFiltersVisible();
    }

    Object onActionFromShowFilters() {
        filterComponent.switchFiltersVisible();
        return filterComponent;
    }

    void onSelectedFromReset() {
        filterComponent.resetFilter();
    }

    @Log
    Object onSuccessFromFiltersForm() throws WaoException {
        boats = null;
        boatSelectedImmatriculation = null;
        companyBoatInfos = null;
        return boatsZone.getBody();
    }

    /*************** RIGHT PANEL ****************/

    public Integer getBoatSelectedImmatriculation() {
        if (boatSelectedImmatriculation == null) {
             boatSelectedImmatriculation =
                     waoActivationContext.getBoatImmatriculation();
        }
        return boatSelectedImmatriculation;
    }

    public void setBoatSelectedImmatriculation(Integer boatSelectedImmatriculation) {
        this.boatSelectedImmatriculation = boatSelectedImmatriculation;
    }

    /*************** Boats List (left panel) ****************/

    /** DataSource containing the boats */
    @Persist
    private BoatDataSource boats;

    /** Current boat in the Grid */
    @Property
    private Boat boat;

    /** Used to change style a line on two */
    private boolean even = true;

    /** Used to refresh boats Grid */
    @InjectComponent
    private Zone boatsZone;

    /**
     * Get dataSource which contains current boats to show on the page. Depends on filters sets by user.
     * BoatDataSource uses serviceBoat to retrieve boats with filters.
     *
     * @return the BoatDataSource which contains the boats map
     * @throws WaoException
     * @see BoatDataSource
     */
    public BoatDataSource getBoats() throws WaoException {
        if (boats == null) {
            boats = new BoatDataSource(getBoatFilter(), serviceBoat);
        }
        return boats;
    }

    /**
     * Initialized row CSS style depends on selection in the list. The style is also changed a line on two.
     *
     * @return the CSS style class to used for the current row in the Grid
     */
    public String getRowStyle() {
        String result = "";
        even = !even;        
        if (boatSelectedImmatriculation != null &&
                boat.getImmatriculation() == boatSelectedImmatriculation) {
            result = "selected";
        } else if (!boat.getActive()) {
            result = "refused";
        } else {
            result = even ? "even" : "odd";
        }
        return result;
    }

    public String getBoatInactive() {
        return !boat.getActive() ? "(inactif)" : "";
    }

    public DateFormat getDateFormat() {
        return new SimpleDateFormat("dd/MM/yyyy");
    }
    
    public boolean canCreateNewContactFromList() throws WaoException {
        if (!user.isAdmin() && !user.isProfessional() && !user.isReadOnly() && sampleRowContext != null) {
            return boat.canCreateContact(user.getProfile().getObsProgram(), user.getCompany());
        }
        return false;
    }

    Block onActionFromShowBoatInfos(Integer boatImma) throws WaoException {
        if (log.isInfoEnabled()) {
            log.info("user want to show boat info for boatImma=" + boatImma);
        }

        boatSelectedImmatriculation = boatImma;
        // Suppress persistant boat informations
        companyBoatInfos = null;

        if (user.isAdmin() && getCompany() != null) {
//            companySelectedId = getCompany().getTopiaId();
        }

        return boatsZone.getBody();
    }

    /**************************** Boat selected Infos *************************/

    @Persist
    private CompanyBoatInfos companyBoatInfos;

    @InjectComponent
    private Zone boatInfosZone;

    @Inject
    private Block displayBoatInfos;

    @Inject
    private Block editBoatInfos;

    @Property
    private ElligibleBoat elligibleBoat;

    @Property
    private SampleRow boatInfosSampleRow;

    private boolean boatInfosEditable;

    private Date boardingFromDate;

    // private GenericSelectModel<Company> companies;

    @Inject
    private ServiceUser serviceUser;

    private Company company;

    public void setCompany(Company company) {
        this.company = company;
    }

    /** Permet de construire l'URL pour le lien permettant de voir tous les
     * navires d'un armateur.
     */
    public String[] getContextForShipOwner() {
        ShipOwner shipOwner = getBoatInfos().getBoat().getShipOwner();
        String shipOwnerName = shipOwner.getFirstName() + " "
                             + shipOwner.getLastName();
        waoActivationContext.setShipOwnerName(shipOwnerName);
        return waoActivationContext.toStrings();
    }

    /** /////////////////////// GLOBAL TO THE BLOCK /////////////////////// **/

    @Inject
    private WaoManager manager;

//    public boolean isActivityCalendarImportRun() {
//        return manager.isActivityCalendarImportRun();
//    }

    public Block getActiveBoatInfosBlock() {
        if (boatInfosEditable) {
            return editBoatInfos;
        }
        return displayBoatInfos;
    }

    public CompanyBoatInfos getCompanyBoatInfos() throws WaoException {
        if (companyBoatInfos == null && boatSelectedImmatriculation != null && getCompany() != null) {        
            companyBoatInfos = serviceBoat.getCompanyBoatInfos(user.getProfile().getObsProgram(),
                                boatSelectedImmatriculation, getCompany());
        }
        return companyBoatInfos;
    }

    /**
     * Retrieve companies select model for admin user. Only active companies
     * will be used for this model.
     *
     * @return the GenericSelectModel<Company>
     * @throws WaoException
     */
//    public GenericSelectModel<Company> getCompanies() throws WaoException {
//        if (companies == null) {
//            List<Company> results = serviceUser.getCompanies(true);
//            companies = new GenericSelectModel<Company>(results, Company.class, "name","topiaId",propertyAccess);
//        }
//        return companies;
//    }

    public List<Company> getCompanies() {
        List<Company> companies = serviceUser.getCompanies(true);
        return companies;
    }

    /**
     * Get the current company. The user one if it's not an admin or the
     * selected one otherwise.
     *
     * @return the current Company
     * @throws WaoException
     */
//    public Company getCompany() throws WaoException {
//        if (company == null) {
//            if (StringUtils.isNotEmpty(companySelectedId)) {
//                company = getCompanies().findObject(companySelectedId);
//            } else if (!user.isAdmin()) {
//                company = user.getCompany();
//            }
//        }
//        return company;
//    }

    public Company getCompany() {
        if (company == null) {
            if ( ! user.isAdmin()) {
                company = user.getCompany();
            }
        }
        return company;
    }

    /**
     * Used to get the selected boat.
     *
     * @return the selected boat from the grid
     * @throws WaoException
     */
    public Boat getBoatSelected() throws WaoException {
        Boat boatSelected;
        if (boatSelectedImmatriculation == null) {
            boatSelected = null;
        } else {
            boatSelected = getBoats().get(boatSelectedImmatriculation);
            if (boatSelected == null) {
                log.warn("immatriculation selected among an obsolete list of boats. imma="
                         + boatSelectedImmatriculation + " filter=" +
                         getBoatFilter());
                boats = null;
                boatSelected = getBoats().get(boatSelectedImmatriculation);
                if (boatSelected == null) {
                    log.error("boat selected is still null");
                    throw new IllegalStateException();
                }
            }
        }
        return boatSelected;
    }

    /**
     * Date for calculate boardings done for the selected boat. All boardings (number of contacts) will be calculated from this date.
     * Initialized as 12 months from today, but can be set by user.
     *
     * @return the Date used to calculate boardings done on the selected boat
     */
    public Date getBoardingFromDate() {
        if (boardingFromDate == null) {
            Calendar calendar = new GregorianCalendar();
            calendar.add(Calendar.MONTH, -12);
            boardingFromDate = calendar.getTime();
        }
        return boardingFromDate;
    }

    /**
     * Used by Tapestry to set the boardingFromDate from calculateBoardings form.
     *
     * @param boardingFromDate used to calculate boardings done on the selected boat
     */
    public void setBoardingFromDate(Date boardingFromDate) {
        this.boardingFromDate = boardingFromDate;
    }

    public BoatInfos getBoatInfos() throws WaoException {
        if (getCompanyBoatInfos() == null) {
            return null;
        }
        return getCompanyBoatInfos().getBoatInfos();
    }

    /**
     * Get CSS style class to apply on lastContact depends on its state.
     *
     * @return the CSS style to apply on contact block
     * @throws WaoException
     * @see WaoManager#getContactStyle(Contact, boolean)
     */
    public String getlastContactStyle() throws WaoException {
        Contact contact = getCompanyBoatInfos().getLastContact();
        return manager.getContactStyle(contact, user.isAdmin());
    }

    /** /////////////////////// FOR EACH ELLIGIBLEBOAT ROW ///////////////// **/

    public List<SampleRow> getSampleRows() {
        return serviceSampling.getSampleRowsForEligibility(user);
    }

    public List<ElligibleBoat> getElligibleBoatsValues() {
        return new ArrayList<ElligibleBoat>(getCompanyBoatInfos().getElligibleBoatsValues());
    }

    public String getElligibleRowInfos() {
        return manager.getTooltipSampleRow(elligibleBoat.getSampleRow());
    }

    public boolean isElligibleBoatCompanyActiveFalse() {
        return BooleanUtils.isFalse(elligibleBoat.getCompanyActive());
    }

    public boolean canCreateNewContactFromElligibleBoat() throws WaoException {
        if (!user.isAdmin() && !user.isReadOnly() &&
                !isElligibleBoatCompanyActiveFalse()) {
            Boat currentBoat = getBoatInfos().getBoat();
            return currentBoat.canCreateContact(user.getProfile().getObsProgram(), user.getCompany());
        }
        return false;
    }

    /**
     * This method detect if the sampleRow from current elligibleBoat is finished.
     * This condition is necessary for confirm message dialog, only this condition
     * is needed to display the dialog message to be sure user is aware about
     * this sampleRow state.
     *
     * @return true if the sampleRow from current elligibleBoat is finished
     */
    public boolean isSampleRowFinished() {
        boolean result = elligibleBoat.getSampleRow().isFinished();
        return result;
    }

    public String getElligibleRowStyle() {
        boolean condition1 = elligibleBoat.getCompanyActive() == null && !elligibleBoat.getGlobalActive();
        return condition1 || isElligibleBoatCompanyActiveFalse() ? "line-through" : "";
    }

    /**
     * Only user with no readOnly rights can edit BoatInfos.
     *
     * @return true if BoatInfos can be edited
     */
    public boolean canEditBoatInfos() {
        return !user.isReadOnly();
    }

    void onSuccess() throws WaoException {
        if (getBoatInfos() != null) {
            boatSelectedImmatriculation = getBoatInfos().getBoat().getImmatriculation();
        }
    }

    Block onSuccessFromCompanySelectForm() throws WaoException {
        companyBoatInfos = null;
        getCompanyBoatInfos();        
        return boatInfosZone.getBody();
    }

    Block onActionFromEditBoatInfos() throws WaoException {
        if (canEditBoatInfos()) {
            boatInfosEditable = true;
        }
        return boatInfosZone.getBody();
    }

    Block onActionFromCancelEditBoatInfos() throws WaoException {
        /*if (propertyChange.isChanged()) {
            if (log.isDebugEnabled()) {
                log.debug("Change property");
            }
        }*/
        
        // Set boatSelectedImmatriculation for boatInfos to reinitialize it (next getBoatInfos() call)
        boatSelectedImmatriculation = getBoatInfos().getBoat().getImmatriculation();
        // Suppress boatInfos to get the one from serviceBoat (not updated yet)
        companyBoatInfos = null;
        return boatInfosZone.getBody();
    }

    void onSelectedFromAddBoatInfosSampleRow() throws WaoException {
        // We stay in edition mode
        boatInfosEditable = true;
        if (boatInfosSampleRow != null) {
            companyBoatInfos.setNewElligibleBoat(boatInfosSampleRow);
        }
    }

    void onSelectedFromRemoveBoatInfosSampleRow(String sampleRowCode) throws WaoException {
        boatInfosEditable = true;
        companyBoatInfos.removeElligibleBoat(sampleRowCode);
    }

    void onSelectedFromActiveBoatInfosSampleRow(String sampleRowCode) throws WaoException {
        boatInfosEditable = true;
        companyBoatInfos.activeElligibleBoat(sampleRowCode);
    }

    Block onSuccessFromBoatInfosForm() throws WaoException {
        if (!boatInfosEditable && canEditBoatInfos()) {
            // Save data
            serviceBoat.createUpdateCompanyBoatInfos(companyBoatInfos);
        }
        //boatSelectedImmatriculation = getBoatInfos().getBoat().getImmatriculation();
        return boatInfosZone.getBody();
    }

    Block onSuccessFromCalculateBoardings() throws WaoException {
        //boatSelectedImmatriculation = getBoatInfos().getBoat().getImmatriculation();
        return boatInfosZone.getBody();
    }

    /**************************** Create new contact **************************/

    @InjectPage
    private Contacts contacts;

    @InjectPage
    private ContactForm contactForm;

    Object onActionFromAddNewContactFromBoat(int boatImmatriculation) throws WaoException {
        // Get boat from list
        boat = getBoats().get(boatImmatriculation);
        //contacts.createNewContact(boat, sampleRow);
        Contact newContact = serviceContact.newContact(user, sampleRowContext, boat);
        return redirectAfterContactCreation(newContact);
    }

    Object onActionFromAddNewContactFromSampleRow(String sampleRowCode) throws WaoException {
        // Get sampleRow from elligibleBoat list
        ElligibleBoat elligible = getCompanyBoatInfos().getElligibleBoat(sampleRowCode);
        SampleRow sampleRow = elligible.getSampleRow();
        // Keep it in manager
        filterComponent.setSampleRow(sampleRow);
        Contact newContact = serviceContact.newContact(user, sampleRow, getBoatSelected());
        return redirectAfterContactCreation(newContact);
    }

    /**
     * Après avoir créé le contact, l'utilisateur doit être redirigé soit vers
     * la liste des contacts, soit directement vers la formulaire de modification
     *
     * @param newContact le contact qui vient d'être créé
     * @return la page à retourner
     */
    protected Object redirectAfterContactCreation(Contact newContact) {
        boolean jumpToContactForm = newContact.getObsProgram() == ObsProgram.OBSDEB;
        Object page;
        if (jumpToContactForm) {
            contactForm.setContactId(newContact.getTopiaId());
            page = contactForm;
        } else {
            contacts.setContactSelectedId(newContact.getTopiaId());
            page = contacts;
        }
        return page;
    }

    /**************************** Go to last ActivityCalendar *****************/

////    @InjectPage
////    private BoatActivityCalendar calendarPage;
//
//    @Inject
//    private PageRenderLinkSource renderLink;
//
//    public boolean hasActivityCalendar() throws WaoException {
//        return getBoatSelected() != null &&
//                getBoatSelected().sizeActivityCalendar() != 0;
//    }
//
//    Link onActionFromShowLastActivityCalendar() throws WaoException {
//        Link result = renderLink.createPageRenderLinkWithContext(
//                BoatActivityCalendar.class,
//                getBoatSelected().getImmatriculation());
//        return result;
////        calendarPage.setBoat(getBoatSelected());
////        return calendarPage;
//    }

}
