/*
 * #%L
 * Wao :: Web Interface
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2009 - 2010 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package fr.ifremer.wao.ui.pages;

import fr.ifremer.wao.WaoBusinessException;
import fr.ifremer.wao.WaoBusinessException.Type;
import fr.ifremer.wao.WaoException;
import fr.ifremer.wao.bean.ConnectedUser;
import fr.ifremer.wao.bean.UserRole;
import fr.ifremer.wao.entity.WaoUser;
import fr.ifremer.wao.service.ServiceUser;
import fr.ifremer.wao.ui.components.FeedBack;
import fr.ifremer.wao.ui.data.AuthenticationUtil;
import fr.ifremer.wao.ui.services.RequiresAuthenticationFilter;
import fr.ifremer.wao.ui.services.ServiceAuthentication;
import org.apache.tapestry5.Link;
import org.apache.tapestry5.OptionModel;
import org.apache.tapestry5.PersistenceConstants;
import org.apache.tapestry5.SelectModel;
import org.apache.tapestry5.annotations.IncludeJavaScriptLibrary;
import org.apache.tapestry5.annotations.InjectComponent;
import org.apache.tapestry5.annotations.Log;
import org.apache.tapestry5.annotations.Persist;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.corelib.components.Form;
import org.apache.tapestry5.internal.OptionModelImpl;
import org.apache.tapestry5.internal.SelectModelImpl;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.services.PageRenderLinkSource;
import org.slf4j.Logger;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * Login
 *
 * Created: 3 mai 2010
 *
 * @author fdesbois
 * $Id: Connexion.java 515 2010-06-14 12:47:38Z fdesbois $
 */
@IncludeJavaScriptLibrary("context:js/wao.js")
public class Connexion {
    
    @Inject
    private Logger logger;

    @Inject
    private ServiceUser serviceUser;

    @Inject
    private ServiceAuthentication serviceAuthentication;

    @Inject
    private PageRenderLinkSource pageRender;
    
    @InjectComponent
    private FeedBack connexionFeedback;

    private Object[] activationContext;

    /**
     * User for connexion form data
     */
    @Property
    private String login;

    @Property
    private String password;

    @Persist(PersistenceConstants.FLASH)
    @Property
    private String email;

    @Persist(PersistenceConstants.FLASH)
    @Property
    private boolean showRoleWindow;

    @InjectComponent
    private Form connexionForm;

    void onActivate(Object... activationContext) {
        if (logger.isDebugEnabled()) {
            logger.debug("Activiation context : " + Arrays.toString(activationContext));
        }
        this.activationContext = activationContext;
    }

    Object[] onPassivate() {
        return activationContext;
    }

    /**
     * ON_VALIDATE_FORM :: Callback for validateForm event from connexionForm.
     * The login and password from the form will be used to connect the user.
     * The result {@link ConnectedUser} will be set to {@code serviceAuthentication}.
     * Note that users will more than one role will not have a connection role set
     * at this time. The page need to be reloaded to display the appropriate popup
     * with user roles list. Two type of errors can be catch during connection :
     * <ul>
     * <li>BAD_CONNECTION : login or password are not corresponding, forgetPassword
     *  can be used by user</li>
     * <li>ILLEGAL_CONNECTION : user is not active anymore</li>
     * </ul>
     */
    @Log
    void onValidateFormFromConnexionForm() {
        try {
            if (logger.isDebugEnabled()) {
                logger.debug("Connection for user with login : " + login);
            }
            ConnectedUser user = serviceUser.connect(login, password);
            // Authenticate the user in appropriate service
            serviceAuthentication.setConnectedUser(user);
            if (logger.isDebugEnabled()) {
                logger.debug("User connected : " + user.getFullName());
            }
        } catch (WaoBusinessException eee) {
            if (eee.getType().equals(Type.BAD_CONNECTION)) {
                connexionFeedback.addError(eee.getMessage());
                email = login;
            } else if (eee.getType().equals(Type.ILLEGAL_CONNECTION)) {
                connexionFeedback.addInfo(eee.getMessage());
            }
            if (logger.isDebugEnabled()) {
                logger.error("WaoBusinessException " + eee.getType() + " : "
                        + eee.getMessage(), eee);
            }
            // Record error to call onFailure callback, errors in form are
            // not displayed, messages are recorded in feedBack component
            connexionForm.recordError(eee.getMessage());
        }
    }

    public boolean hasConnexionErrors() {
        return connexionFeedback.hasErrors();
    }

    /**
     * ON_SUCCESS :: Callback method for success event on connexionForm.
     * A control on connectedUser roles is done to display the appropriate popup
     * if connection role is not set. Otherwise the page will be loaded depends
     * on redirectLink.
     *
     * @return the current page if user need to select a role from his list,
     *         otherwise the link to the page to load.
     * @throws WaoException
     * @see #getRedirectLink()
     */
    @Log
    Object onSuccessFromConnexionForm() throws WaoException {
        if (serviceAuthentication.getConnectedUser().getRole() == null) {
            if (logger.isDebugEnabled()) {
                logger.debug("Not only one role, window will be open to select the role");
            }
            showRoleWindow = true;
            return this;
        }
        return getRedirectLink();
    }

    /**
     * ON_FAILURE :: Callback for failure event on connexionForm. The connexion
     * page will be reloaded to display errors.
     *
     * @return the current page
     */
    @Log
    Object onFailureFromConnexionForm() {
        if (logger.isDebugEnabled()) {
            logger.debug("Failure, errors are recorded (email = " + email + ")");
        }
        return this;
    }

    /**
     * Retrieve the redirect link to load after connection submission. The
     * result link depends on {@code activationContext} provided during redirection
     * from {@link RequiresAuthenticationFilter}.
     *
     * @return the Link to load page after connection
     */
    private Link getRedirectLink() {
        String redirectPage =
                AuthenticationUtil.getRedirectPageName(activationContext);

        // If redirectPage is not defined, the default one is the Index page
        if (redirectPage == null) {
            redirectPage = "index";
        }

        Object[] redirectPageContext =
                AuthenticationUtil.getRedirectPageContext(activationContext);

        if (logger.isDebugEnabled()) {
            logger.debug("Redirect page '" + redirectPage + "' " +
                         "context : " + Arrays.toString(redirectPageContext));
        }

        Link redirectLink = null;

        // Check existence of redirectPageContext, the redirectLink depends on
        // context
        if (redirectPageContext == null) {
            redirectLink = pageRender.createPageRenderLink(redirectPage);
        } else {
            redirectLink = pageRender.createPageRenderLinkWithContext(
                                        redirectPage, redirectPageContext);
        }
        return redirectLink;
    }

    void onSuccessFromForgetPassword() throws WaoException {
        try {
            serviceUser.forgetPassword(email);
            connexionFeedback.addInfo("Un email avec votre nouveau mot de " +
                    "passe vous a été envoyé.");
        } catch (WaoBusinessException eee) {
            connexionFeedback.addError(eee.getMessage());
        }
    } 

    private SelectModel roleSelectModel;

    @Property
    private UserRole userRole;

    /**
     * Retrieve the SelectModel for user roles. Will be loaded only if user is
     * already connected. This list is useful if user as more than one role.
     *
     * @return the SelectModel for user roles
     */
    public SelectModel getRoleSelectModel() {
        if (serviceAuthentication.existConnectedUser()) {
            WaoUser user = serviceAuthentication.getConnectedUser().getUser();

            if (logger.isDebugEnabled()) {
                logger.debug("Load user roles for " + user.getLogin());
            }

            List<OptionModel> options = new ArrayList<OptionModel>();
            for (UserRole role : user.getUserRoles()) {
                OptionModel option = new OptionModelImpl(role.getLabel(), role);
                options.add(option);
            }
            roleSelectModel = new SelectModelImpl(null, options);
        } else if (roleSelectModel == null) {
            roleSelectModel = new SelectModelImpl(new OptionModelImpl(null));
        }
        return roleSelectModel;
    }

    /**
     * ON_SUCCESS :: Callback method for success event on roleForm. Will be
     * triggered when roleForm is submit on roleSelect change. The userRole
     * will be set to connectedUser and redirectPage will be loaded.
     *
     * @return the redirect link to load after submission
     * @see #getRedirectLink()
     */
    Link onSuccessFromRoleForm() {
        if (logger.isDebugEnabled()) {
            logger.debug("Connection with role : " + userRole);
        }
        serviceAuthentication.getConnectedUser().setRole(userRole);
        return getRedirectLink();
    }
}
