/*
 * #%L
 * Wao :: Web Interface
 * 
 * $Id: SamplingPlan.java 462 2010-04-19 23:41:59Z tchemit $
 * $HeadURL: svn+ssh://fdesbois@labs.libre-entreprise.org/svnroot/suiviobsmer/tags/wao-1.2/wao-ui/src/main/java/fr/ifremer/wao/ui/pages/SamplingPlan.java $
 * %%
 * Copyright (C) 2009 - 2010 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package fr.ifremer.wao.ui.pages;

import fr.ifremer.wao.WaoBusinessException;
import fr.ifremer.wao.WaoException;
import fr.ifremer.wao.bean.FacadeRow;
import fr.ifremer.wao.bean.ImportResults;
import fr.ifremer.wao.bean.SamplingFilter;
import fr.ifremer.wao.bean.SamplingFilterImpl;
import fr.ifremer.wao.entity.Company;
import fr.ifremer.wao.entity.FishingZone;
import fr.ifremer.wao.entity.SampleMonth;
import fr.ifremer.wao.entity.SampleRow;
import fr.ifremer.wao.entity.WaoUser;
import fr.ifremer.wao.service.ServiceReferential;
import fr.ifremer.wao.service.ServiceSampling;
import fr.ifremer.wao.ui.base.AbstractFilteredPage;
import fr.ifremer.wao.ui.components.Layout;
import fr.ifremer.wao.ui.data.ExportStreamResponse;
import java.io.IOException;
import java.io.InputStream;
import java.text.DateFormat;
import java.text.NumberFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import org.apache.tapestry5.StreamResponse;
import org.apache.tapestry5.annotations.IncludeStylesheet;
import org.apache.tapestry5.annotations.InjectComponent;
import org.apache.tapestry5.annotations.Log;
import org.apache.tapestry5.annotations.Persist;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.annotations.SessionState;
import org.apache.tapestry5.corelib.components.Form;
import org.apache.tapestry5.corelib.components.Zone;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.upload.services.UploadedFile;
import org.nuiton.util.PeriodDates;
import org.nuiton.util.StringUtil;
import org.nuiton.util.StringUtil.ToString;
import org.slf4j.Logger;

/**
 * SampingPlan
 *
 * Created: 9 nov. 2009
 *
 * @author fdesbois <fdesbois@codelutin.com>
 */
@IncludeStylesheet("context:css/sampling.css")
public class SamplingPlan extends AbstractFilteredPage {

    /**
     * No restriction to display this page.
     *
     * @return true
     */
    @Override
    public boolean canDisplay() {
        return true;
    }

    @Inject
    private Logger logger;

    @Inject
    private ServiceSampling serviceSampling;

    @Inject
    private ServiceReferential serviceReferential;

    @InjectComponent
    private Layout layout;

    @SessionState
    @Property
    private WaoUser user;

    private long nbTidesExpectedTime;

    private long nbTidesRealTime;

    private long totalTidesExpectedTime;

    private long totalTidesRealTime;

    /**
     * Page initialization
     */
    void setupRender() {
        if (isFiltersVisible()) {
            // Initialize filters
            initSelectFilters(true, false, true);
        }
        // Initialize fullView depends on user admin role
        if (fullView == null) {
            fullView = user.isAdmin();
        }
    }

    /**************************** IMPORT (ADMIN) *******************************/

    /**
     * Fichier CSV contenant un plan d'échantillonnage
     */
    @Property
    private UploadedFile samplingPlanCsvFile;

    public boolean canImportSamplingPlan() {
        return user.isAdmin() && !user.getReadOnly();
    }

    @Log
    void onSuccessFromImportSamplingPlan() throws WaoException {
        if (canImportSamplingPlan()) {
            try {
                ImportResults result = serviceSampling.importSamplingPlanCsv(
                        samplingPlanCsvFile.getStream());
                layout.addInfo(result.getNbRowsImported() +
                        " lignes du plan importés,  " +
                        result.getNbRowsRefused() + " refusés.");
                for (String error : result.getErrors()) {
                    layout.addInfo(error);
                }
            } catch (WaoBusinessException eee) {
                layout.addError(eee.getMessage());
            }
        }
    }
    
    /**************************** EXPORT **************************************/

    StreamResponse onActionFromExportSamplingPlan() {
        return new ExportStreamResponse("wao-echantillonnage") {

            @Override
            public InputStream getStream() throws IOException {
                InputStream result = null;
                try {
                    result = serviceSampling.exportSamplingPlanCsv(
                                                    user, getPeriod());
                } catch (WaoException eee) {
                    throw new IOException(eee);
                }
                return result;
            }
        };
    }

    /**************************** FILTERS *************************************/

    @Persist
    private SamplingFilter filter;

    @InjectComponent
    private Zone filtersZone;

    @InjectComponent
    private Form filtersForm;

    private boolean reset;

//    @Property
//    private Date periodBegin;
//
//    @Property
//    private Date periodEnd;

    public PeriodDates getPeriod() {
        return getFilter().getPeriod();
    }

    @Override
    public SamplingFilter getFilter() throws WaoException {
        if (filter == null) {
            filter = new SamplingFilterImpl();
            // Initialize period
            PeriodDates period = PeriodDates.createMonthsPeriodFromToday(11);
            filter.setPeriod(period);
        }
        return filter;
    }

    @Override
    protected boolean isAvailableDataForFiltersOnly() {
        return false;
    }

    public boolean isFiltersVisible() {
        boolean companyFiltered = getFilter().getCompany() != null
                && user.isAdmin();
        return (!getFilter().isPeriodFilteredOnly() || companyFiltered);
    }

    Object onActionFromShowFilters() {
        // Initialize filters
        initSelectFilters(true, false, true);
        return filtersZone.getBody();
    }

    public String[] onProvideCompletionsFromCodeDCF5(String input)
            throws WaoException {
        List<String> results = serviceReferential.getCodesDCF5Contains(input);
        return results.toArray(new String[0]);
    }

    void onSelectedFromReset() {
        reset = true;
    }

//    void onSuccessFromPeriodForm() {
//        // Reset data
//        data = null;
//    }
    
    Object onSuccessFromFiltersForm() {
        if (isEdited()) {
            // Reset data
//            data = null;
            return filtersZone.getBody();
        }
        if (reset) {
            // Don't reset period in filters
            PeriodDates period = getFilter().getPeriod();
            filter = null;
            getFilter().setPeriod(period);
        }
        return this;
    }

    /**************************** MAIN ACTIONS ********************************/

    @Persist
    @Property
    private Boolean fullView;

    /**
     * ACTION:: Used to change the display mode of the samplingPlan table.
     * This change affect the loading of the css style over the main table.
     * 
     * @see #getMainClass()
     */
    void onActionFromToggleDisplayMode() {
        fullView = !fullView;
    }

    void onActionFromChangeFilterEstimatedTides() {
        boolean oldValue = getFilter().getEstimatedTides();
        if (logger.isDebugEnabled()) {
            logger.debug("Change estimatedTides in filter to : " + !oldValue);
        }
        getFilter().setEstimatedTides(!oldValue);
    }

    /**************************** MAIN TABLE **********************************/

    /** ------------------------- DATA ------------------------------------- **/

    /**
     * Main data for samplingPlan : List of SampleRow ordered by FishingZone.
     */
//    @Persist
    private List<SampleRow> data;

    private List<Date> months;

    @Property
    private Date month;

    /**
     * Current SampleRow from loop
     */
    @Property
    private SampleRow row;

    @Property
    private String currentFacadeName;

    @Property
    private String currentSectorName;

    /**
     * Return List of SampleRow from suiviobsmer-business
     * @return List of SampleRow
     * @throws WaoException
     */
    public List<SampleRow> getData() throws WaoException {
        if (data == null) {            
            Company company = null;
            if (!user.isAdmin()) {
                company = user.getCompany();
            }
            FacadeRow facade =
                    serviceSampling.getSampleRowsOrderedByFishingZone(
                                                        getFilter(), company);
            data = facade.getValues();
        }
        return data;
    }

    public List<Date> getMonths() {
        if (months == null) {
            months = getPeriod().getMonths();
        }
        return months;
    }

    public boolean isFacadeChanged() {
        if (!row.getFacade().equals(currentFacadeName)) {
            currentFacadeName = row.getFacade();
            return true;
        }
        return false;
    }

    public boolean isSectorChanged() {
        if (!row.getSectors().equals(currentSectorName)) {
            currentSectorName = row.getSectors();
            return true;
        }
        return false;
    }

    public String getCompanyName() {
        return row.getCompany() != null ? row.getCompany().getName() : "";
    }

    public boolean isCurrentMonth() {
        String currentStr = getDateFormat().format(new Date());
        String monthStr = getDateFormat().format(month);
        return currentStr.equals(monthStr);
    }

    public Integer getNbTidesExpected() {
        Integer result = null;
        SampleMonth sampleMonth = row.getSampleMonth(month);
        if (sampleMonth != null) {
            result = sampleMonth.getExpectedTidesValue();
        }
        return result;
    }

    private static final ToString<FishingZone> FISHING_ZONE_TO_STRING_DISTRICT =
            new ToString<FishingZone>() {

        @Override
        public String toString(FishingZone o) {
            return o.getDistrictCode();
        }
    };

    public String getFishingZones() {
        return StringUtil.join(row.getFishingZone(),
                FISHING_ZONE_TO_STRING_DISTRICT, ", ", false);
//        String result = "";
//        for (FishingZone zone : row.getFishingZone()) {
//            result += zone.getDistrictCode() + ", ";
//        }
//        return result.substring(0, result.length()-2);

    }

    public Integer getNbTidesReal() {
        Integer result = null;
        SampleMonth sampleMonth = row.getSampleMonth(month);
        if (sampleMonth != null) {
            result = getFilter().getEstimatedTides() ?
                sampleMonth.getEstimatedTidesValue() :
                sampleMonth.getRealTidesValue();
        }
        return result;
    }

    public boolean hasNbTidesReal() {
        Date current = new Date();
        boolean validMonth = month.before(current) || isCurrentMonth();
        return validMonth && getNbTidesReal() != null;
    }

//    public int getTotalTidesExpected() {
//        long start = System.currentTimeMillis();
//        int total = 0;
//        for (Date currentMonth : getMonths()) {
//            SampleMonth sampleMonth = row.getSampleMonth(currentMonth);
//            if (sampleMonth != null) {
//                total += sampleMonth.getExpectedTidesValue();
//            }
//        }
//        long stop = System.currentTimeMillis();
//        totalTidesExpectedTime += (stop - start);
//        return total;
//    }
//
//    public int getTotalTidesReal() {
//        long start = System.currentTimeMillis();
//        int total = 0;
//        for (Date currentMonth : getMonths()) {
//            SampleMonth sampleMonth = row.getSampleMonth(currentMonth);
//            if (sampleMonth != null) {
//                total += sampleMonth.getRealTidesValue();
//            }
//        }
//        long stop = System.currentTimeMillis();
//        totalTidesRealTime += (stop - start);
//        return total;
//    }

    public String getTotalPercentage() {
        double result = 0;
        if (row.getTotalTidesExpected() > 0) {
            result = (double)row.getTotalTidesReal() / (double)row.getTotalTidesExpected();
        }
        return NumberFormat.getPercentInstance().format(result);
    }

    public NumberFormat getNumberFormat() {
        return NumberFormat.getNumberInstance();
    }

    /** ------------------------- HTML & STYLE ----------------------------- **/

    @Property
    private int rowIndex;

    public String getRowComment() {
        String comment = row.getComment();
        // Problem whith " in chenilleKit ToolTip component
        comment = comment.replace("\n", "<br />");
        return comment;
    }

    public int getNbColumnsForProfession() {
        // code, program.name, fishingZone.districts, profession.code, profession.libelle
        int fixed = 5;
        if (fullView) {
            // company, program.periodBegin, program.periodEnd, fishingZonesInfos
            // profession.meshSize, profession.size, profession.other, profession.species
            fixed += 8;
        }
        return fixed;
    }

    public int getNbColumnsForMonths() {
        return getMonths().size() + 1;
    }

    public int getNbColumnsForOther() {
        return 4;
    }

    public int getNbColumnsTotal() {
        return getNbColumnsForProfession() +
                getNbColumnsForMonths() + getNbColumnsForOther();
    }

    public String getMainClass() {
        return fullView ? "admin" : "user";
    }

    public String getParityClass() {
        return rowIndex % 2 == 0 ? "even" : "odd";
    }

    public String getActionsClass() {
        return fullView ? "width100" : "width50";
    }

    public String getRealTidesClass() {
        String result = "real-warning";
        if (getNbTidesReal() < getNbTidesExpected()) {
            result += "-inf";
        } else if (getNbTidesReal() > getNbTidesExpected()) {
            result += "-sup";
        }
        return result;
    }

    public String getMonthCurrentClass() {
        return isCurrentMonth() ? "selected" : "";
    }

    public DateFormat getDateFormat() {
        return new SimpleDateFormat("MM/yyyy");
    }

    /** ------------------------- ACTIONS ---------------------------------- **/

    /**
     * Can edit the sampleRow. Only for admin with no readOnly rights.
     * 
     * @return true if the sampleRow can be edited.
     */
    public boolean canEditSampleRow() {
        return user.isAdmin() && !user.getReadOnly();
    }

    /**
     * Used to filter period using dates from the program selected in table.
     * @param rowIndex index of the row in the table
     * @throws WaoException for a data problem
     */
    void onActionFromFilterPeriodDates(int rowIndex) throws WaoException {
        row = getData().get(rowIndex);
        getPeriod().setFromDate(row.getPeriodBegin());
        getPeriod().setThruDate(row.getPeriodEnd());
    }

    void onActionFromDeleteSampleRow(int rowIndex) throws WaoException {
        row = getData().get(rowIndex);
        serviceSampling.deleteSampleRow(row);
    }

    public boolean canAccessBoats() {
        // Test if the row isFinished with a gap of 1 month (today - 1 month)
        return !row.isFinished(-1);
    }
        
}
