package fr.ifremer.tutti.ui.swing.content.protocol;

/*
 * #%L
 * Tutti :: UI
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.data.FishingOperation;
import fr.ifremer.tutti.persistence.entities.data.FishingOperations;
import fr.ifremer.tutti.persistence.entities.data.SampleCategoryModel;
import fr.ifremer.tutti.persistence.entities.data.SampleCategoryModelEntry;
import fr.ifremer.tutti.persistence.entities.protocol.CaracteristicMappingRow;
import fr.ifremer.tutti.persistence.entities.protocol.CaracteristicType;
import fr.ifremer.tutti.persistence.entities.protocol.OperationFieldMappingRow;
import fr.ifremer.tutti.persistence.entities.protocol.SpeciesProtocol;
import fr.ifremer.tutti.persistence.entities.protocol.TuttiProtocol;
import fr.ifremer.tutti.persistence.entities.protocol.TuttiProtocols;
import fr.ifremer.tutti.persistence.entities.referential.Caracteristic;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.persistence.entities.referential.Speciess;
import fr.ifremer.tutti.service.DecoratorService;
import fr.ifremer.tutti.service.TuttiDecorator;
import fr.ifremer.tutti.ui.swing.content.home.actions.CloneProtocolAction;
import fr.ifremer.tutti.ui.swing.content.home.actions.EditProtocolAction;
import fr.ifremer.tutti.ui.swing.content.home.actions.ImportProtocolAction;
import fr.ifremer.tutti.ui.swing.content.operation.catches.SpeciesAbleBatchRowHelper;
import fr.ifremer.tutti.ui.swing.content.protocol.rtp.RtpCellEditor;
import fr.ifremer.tutti.ui.swing.content.protocol.rtp.RtpCellRenderer;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiUIHandler;
import fr.ifremer.tutti.ui.swing.util.TuttiUIUtil;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.swing.editor.bean.BeanDoubleList;
import jaxx.runtime.swing.editor.bean.BeanDoubleListModel;
import jaxx.runtime.swing.editor.bean.BeanFilterableComboBox;
import jaxx.runtime.swing.editor.bean.BeanUIUtil;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.autocomplete.ComboBoxCellEditor;
import org.jdesktop.swingx.autocomplete.ObjectToStringConverter;
import org.jdesktop.swingx.decorator.ComponentAdapter;
import org.jdesktop.swingx.decorator.HighlightPredicate;
import org.jdesktop.swingx.decorator.Highlighter;
import org.jdesktop.swingx.table.DefaultTableColumnModelExt;
import org.jdesktop.swingx.table.TableColumnExt;
import org.nuiton.decorator.Decorator;
import org.nuiton.jaxx.application.swing.table.ColumnIdentifier;
import org.nuiton.jaxx.application.swing.util.CloseableUI;

import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JTabbedPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableColumnModel;
import java.awt.Color;
import java.awt.Component;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static org.nuiton.i18n.I18n.n;
import static org.nuiton.i18n.I18n.t;


/**
 * @author Tony Chemit - chemit@codelutin.com
 * @since 0.3
 */
public class EditProtocolUIHandler extends AbstractTuttiUIHandler<EditProtocolUIModel, EditProtocolUI> implements CloseableUI {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(EditProtocolUIHandler.class);

    public static String getTitle(boolean exist) {

        String result;
        if (exist) {
            result = t("tutti.editProtocol.title.edit.protocol");
        } else {
            result = t("tutti.editProtocol.title.create.protocol");
        }
        return result;
    }

//    protected SelectSpeciesUI dialog;

    protected SampleCategoryModel sampleCategoryModel;

    protected List<BeanDoubleList<Caracteristic>> allDoubleLists;

    public JXTable getSpeciesTable() {
        return ui.getSpeciesTable();
    }

    public JXTable getBenthosTable() {
        return ui.getBenthosTable();
    }

    public JXTable getCaracteristicsMappingTable() {
        return ui.getCaracteristicsMappingTable();
    }

    public JXTable getOperationFieldsMappingTable() {
        return ui.getOperationFieldsMappingTable();
    }

    public EditProtocolSpeciesTableModel getSpeciesTableModel() {
        return (EditProtocolSpeciesTableModel) getSpeciesTable().getModel();
    }

    public EditProtocolSpeciesTableModel getBenthosTableModel() {
        return (EditProtocolSpeciesTableModel) getBenthosTable().getModel();
    }

    public boolean isSpeciesSelected(Object selectedItem) {
        return selectedItem != null && selectedItem instanceof Species;
    }

    public EditProtocolCaracteristicsTableModel getCaracteristicMappingTableModel() {
        return (EditProtocolCaracteristicsTableModel) getCaracteristicsMappingTable().getModel();
    }

    public EditProtocolOperationFieldsTableModel getEditProtocolOperationFieldsMappingTableModel() {
        return (EditProtocolOperationFieldsTableModel) getOperationFieldsMappingTable().getModel();
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    public SwingValidator<EditProtocolUIModel> getValidator() {
        return ui.getValidator();
    }

    @Override
    public void beforeInit(EditProtocolUI ui) {
        super.beforeInit(ui);

        this.sampleCategoryModel = getContext().getDataContext().getSampleCategoryModel();

        incrementsMessage("Chargement des réferentiels");

        getDataContext().resetValidationDataContext();

        EditProtocolUIModel model = new EditProtocolUIModel();

        // load cache data

        List<Species> allSpecies = Lists.newArrayList(getDataContext().getSpecies());
        model.setAllSpecies(allSpecies);

        Multimap<String, Species> allSpeciesByTaxonId =
                Speciess.splitByReferenceTaxonId(allSpecies);
        model.setAllSpeciesByTaxonId(allSpeciesByTaxonId);

        List<Species> referentSpecies = getDataContext().getReferentSpecies();
        Map<String, Species> allReferentSpeciesByTaxonId = Speciess.splitReferenceSpeciesByReferenceTaxonId(referentSpecies);
        model.setAllReferentSpeciesByTaxonId(allReferentSpeciesByTaxonId);

        List<Caracteristic> caracteristics = Lists.newArrayList(getDataContext().getCaracteristics());
        model.setCaracteristics(caracteristics);

        Map<String, Caracteristic> allCaracteristic = TuttiEntities.splitById(caracteristics);
        model.setAllCaracteristic(allCaracteristic);

        // can't load directly model from database here, since we want to
        // fill only the model with rows (transformed from speciesProtocol)
        // As we still don't have the table model at this point, wait the
        // afterUI method to fill model

        listModelIsModify(model);

        this.ui.setContextValue(model);
    }

    @Override
    public void afterInit(EditProtocolUI ui) {

        initUI(this.ui);

        EditProtocolUIModel model = getModel();

        TuttiProtocol protocol;

        if (getContext().isProtocolFilled()) {

            // load existing protocol

            incrementsMessage("Chargement du protocole");

            protocol = getDataContext().getProtocol();

            if (EditProtocolAction.CLEAN_PROTOCOL_ENTRY.getContextValue(this.ui) != null) {

                // clean protocol
                protocol = EditProtocolAction.CLEAN_PROTOCOL_ENTRY.getContextValue(this.ui);

                EditProtocolAction.CLEAN_PROTOCOL_ENTRY.removeContextValue(this.ui);

                model.setCleaned(true);
                this.ui.getSaveWarning().setText(t("tutti.editProtocol.warn.clean"));
            }
            model.fromBean(protocol);

        } else if ((protocol = ImportProtocolAction.IMPORT_PROTOCOL_ENTRY.getContextValue(this.ui)) != null) {

            incrementsMessage("Import du protocole");

            // import protocol

            ImportProtocolAction.IMPORT_PROTOCOL_ENTRY.removeContextValue(this.ui);

            model.fromBean(protocol);
            model.setImported(true);
            this.ui.getSaveWarning().setText(t("tutti.editProtocol.warn.import"));

        } else if ((protocol = CloneProtocolAction.CLONE_PROTOCOL_ENTRY.getContextValue(this.ui)) != null) {

            incrementsMessage("Clone du protocole");

            // clone protocol

            CloneProtocolAction.CLONE_PROTOCOL_ENTRY.removeContextValue(this.ui);

            model.fromBean(protocol);
            model.setCloned(true);
            this.ui.getSaveWarning().setText(t("tutti.editProtocol.warn.clone"));

        } else {

            incrementsMessage("Création d'un nouveau protocol");

            // create new protocol

            if (log.isDebugEnabled()) {
                log.debug("Will create a new protocol");
            }
        }

        if (model.getLengthClassesPmfmId() == null) {
            model.setLengthClassesPmfmId(Lists.<String>newArrayList());
        }

        SwingValidator validator = this.ui.getValidator();
        listenValidatorValid(validator, model);

        registerValidators(validator);

        Collection<Species> referents =
                model.getAllReferentSpeciesByTaxonId().values();

        initBeanFilterableComboBox(this.ui.getSpeciesComboBox(),
                                   Lists.newArrayList(referents), null);
        initBeanFilterableComboBox(this.ui.getBenthosComboBox(),
                                   Lists.newArrayList(referents), null);
        initBeanFilterableComboBox(ui.getCaracteristicMappingComboBox(),
                                   new ArrayList<Caracteristic>(model.getCaracteristics()),
                                   null);

        List<EditProtocolCaracteristicsRowModel> caracteristicMappingRows;
        List<EditProtocolOperationFieldsRowModel> operationFieldMappingRows;
        List<EditProtocolSpeciesRowModel> speciesRows;
        List<EditProtocolSpeciesRowModel> benthosRows;

        model.addPropertyChangeListener(EditProtocolUIModel.PROPERTY_IMPORT_COLUMNS, new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                populateImportColumnTableEditors();

            }
        });

        incrementsMessage("Préparation des interfaces graphiques");

        // build species and benthos rows
        if (protocol == null) {
            caracteristicMappingRows = Lists.newArrayList();
            operationFieldMappingRows = Lists.newArrayList();
            speciesRows = Lists.newArrayList();
            benthosRows = Lists.newArrayList();
        } else {

            caracteristicMappingRows = toProtocolCaracteristicRows(protocol.getCaracteristicMapping());
            operationFieldMappingRows = toProtocolOperationFieldsRows(protocol.getOperationFieldMapping());

            speciesRows = toSpeciesRows(protocol.getSpecies());
            benthosRows = toSpeciesRows(protocol.getBenthos());

            if (log.isDebugEnabled()) {
                log.debug("Will edit protocol with " +
                          speciesRows.size() + " species and " +
                          benthosRows.size() + " benthos declared.");
            }
        }

        // set to model (will propagate to tableModel)
        model.setSpeciesRow(speciesRows);
        model.setBenthosRow(benthosRows);
        model.setCaracteristicMappingRows(caracteristicMappingRows);
        model.setOperationFieldMappingRows(operationFieldMappingRows);

        this.ui.getSpeciesComboBox().reset();
        this.ui.getBenthosComboBox().reset();
        this.ui.getCaracteristicMappingComboBox().reset();

        {
            // create species table model

            JXTable table = getSpeciesTable();

            DefaultTableColumnModelExt columnModel = new DefaultTableColumnModelExt();

            TableColumnExt speciesColumn = addColumnToModel(columnModel,
                                                            null,
                                                            newTableCellRender(Species.class),
                                                            EditProtocolSpeciesTableModel.SPECIES_ID);
            speciesColumn.setSortable(true);
            DecoratorService.SpeciesDecorator speciesDecorator = new DecoratorService.SpeciesDecorator();
            speciesColumn.putClientProperty(SpeciesAbleBatchRowHelper.SPECIES_DECORATOR, speciesDecorator);
            speciesColumn.setCellRenderer(newTableCellRender(speciesDecorator));

            TableColumnExt speciesSurveyCodeColumn = addColumnToModel(columnModel,
                                                                      null,
                                                                      null,
                                                                      EditProtocolSpeciesTableModel.SURVEY_CODE_ID);
            speciesSurveyCodeColumn.setSortable(true);

            addLengthClassesColumnToModel(columnModel, model.getLengthClassesPmfmId());

            addBooleanColumnToModel(columnModel, EditProtocolSpeciesTableModel.WEIGHT_ENABLED, table);
            addBooleanColumnToModel(columnModel, EditProtocolSpeciesTableModel.COUNT_IF_NO_FREQUENCY_ENABLED, table);

            for (SampleCategoryModelEntry sampleCategoryModelEntry : sampleCategoryModel.getCategory()) {

                if (sampleCategoryModelEntry.getOrder() == 0) {
                    // first category is not editable
                    continue;
                }

                MandatorySampleCategoryColumnIdentifier identifier = MandatorySampleCategoryColumnIdentifier.newId(
                        EditProtocolSpeciesRowModel.PROPERTY_MANDATORY_SAMPLE_CATEGORY_ID,
                        sampleCategoryModelEntry.getCategoryId(),
                        sampleCategoryModelEntry.getLabel(),
                        sampleCategoryModelEntry.getLabel()
                );

                addBooleanColumnToModel(columnModel, identifier, table);
            }

            addBooleanColumnToModel(columnModel, EditProtocolSpeciesTableModel.CALCIFY_SAMPLE_ENABLED, table);
            addColumnToModel(columnModel,
                    RtpCellEditor.newEditor(ui),
                    new RtpCellRenderer(),
                    EditProtocolSpeciesTableModel.USE_RTP);

            initTable(table,
                      columnModel,
                      speciesColumn,
                      EditProtocolSpeciesTableModel.USE_RTP,
                      speciesRows,
                      new ListSelectionListener() {

                          @Override
                          public void valueChanged(ListSelectionEvent e) {
                              ListSelectionModel source = (ListSelectionModel) e.getSource();
                              getModel().setRemoveSpeciesEnabled(!source.isSelectionEmpty());
                          }
                      });
        }

        {
            // create benthos table model

            JXTable table = getBenthosTable();

            DefaultTableColumnModelExt columnModel = new DefaultTableColumnModelExt();

            TableColumnExt speciesColumn = addColumnToModel(columnModel,
                                                            null,
                                                            newTableCellRender(Species.class),
                                                            EditProtocolSpeciesTableModel.SPECIES_ID);
            speciesColumn.setSortable(true);
            DecoratorService.SpeciesDecorator speciesDecorator = new DecoratorService.SpeciesDecorator();
            speciesColumn.putClientProperty(SpeciesAbleBatchRowHelper.SPECIES_DECORATOR, speciesDecorator);
            speciesColumn.setCellRenderer(newTableCellRender(speciesDecorator));

            TableColumnExt speciesSurveyCodeColumn = addColumnToModel(columnModel,
                                                                      null,
                                                                      null,
                                                                      EditProtocolSpeciesTableModel.SURVEY_CODE_ID);
            speciesSurveyCodeColumn.setSortable(true);

            addLengthClassesColumnToModel(columnModel, model.getLengthClassesPmfmId());

            addBooleanColumnToModel(columnModel, EditProtocolSpeciesTableModel.WEIGHT_ENABLED, table);
            addBooleanColumnToModel(columnModel, EditProtocolSpeciesTableModel.COUNT_IF_NO_FREQUENCY_ENABLED, table);

            for (SampleCategoryModelEntry sampleCategoryModelEntry : sampleCategoryModel.getCategory()) {

                if (sampleCategoryModelEntry.getOrder() == 0) {
                    // first category is not editable
                    continue;
                }

                MandatorySampleCategoryColumnIdentifier identifier = MandatorySampleCategoryColumnIdentifier.newId(
                        EditProtocolSpeciesRowModel.PROPERTY_MANDATORY_SAMPLE_CATEGORY_ID,
                        sampleCategoryModelEntry.getCategoryId(),
                        sampleCategoryModelEntry.getLabel(),
                        sampleCategoryModelEntry.getLabel()
                );

                addBooleanColumnToModel(columnModel, identifier, table);
            }

            addBooleanColumnToModel(columnModel, EditProtocolSpeciesTableModel.CALCIFY_SAMPLE_ENABLED, table);

            addColumnToModel(columnModel,
                    RtpCellEditor.newEditor(ui),
                    new RtpCellRenderer(),
                    EditProtocolSpeciesTableModel.USE_RTP);

            initTable(table,
                      columnModel,
                      speciesColumn,
                      EditProtocolSpeciesTableModel.USE_RTP,
                      benthosRows,
                      new ListSelectionListener() {

                          @Override
                          public void valueChanged(ListSelectionEvent e) {
                              ListSelectionModel source = (ListSelectionModel) e.getSource();
                              getModel().setRemoveBenthosEnabled(!source.isSelectionEmpty());
                          }
                      });
        }

        allDoubleLists = Lists.newArrayList(
                this.ui.getLengthClassesList(),
                this.ui.getIndividualObservationList()
        );

        initDoubleList(EditProtocolUIModel.PROPERTY_LENGTH_CLASSES_PMFM_ID,
                       this.ui.getLengthClassesList(),
                       Lists.newArrayList(model.getCaracteristics()),
                       model.getLengthClassesPmfmId());

        initDoubleList(EditProtocolUIModel.PROPERTY_INDIVIDUAL_OBSERVATION_PMFM_ID,
                       this.ui.getIndividualObservationList(),
                       Lists.newArrayList(model.getCaracteristics()),
                       model.getIndividualObservationPmfmId());

        // init caracteristics mappingtable
        {
            JXTable caracteristicsMappingTable = getCaracteristicsMappingTable();

            DefaultTableColumnModelExt columnModel = new DefaultTableColumnModelExt();

            addColumnToModel(columnModel,
                             null,
                             newTableCellRender(Caracteristic.class),
                             EditProtocolCaracteristicsTableModel.PSFM_ID);

            addComboDataColumnToModel(columnModel,
                                      EditProtocolCaracteristicsTableModel.TYPE,
                                      getDecorator(CaracteristicType.class, null),
                                      Lists.newArrayList(CaracteristicType.getTabTypes()));

            addColumnToModel(columnModel,
                             EditProtocolCaracteristicsTableModel.IMPORT_FILE_COLUMN);

            List<Caracteristic> caracteristics = new ArrayList<Caracteristic>(getModel().getCaracteristics());
            EditProtocolCaracteristicsTableModel tableModel = new EditProtocolCaracteristicsTableModel(columnModel, caracteristics);
            caracteristicsMappingTable.setModel(tableModel);
            caracteristicsMappingTable.setColumnModel(columnModel);

            // by default do not authorize to change column orders
            caracteristicsMappingTable.getTableHeader().setReorderingAllowed(false);

            addHighlighters(caracteristicsMappingTable);

            caracteristicsMappingTable.getSelectionModel().addListSelectionListener(new ListSelectionListener() {

                @Override
                public void valueChanged(ListSelectionEvent e) {

                    int rowIndex = getCaracteristicsMappingTable().getSelectedRow();

                    boolean enableRemoveCaracteristicMapping = false;
                    boolean enableMoveUpCaracteristicMapping = false;
                    boolean enableMoveDownCaracteristicMapping = false;

                    if (rowIndex != -1) {

                        // there is a selected row
                        enableRemoveCaracteristicMapping = true;

                        enableMoveUpCaracteristicMapping = rowIndex > 0;

                        enableMoveDownCaracteristicMapping = rowIndex < getCaracteristicsMappingTable().getModel().getRowCount() - 1;
                    }
                    EditProtocolUIModel model = getModel();
                    model.setRemoveCaracteristicMappingEnabled(enableRemoveCaracteristicMapping);
                    model.setMoveUpCaracteristicMappingEnabled(enableMoveUpCaracteristicMapping);
                    model.setMoveDownCaracteristicMappingEnabled(enableMoveDownCaracteristicMapping);

                }
            });

            // always scroll to selected row
            SwingUtil.scrollToTableSelection(caracteristicsMappingTable);

            tableModel.setRows(caracteristicMappingRows);
        }

        // init operation fields table
        {
            JXTable operationFieldsTable = getUI().getOperationFieldsMappingTable();

            DefaultTableColumnModelExt columnModel = new DefaultTableColumnModelExt();

            addColumnToModel(columnModel,
                             null,
                             new DefaultTableCellRenderer() {

                                 @Override
                                 public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
                                     setText(t("tutti.editFishingOperation.field." + value.toString()));
                                     return this;
                                 }
                             },
                             EditProtocolOperationFieldsTableModel.FIELD);

            addColumnToModel(columnModel,
                             EditProtocolOperationFieldsTableModel.IMPORT_FILE_COLUMN);

            EditProtocolOperationFieldsTableModel tableModel = new EditProtocolOperationFieldsTableModel(columnModel);
            operationFieldsTable.setModel(tableModel);
            operationFieldsTable.setColumnModel(columnModel);

            // by default do not authorize to change column orders
            operationFieldsTable.getTableHeader().setReorderingAllowed(false);

            addHighlighters(operationFieldsTable);

            tableModel.setRows(operationFieldMappingRows);

            // always scroll to selected row
            SwingUtil.scrollToTableSelection(operationFieldsTable);
        }

        populateImportColumnTableEditors();

        // if new protocol can already cancel his creation
        model.setModify(model.isCreate() || model.isCleaned());

        this.ui.getCaracteristicPane().addChangeListener(new ChangeListener() {

            public void stateChanged(ChangeEvent e) {
                JTabbedPane tabPanel = (JTabbedPane) e.getSource();
                int selectedIndex = tabPanel.getSelectedIndex();
                BeanDoubleList<Caracteristic> selectedDoubleList;
                log.debug("selected " + selectedIndex);
                switch (selectedIndex) {
                    case 0:
                        selectedDoubleList = EditProtocolUIHandler.this.ui.getLengthClassesList();
                        break;

                    case 1:
                        selectedDoubleList = EditProtocolUIHandler.this.ui.getIndividualObservationList();
                        break;

                    default:
                        selectedDoubleList = null;
                }
                if (selectedDoubleList != null) {
                    selectedDoubleList.getHandler().refreshFilteredElements();

                } else {
                    // if the selected tab is the operation caracteristic mapping tab, then update the combobox
                    List<Caracteristic> selectedCaracteristics = new ArrayList<Caracteristic>(getModel().getCaracteristics());
                    for (BeanDoubleList<Caracteristic> doubleList : allDoubleLists) {
                        selectedCaracteristics.removeAll(doubleList.getModel().getSelected());
                    }
                    selectedCaracteristics.removeAll(getModel().getUsedCaracteristics());
                    getUI().getCaracteristicMappingComboBox().setData(selectedCaracteristics);
                }
            }
        });

        model.setVersion(TuttiProtocols.CURRENT_PROTOCOL_VERSION);

//        dialog = new SelectSpeciesUI(false, this.ui);

    }

    @Override
    protected JComponent getComponentToFocus() {
        return getUI().getNameField();
    }

    public List<EditProtocolSpeciesRowModel> toSpeciesRows(List<SpeciesProtocol> speciesProtocols) {
        BeanFilterableComboBox<Species> speciesComboBox = ui.getSpeciesComboBox();
        Preconditions.checkNotNull(speciesComboBox.getData());
        BeanFilterableComboBox<Species> benthosComboBox = ui.getBenthosComboBox();
        Preconditions.checkNotNull(benthosComboBox.getData());

        EditProtocolUIModel model = getModel();

        Map<String, Species> allReferentSpeciesByTaxonId = model.getAllReferentSpeciesByTaxonId();
        Map<String, Caracteristic> allCaracteristic = model.getAllCaracteristic();

        List<Caracteristic> lengthClassesPmfmId = Lists.newArrayList();

        Set<Species> speciesSet = Sets.newHashSet();
        List<EditProtocolSpeciesRowModel> result = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(speciesProtocols)) {
            for (SpeciesProtocol speciesProtocol : speciesProtocols) {
                Integer taxonId = speciesProtocol.getSpeciesReferenceTaxonId();

                if (taxonId == null) {

                    continue;
                }
                String taxonIdStr = String.valueOf(taxonId);

                // remove all synonyms from available synonym list
                Collection<Species> allSynonyms = model.getAllSynonyms(taxonIdStr);
                model.getAllSynonyms().removeAll(allSynonyms);

                // get species referent taxon
                Species species = allReferentSpeciesByTaxonId.get(taxonIdStr);

                // make sure it exists
                Preconditions.checkNotNull(species, "Espèce inconnue : " + taxonIdStr);
                speciesSet.add(species);

                EditProtocolSpeciesRowModel row = EditProtocolSpeciesTableModel.newRow(sampleCategoryModel);
                row.setSpecies(species);
                String lengthStepPmfmId = speciesProtocol.getLengthStepPmfmId();

                Caracteristic lengthStepPmfm = allCaracteristic.get(lengthStepPmfmId);
                if (lengthStepPmfmId != null &&
                    !lengthClassesPmfmId.contains(lengthStepPmfm) &&
                    !model.containsLengthClassesPmfmId(lengthStepPmfmId)) {
                    if (log.isInfoEnabled()) {
                        log.info("Found a new lengthStep pmfm: " + lengthStepPmfmId);
                    }
                    lengthClassesPmfmId.add(lengthStepPmfm);
                }
                row.setLengthStepPmfm(lengthStepPmfm);
                row.fromBean(speciesProtocol);

                // make sure to get a clean copy of the list
                row.setMandatorySampleCategoryId(Lists.newArrayList(speciesProtocol.getMandatorySampleCategoryId()));
                result.add(row);
            }

            // remove once for all in comboboxes
            speciesComboBox.removeItems(speciesSet);
            benthosComboBox.removeItems(speciesSet);
        }

        if (CollectionUtils.isNotEmpty(lengthClassesPmfmId)) {
            // detect some new length step pmfp to add in protocol

            // add it to model
            model.addAllLengthClassesPmfmId(TuttiEntities.collecIds(lengthClassesPmfmId));

            // add it to ui (no binding here, must do it manually)
            BeanDoubleList<Caracteristic> lengthClassesList = ui.getLengthClassesList();
            lengthClassesList.getModel().addToSelected(lengthClassesPmfmId);
        }
        return result;
    }

    public List<EditProtocolCaracteristicsRowModel> toProtocolCaracteristicRows(List<CaracteristicMappingRow> caracteristicMappingRows) {
        BeanFilterableComboBox<Caracteristic> caracteristicMappingComboBox = ui.getCaracteristicMappingComboBox();
        Preconditions.checkNotNull(caracteristicMappingComboBox.getData());

        EditProtocolUIModel model = getModel();

        Set<Caracteristic> caracteristicSet = Sets.newHashSet();
        List<EditProtocolCaracteristicsRowModel> result = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(caracteristicMappingRows)) {
            for (CaracteristicMappingRow caracteristicMappingRow : caracteristicMappingRows) {
                String pmfmId = caracteristicMappingRow.getPmfmId();

                if (pmfmId == null) {

                    continue;
                }

                // make sure it exists
                caracteristicSet.add(model.getAllCaracteristic().get(pmfmId));

                EditProtocolCaracteristicsRowModel row = createEditProtocolCaracteristicsRowModel();
                row.fromBean(caracteristicMappingRow);
                row.setValid(row.getType() != null);

                result.add(row);
            }

            caracteristicMappingComboBox.removeItems(caracteristicSet);
        }

        return result;
    }

    public List<EditProtocolOperationFieldsRowModel> toProtocolOperationFieldsRows(Collection<OperationFieldMappingRow> operationFieldMappingRows) {
        List<EditProtocolOperationFieldsRowModel> result = Lists.newArrayList();

        if (operationFieldMappingRows == null) {
            operationFieldMappingRows = new ArrayList<OperationFieldMappingRow>();
        }
        Map<String, OperationFieldMappingRow> rowsByField = Maps.uniqueIndex(operationFieldMappingRows,
                                                                             new Function<OperationFieldMappingRow, String>() {

                                                                                 @Override
                                                                                 public String apply(OperationFieldMappingRow operationFieldMappingRow) {
                                                                                     return operationFieldMappingRow.getField();
                                                                                 }
                                                                             });

        n("tutti.editFishingOperation.field.gearShootingStartDay");
        n("tutti.editFishingOperation.field.gearShootingStartTime");
        n("tutti.editFishingOperation.field.gearShootingEndDay");
        n("tutti.editFishingOperation.field.gearShootingEndTime");

        List<String> properties = Lists.newArrayList(
                FishingOperation.PROPERTY_STATION_NUMBER,
                FishingOperation.PROPERTY_FISHING_OPERATION_NUMBER,
                FishingOperations.PROPERTY_GEAR_SHOOTING_START_DAY,
                FishingOperations.PROPERTY_GEAR_SHOOTING_START_TIME,
                FishingOperation.PROPERTY_GEAR_SHOOTING_START_LATITUDE,
                FishingOperation.PROPERTY_GEAR_SHOOTING_START_LONGITUDE,
                FishingOperations.PROPERTY_GEAR_SHOOTING_END_DAY,
                FishingOperations.PROPERTY_GEAR_SHOOTING_END_TIME,
                FishingOperation.PROPERTY_GEAR_SHOOTING_END_LATITUDE,
                FishingOperation.PROPERTY_GEAR_SHOOTING_END_LONGITUDE,
                FishingOperation.PROPERTY_FISHING_OPERATION_RECTILIGNE,
                FishingOperation.PROPERTY_TRAWL_DISTANCE,
                FishingOperation.PROPERTY_FISHING_OPERATION_VALID,
                FishingOperation.PROPERTY_MULTIRIG_AGGREGATION,
                FishingOperation.PROPERTY_RECORDER_PERSON,
                FishingOperation.PROPERTY_GEAR,
                FishingOperation.PROPERTY_VESSEL,
                FishingOperation.PROPERTY_STRATA,
                FishingOperation.PROPERTY_SUB_STRATA,
                FishingOperation.PROPERTY_LOCATION,
                FishingOperation.PROPERTY_SECONDARY_VESSEL
        );
        //TODO voir avec Vincent comment on gère le cas où aucun des champs clé n'est renseigné

        for (String property : properties) {
            OperationFieldMappingRow row = rowsByField.get(property);
            EditProtocolOperationFieldsRowModel newRow = createOperationFieldRow(property, row != null ? row.getImportColumn() : null);
            result.add(newRow);
        }
        return result;
    }

    @Override
    public void onCloseUI() {
        if (log.isDebugEnabled()) {
            log.debug("closing: " + ui);
        }
        clearValidators();
        ui.getTabPanel().setSelectedIndex(0);

    }

    @Override
    public boolean quitUI() {
        boolean result = quitScreen(
                getModel().isValid(),
                getModel().isModify(),
                t("tutti.editProtocol.askCancelEditBeforeLeaving.cancelSaveProtocol"),
                t("tutti.editProtocol.askSaveBeforeLeaving.saveProtocol"),
                ui.getSaveButton().getAction()
        );
        return result;
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    public void addDoubleListListeners() {
        EditProtocolUIModel model = getModel();

        for (BeanDoubleList<Caracteristic> list : allDoubleLists) {
            String id = (String) list.getClientProperty("_updateListenerId");
            UpdateSelectedList updateListener = (UpdateSelectedList) list.getClientProperty("_updateListener");
            model.addPropertyChangeListener(id, updateListener);
        }
    }

    public void removeDoubleListListeners() {
        EditProtocolUIModel model = getModel();

        for (BeanDoubleList<Caracteristic> list : allDoubleLists) {
            String id = (String) list.getClientProperty("_updateListenerId");
            UpdateSelectedList updateListener = (UpdateSelectedList) list.getClientProperty("_updateListener");
            model.removePropertyChangeListener(id, updateListener);
        }
    }

//    //FIXME tchemit-2014-01-09 Bad place for this!
//    public Species openSelectOtherSpeciesDialog(String title, List<Species> species) {
//        SelectSpeciesUIModel model = dialog.getModel();
//        model.setSpecies(species);
//        model.setSelectedSpecies(null);
//
//        openDialog(dialog, title, new Dimension(400, 130));
//
//        return model.getSelectedSpecies();
//    }

    public EditProtocolCaracteristicsRowModel createEditProtocolCaracteristicsRowModel() {
        EditProtocolCaracteristicsRowModel newRow = new EditProtocolCaracteristicsRowModel(getModel().getCaracteristics());

        newRow.addPropertyChangeListener(new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                EditProtocolCaracteristicsRowModel row = (EditProtocolCaracteristicsRowModel) evt.getSource();
                EditProtocolUIModel model = getModel();

                if (EditProtocolCaracteristicsRowModel.PROPERTY_IMPORT_COLUMN.equals(evt.getPropertyName())) {
                    String oldValue = (String) evt.getOldValue();
                    if (oldValue != null) {
                        model.decNumberOfRows(oldValue);
                    }
                    String newValue = (String) evt.getNewValue();
                    if (newValue != null) {
                        model.incNumberOfRows(newValue);
                    }

                    recomputeRowsValidState();

                } else {
                    row.setValid(isCaracteristicsRowValid(row));
                }

                if (row.isValid()) {
                    model.setModify(true);
                }
            }
        });

        return newRow;
    }

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected void initDoubleList(String propertyId,
                                  BeanDoubleList<Caracteristic> widget,
                                  List<Caracteristic> availableCaracteristics,
                                  List<String> selectedCaracteristics) {

        initBeanList(widget, availableCaracteristics,
                     Lists.<Caracteristic>newArrayList());

        UpdateSelectedList listener = new UpdateSelectedList(widget, getModel().getAllCaracteristic());
        widget.putClientProperty("_updateListener", listener);
        widget.putClientProperty("_updateListenerId", propertyId);
        listener.select(selectedCaracteristics);

        // add a filter to keep only in universe everything except what is selected by other double lists
        List<BeanDoubleList<Caracteristic>> list = Lists.newArrayList(allDoubleLists);
        list.remove(widget);
        widget.getHandler().addFilter(new SelectValuePredicate(list));
        widget.getHandler().addFilter(new Predicate<Caracteristic>() {

            @Override
            public boolean apply(Caracteristic caracteristic) {
                return !getModel().isCaracteristicUsedInMapping(caracteristic);
            }
        });
    }

    protected void selectLengthClasses(List<String> ids, JComboBox comboBox) {

        Map<String, Caracteristic> allCaracteristic = getModel().getAllCaracteristic();
        List<Caracteristic> selection = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(ids)) {
            for (String id : ids) {
                selection.add(allCaracteristic.get(id));
            }
        }

        List<Caracteristic> dataToList = Lists.newArrayList(selection);

        // add a null value at first position
        if (!dataToList.isEmpty() && dataToList.get(0) != null) {
            dataToList.add(0, null);
        }
        SwingUtil.fillComboBox(comboBox, dataToList, null);
    }

    protected void addLengthClassesColumnToModel(TableColumnModel model,
                                                 List<String> selectedIds) {

        Decorator<Caracteristic> decorator =
                getDecorator(Caracteristic.class, null);

        final JComboBox comboBox = new JComboBox();

        getModel().addPropertyChangeListener(EditProtocolUIModel.PROPERTY_LENGTH_CLASSES_PMFM_ID, new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                List<String> ids = (List<String>) evt.getNewValue();
                selectLengthClasses(ids, comboBox);
            }
        });
        comboBox.setRenderer(newListCellRender(decorator));

        selectLengthClasses(selectedIds, comboBox);
        ObjectToStringConverter converter = BeanUIUtil.newDecoratedObjectToStringConverter(decorator);
        BeanUIUtil.decorate(comboBox, converter);
        ComboBoxCellEditor editor = new ComboBoxCellEditor(comboBox);

        addColumnToModel(model,
                         editor,
                         newTableCellRender(decorator),
                         EditProtocolSpeciesTableModel.LENGTH_STEP_PMFM_ID);
    }

    protected void initTable(final JXTable table,
                             DefaultTableColumnModelExt columnModel,
                             TableColumnExt speciesColumn,
                             ColumnIdentifier rtpIdentifier,
                             List<EditProtocolSpeciesRowModel> rows,
                             ListSelectionListener selectionListener) {

        final EditProtocolSpeciesTableModel tableModel =
                new EditProtocolSpeciesTableModel(sampleCategoryModel, columnModel);
        table.setModel(tableModel);
        table.setColumnModel(columnModel);

        JTableHeader tableHeader = table.getTableHeader();

        // by default do not authorize to change column orders
        tableHeader.setReorderingAllowed(false);

        addHighlighters(table);
        if (rtpIdentifier != null) {
            addRtpHighlighter(table, rtpIdentifier);
        }

        // always scroll to selected row
        SwingUtil.scrollToTableSelection(table);

        // add selection listener
        table.getSelectionModel().addListSelectionListener(selectionListener);

        // when model change, then rebuild the species comparator + set model as modified
        tableModel.addTableModelListener(new TableModelListener() {

            @Override
            public void tableChanged(TableModelEvent e) {
                getModel().setModify(true);

                int type = e.getType();
                if (type == TableModelEvent.DELETE ||
                    type == TableModelEvent.INSERT ||
                    e.getLastRow() == Integer.MAX_VALUE) {

                    // get species column
                    TableColumnExt tableColumn =
                            (TableColumnExt) table.getColumns().get(0);

                    // get column comparator
                    TuttiDecorator.TuttiDecoratorComparator<Species> comparator =
                            (TuttiDecorator.TuttiDecoratorComparator<Species>)
                                    tableColumn.getComparator();

                    // get column comparator
                    TuttiDecorator<Species> decorator =
                            SpeciesAbleBatchRowHelper.getSpeciesColumnDecorator(tableColumn);

                    boolean comparatorNull = comparator == null;
                    if (comparatorNull) {

                        // first time coming here, add the comparator
                        comparator = decorator.getCurrentComparator();
                    }

                    // init comparator with model species list
                    comparator.init(decorator, tableModel.getSpeciesList());

                    if (comparatorNull) {

                        // affect it to colum
                        tableColumn.setComparator(comparator);
                    }
                }
            }
        });

        // create popup to change species decorator

        SpeciesAbleBatchRowHelper.installSpeciesColumnComparatorPopup(
                table,
                speciesColumn,
                null,
                t("tutti.species.refTaxCode.tip"),
                t("tutti.species.name.tip")
        );

        // at the very end, set rows to model
        tableModel.setRows(rows);
    }

    protected void addRtpHighlighter(JXTable table, ColumnIdentifier identifier) {
        Color cellWithValueColor = getConfig().getColorCellWithValue();

        Highlighter commentHighlighter = TuttiUIUtil.newBackgroundColorHighlighter(
                new HighlightPredicate.AndHighlightPredicate(
                        new HighlightPredicate.IdentifierHighlightPredicate(identifier),
                        // for not null value
                        new HighlightPredicate() {
                            @Override
                            public boolean isHighlighted(Component renderer, ComponentAdapter adapter) {
                                return (boolean) adapter.getValue();
                            }
                        }), cellWithValueColor);
        table.addHighlighter(commentHighlighter);
    }

    protected EditProtocolOperationFieldsRowModel createOperationFieldRow(String property, String column) {
        EditProtocolOperationFieldsRowModel newRow = new EditProtocolOperationFieldsRowModel();
        newRow.addPropertyChangeListener(new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                EditProtocolOperationFieldsRowModel row = (EditProtocolOperationFieldsRowModel) evt.getSource();
                EditProtocolUIModel model = getModel();

                //update number of rows by column
                if (EditProtocolOperationFieldsRowModel.PROPERTY_IMPORT_COLUMN.equals(evt.getPropertyName())) {
                    String oldValue = (String) evt.getOldValue();
                    if (oldValue != null) {
                        model.decNumberOfRows(oldValue);
                    }
                    String newValue = (String) evt.getNewValue();
                    if (newValue != null) {
                        model.incNumberOfRows(newValue);
                    }

                    recomputeRowsValidState();
                }
            }
        });
        newRow.setValid(true);
        newRow.setField(property);
        newRow.setImportColumn(column);
        newRow.setValid(true);
        newRow.addPropertyChangeListener(new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                getModel().setModify(true);
            }
        });
        return newRow;
    }

    protected void recomputeRowsValidState() {
        EditProtocolUIModel model = getModel();

        for (EditProtocolCaracteristicsRowModel row : model.getCaracteristicMappingRows()) {
            row.setValid(isCaracteristicsRowValid(row));
        }

        for (EditProtocolOperationFieldsRowModel row : model.getOperationFieldMappingRows()) {
            row.setValid(isOperationFieldsRowValid(row));
        }

        getCaracteristicsMappingTable().repaint();
        getOperationFieldsMappingTable().repaint();
    }

    protected boolean isOperationFieldsRowValid(EditProtocolOperationFieldsRowModel row) {
        EditProtocolUIModel model = getModel();
        String importColumn = row.getImportColumn();
        return importColumn == null || model.numberOfRows(importColumn) < 2;
    }

    protected boolean isCaracteristicsRowValid(EditProtocolCaracteristicsRowModel row) {
        EditProtocolUIModel model = getModel();
        String importColumn = row.getImportColumn();
        return row.getType() != null &&
               (importColumn == null || model.numberOfRows(importColumn) < 2);
    }

    protected void populateImportColumnTableEditors() {
        Collection<String> importColumns = getModel().getImportColumns();
        ArrayList<String> dataToList = new ArrayList<String>();

        if (importColumns != null) {
            dataToList.addAll(importColumns);

            // add a null value at first position
            if (!dataToList.isEmpty() && dataToList.get(0) != null) {
                dataToList.add(0, null);
            }

        } else {
            dataToList.add(null);
        }

        populateImportColumnTableEditor(getUI().getCaracteristicsMappingTable(), 2, dataToList);
        populateImportColumnTableEditor(getUI().getOperationFieldsMappingTable(), 1, dataToList);
    }

    protected void populateImportColumnTableEditor(JXTable table, int columnIndex, ArrayList<String> dataToList) {
        JComboBox comboBox = new JComboBox();
        SwingUtil.fillComboBox(comboBox, dataToList, null);

        TableColumnExt col = table.getColumnExt(columnIndex);
        ComboBoxCellEditor editor = new ComboBoxCellEditor(comboBox);
        col.setCellEditor(editor);
    }

    protected static class UpdateSelectedList implements PropertyChangeListener {

        private final BeanDoubleListModel<Caracteristic> model;

        private final Map<String, Caracteristic> caracteristicMap;

        public UpdateSelectedList(BeanDoubleList<Caracteristic> doubleList,
                                  Map<String, Caracteristic> caracteristicMap) {
            this.model = doubleList.getModel();
            this.caracteristicMap = caracteristicMap;
        }

        private boolean valueIsAdjusting;

        @Override
        public void propertyChange(PropertyChangeEvent evt) {

            if (!valueIsAdjusting) {

                valueIsAdjusting = true;

                try {
                    List<String> selectedIds = (List<String>) evt.getNewValue();
                    if (log.isInfoEnabled()) {
                        log.info("[" + evt.getPropertyName() + "] selected ids: " + selectedIds);
                    }

                    select(selectedIds);
                } finally {
                    valueIsAdjusting = false;
                }
            }
        }

        public void select(List<String> selectedIds) {

            List<Caracteristic> selection = Lists.newArrayList();
            if (CollectionUtils.isNotEmpty(selectedIds)) {
                for (String selectedId : selectedIds) {
                    Caracteristic e = caracteristicMap.get(selectedId);
                    Preconditions.checkNotNull(e, "Could not find caracteristic with id: " + selectedId);
                    selection.add(e);
                }
            }
            model.setSelected(selection);
        }
    }

    protected static class SelectValuePredicate implements Predicate<Caracteristic> {

        protected final List<BeanDoubleList<Caracteristic>> lists;

        public SelectValuePredicate(List<BeanDoubleList<Caracteristic>> lists) {
            this.lists = lists;
        }

        @Override
        public boolean apply(Caracteristic input) {

            boolean result = true;
            for (BeanDoubleList<Caracteristic> list : lists) {
                if (list.getModel().getSelected().contains(input)) {
                    result = false;
                    break;
                }
            }
            return result;
        }
    }

}
