package fr.ifremer.tutti.ui.swing.content.cruise;

/*
 * #%L
 * Tutti :: UI
 * $Id: ValidateCruiseUIHandler.java 1566 2014-02-04 08:31:02Z tchemit $
 * $HeadURL: https://svn.codelutin.com/tutti/tags/tutti-3.3.4/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/cruise/ValidateCruiseUIHandler.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.nuiton.jaxx.application.swing.action.ApplicationActionUIModel;
import org.nuiton.jaxx.application.swing.util.CloseableUI;
import fr.ifremer.tutti.persistence.ProgressionModel;
import fr.ifremer.tutti.persistence.entities.data.FishingOperation;
import fr.ifremer.tutti.ui.swing.action.EditCruiseFishingOperationAction;
import fr.ifremer.tutti.ui.swing.content.operation.FishingOperationsUI;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiUIHandler;
import fr.ifremer.tutti.ui.swing.util.RemoveablePropertyChangeListener;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.validator.NuitonValidatorResult;
import org.nuiton.validator.NuitonValidatorScope;

import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPopupMenu;
import javax.swing.JTree;
import javax.swing.SwingUtilities;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;
import java.awt.Component;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.beans.PropertyChangeEvent;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static org.nuiton.i18n.I18n.t;

/**
 * @author kmorin <kmorin@codelutin.com>
 * @since 1.4
 */
public class ValidateCruiseUIHandler extends AbstractTuttiUIHandler<ValidateCruiseUIModel, ValidateCruiseUI> implements CloseableUI {

    private final static Log log = LogFactory.getLog(ValidateCruiseUIHandler.class);

    protected EditCruiseFishingOperationAction editFishingOperationAction;

    @Override
    public void beforeInit(ValidateCruiseUI ui) {

        super.beforeInit(ui);

        ValidateCruiseUIModel model = new ValidateCruiseUIModel();

        ProgressionModel progressionModel = new ProgressionModel();
        ApplicationActionUIModel actionUIModel = getContext().getActionUI().getModel();
        actionUIModel.setProgressionModel(progressionModel);
        model.setValidator(getValidateCruiseOperationsService().validateCruiseOperations(progressionModel));
        actionUIModel.setProgressionModel(null);

        model.addPropertyChangeListener(ValidateCruiseUIModel.PROPERTY_SELECTED_FISHING_OPERATION,
                                        new RemoveablePropertyChangeListener() {
                                            @Override
                                            public void propertyChange(PropertyChangeEvent evt) {
                                                FishingOperation operation = (FishingOperation) evt.getNewValue();
                                                editFishingOperationAction.setFishingOperation(operation);
                                                getContext().getActionEngine().runAction(editFishingOperationAction);
                                            }
                                        });

        this.ui.setContextValue(model);
    }

    @Override
    public void afterInit(ValidateCruiseUI ui) {
        initUI(this.ui);

        editFishingOperationAction = getContext().getActionFactory().createLogicAction(this, EditCruiseFishingOperationAction.class);

        this.ui.getOperationPanel().getModel().setSelectedFishingOperation(null);

        DefaultMutableTreeNode root = new DefaultMutableTreeNode();
        Map<FishingOperation, NuitonValidatorResult> validator = getModel().getValidator();

        for (FishingOperation operation : validator.keySet()) {
            NuitonValidatorResult result = validator.get(operation);
            OperationTreeNode node = new OperationTreeNode(operation, result);
            root.add(node);
        }
        TreeModel model = new DefaultTreeModel(root);

        JTree navigation = this.ui.getNavigation();
        navigation.setModel(model);
        navigation.setCellRenderer(new ValidationTreeCellRenderer());
        SwingUtil.expandTree(navigation);
        SwingUtil.addExpandOnClickListener(navigation);

        FishingOperationsUI operationPanel = this.ui.getOperationPanel();
        operationPanel.getTopPanel().setVisible(false);
    }

    @Override
    public void onCloseUI() {
        if (log.isDebugEnabled()) {
            log.debug("closing: " + ui);
        }
        clearValidators();
    }

    @Override
    public SwingValidator<ValidateCruiseUIModel> getValidator() {
        return null;
    }

    @Override
    protected JComponent getComponentToFocus() {
        return null;
    }

    @Override
    public boolean quitUI() {
        return true;
    }

    public void autoSelectNode(MouseEvent e, JPopupMenu popup) {
        JTree navigation = (JTree) e.getSource();

        boolean rightClick = SwingUtilities.isRightMouseButton(e);

        if (rightClick) {

            // get the path at this point
            TreePath path = navigation.getPathForLocation(e.getX(), e.getY());

            // select row (could empty selection)
            if (path == null) {
                navigation.clearSelection();

            } else {
                navigation.setSelectionPath(path);
            }

            // on right click show popup
            popup.show(navigation, e.getX(), e.getY());

        }

        TreePath selectionPath = navigation.getSelectionPath();
        FishingOperation operation = null;

        if (selectionPath != null && selectionPath.getPathCount() > 1) {
            Object o = selectionPath.getPathComponent(1);

            if (o != null) {
                operation = (FishingOperation) ((OperationTreeNode) o).getUserObject();
            }
        }
        getModel().setSelectedFishingOperation(operation);
    }

    public void openNodeMenu(KeyEvent e, JPopupMenu popup) {

        if (e.getKeyCode() == KeyEvent.VK_CONTEXT_MENU) {

            JTree navigation = (JTree) e.getSource();

            TreePath path = navigation.getSelectionPath();

            if (path != null) {
                Rectangle r = navigation.getPathBounds(path);

                // get the point in the middle lower of the node
                Point p = new Point(r.x + r.width / 2, r.y + r.height);

                popup.show(navigation, p.x, p.y);
            }
        }
    }

    public void updateCurrentOperationNode(NuitonValidatorResult validationResult) {
        JTree navigation = ui.getNavigation();
        TreePath selectionPath = navigation.getSelectionPath();
        Preconditions.checkArgument(selectionPath.getPathCount() > 1);

        OperationTreeNode operationNode = (OperationTreeNode) selectionPath.getPathComponent(1);
        operationNode.removeAllChildren();
        operationNode.createChildren(validationResult);

        DefaultTreeModel treeModel = (DefaultTreeModel) navigation.getModel();
        treeModel.reload(operationNode);
        navigation.setSelectionPath(selectionPath);
    }

    protected class ValidationTreeCellRenderer extends DefaultTreeCellRenderer {

        private static final long serialVersionUID = 1L;

        @Override
        public Component getTreeCellRendererComponent(JTree tree, Object value, boolean selected, boolean expanded, boolean leaf, int row, boolean hasFocus) {
            JLabel label = (JLabel) super.getTreeCellRendererComponent(tree, value, selected, expanded, leaf, row, hasFocus);

            if (value != null) {
                if (value instanceof OperationTreeNode) {
                    OperationTreeNode node = (OperationTreeNode) value;
                    String text = getDecorator(FishingOperation.class, null).toString(node.getUserObject());
                    label.setText(text);
                    label.setToolTipText(text);

                } else if (value instanceof MessageTreeNode) {
                    MessageTreeNode node = (MessageTreeNode) value;
                    String text = t(String.valueOf(node.getUserObject()));
                    label.setText(text);
                    label.setToolTipText(text);
                    String iconName = node.getScope().toString().toLowerCase();
                    label.setIcon(SwingUtil.createImageIcon(iconName + ".png"));
                }
            }
            return label;
        }
    }

    protected class OperationTreeNode extends DefaultMutableTreeNode {

        private static final long serialVersionUID = 1L;

        public OperationTreeNode(FishingOperation operation,
                                 NuitonValidatorResult validationResult) {
            super(operation);

            setAllowsChildren(true);
            createChildren(validationResult);
        }

        public void createChildren(NuitonValidatorResult validationResult) {
            int messageNb = 0;
            messageNb += addMessages(NuitonValidatorScope.ERROR,
                                     validationResult.getMessagesForScope(NuitonValidatorScope.ERROR));
            messageNb += addMessages(NuitonValidatorScope.WARNING,
                                     validationResult.getMessagesForScope(NuitonValidatorScope.WARNING));

            if (messageNb == 0) {
                addMessages(NuitonValidatorScope.INFO, Lists.newArrayList(t("tutti.validator.info.operation.noError")));
            }
        }

        protected int addMessages(NuitonValidatorScope scope, List<String> messages) {
            Set<String> messageSet = Sets.newHashSet(messages);
            for (String message : messageSet) {
                MessageTreeNode child = new MessageTreeNode(scope, message);
                this.add(child);
            }
            return messages.size();
        }
    }

    protected class MessageTreeNode extends DefaultMutableTreeNode {

        private static final long serialVersionUID = 1L;

        protected NuitonValidatorScope scope;

        public MessageTreeNode(NuitonValidatorScope scope, String message) {
            super(message);
            this.scope = scope;
        }

        public NuitonValidatorScope getScope() {
            return scope;
        }

    }
}
