package fr.ifremer.tutti.ui.swing.action;

/*
 * #%L
 * Tutti :: UI
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import fr.ifremer.tutti.persistence.ProgressionModel;
import fr.ifremer.tutti.persistence.entities.data.Program;
import fr.ifremer.tutti.service.genericformat.GenericFormatExportService;
import fr.ifremer.tutti.ui.swing.content.home.SelectCruiseUI;
import fr.ifremer.tutti.ui.swing.content.home.SelectCruiseUIHandler;
import fr.ifremer.tutti.ui.swing.content.home.SelectCruiseUIModel;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.jaxx.application.ApplicationBusinessException;

import java.io.File;

import static org.nuiton.i18n.I18n.t;

/**
 * To export a selected program.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 2.0
 */
public class ExportGenericFormatProgramCruisesAction extends AbstractTuttiAction<SelectCruiseUIModel, SelectCruiseUI, SelectCruiseUIHandler> {

    /** Logger. */
    private static final Log log = LogFactory.getLog(ExportGenericFormatProgramCruisesAction.class);

    protected File file;

    public ExportGenericFormatProgramCruisesAction(SelectCruiseUIHandler handler) {
        super(handler, true);
    }

    @Override
    public boolean prepareAction() throws Exception {

        boolean doAction = super.prepareAction();

        Program program = getModel().getProgram();
        if (doAction) {

            // choose file to export
            file = saveFileWithStartDirectory(
                    getConfig().getExportBackupDirectory(),
                    false,
                    "exportProgram-" + program.getName(),
                    "zip",
                    t("tutti.selectCruise.title.choose.exportProgramFile"),
                    t("tutti.selectCruise.action.chooseProgramExportFile"),
                    "^.+\\.zip$", t("tutti.common.file.genericFormat")
            );
            doAction = file != null;
        }
        return doAction;
    }

    @Override
    public void releaseAction() {
        file = null;
        super.releaseAction();
    }

    @Override
    public void doAction() throws Exception {
        Program program = getModel().getProgram();
        Preconditions.checkNotNull(program);
        Preconditions.checkNotNull(file);

        if (log.isInfoEnabled()) {
            log.info("Will save program " + program.getId() + " to file: " + file);
        }

        GenericFormatExportService service = getContext().getGenericFormatExportService();

        createProgressionModelIfRequired(1);

        ProgressionModel progressionModel = getProgressionModel();

        progressionModel.increments(t("tutti.exportProgram.action.computeNbSteps"));

        int nbSteps = service.getExportProgramNbSteps(program.getId());
        progressionModel.adaptTotal(nbSteps);


        ApplicationBusinessException exportError = null;

        try {
            service.exportProgram(program.getId(), file, progressionModel);
        } catch (ApplicationBusinessException e) {

            String errorMessage;
            errorMessage = t("tutti.exportProgram.action.exportErrors", program.getName(), e.getMessage());

            exportError = new ApplicationBusinessException(errorMessage);
        }

        handler.resetEditProgramAction();

        sendMessage(t("tutti.exportProgram.action.success", program.getName(),
                      file.getName()));

        if (exportError != null) {
            throw exportError;
        }
    }
}
