package fr.ifremer.tutti.ui.swing.util;

/*
 * #%L
 * Tutti :: UI
 * $Id: TuttiUIUtil.java 1298 2013-10-14 18:30:59Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-2.9/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/util/TuttiUIUtil.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import fr.ifremer.tutti.TuttiBusinessException;
import fr.ifremer.tutti.TuttiTechnicalException;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.service.TuttiDecorator;
import fr.ifremer.tutti.ui.swing.util.table.AbstractTuttiTableModel;
import jaxx.runtime.SwingUtil;
import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.decorator.HighlightPredicate;
import org.jdesktop.swingx.decorator.Highlighter;
import org.jdesktop.swingx.table.TableColumnExt;

import javax.swing.ButtonGroup;
import javax.swing.JLabel;
import javax.swing.JPopupMenu;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.JSeparator;
import javax.swing.JTable;
import javax.swing.SwingUtilities;
import javax.swing.table.JTableHeader;
import java.awt.Color;
import java.awt.Desktop;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.InputStream;
import java.io.Serializable;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLConnection;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.List;

import static org.nuiton.i18n.I18n._;

/**
 * Created: 14/06/12
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public final class TuttiUIUtil {

    public static final String SPECIES_DECORATOR = "decorator";

    public static final String SPECIES_DECORATOR_INDEX = "decoratorIndex";

    /** Logger. */
    private static final Log log = LogFactory.getLog(TuttiUIUtil.class);

    public static final double EARTH_RADIUS = 6378288.0;

    private TuttiUIUtil() {
        // never instanciate util class
    }

    public static void setProperty(Object bean, String property, Object value) {
        Preconditions.checkNotNull(bean);
        Preconditions.checkNotNull(property);
        try {
            PropertyUtils.setSimpleProperty(bean, property, value);
        } catch (Exception e) {
            throw new TuttiTechnicalException(_("tutti.property.set.error", property, bean.getClass().getName()), e);
        }
    }

    public static Object getProperty(Object bean, String property) {
        Preconditions.checkNotNull(bean);
        Preconditions.checkNotNull(property);
        try {
            return PropertyUtils.getSimpleProperty(bean, property);
        } catch (Exception e) {
            throw new TuttiTechnicalException(_("tutti.property.get.error", property, bean.getClass().getName()), e);
        }
    }

    public static Highlighter newBackgroundColorHighlighter(HighlightPredicate predicate, Color color) {
        return new TuttiColorHighlighter(predicate, color, false);
    }

    public static Highlighter newForegroundColorHighlighter(HighlightPredicate predicate, Color color) {
        return new TuttiColorHighlighter(predicate, color, true);
    }

    public static int computeDistanceInMeters(Float startLatitude,
                                              Float startLongitude,
                                              Float endLatitude,
                                              Float endLongitude) {

        double sLat = startLatitude * Math.PI / 180.0;
        double sLong = startLongitude * Math.PI / 180.0;
        double eLat = endLatitude * Math.PI / 180.0;
        double eLong = endLongitude * Math.PI / 180.0;

        Double d = EARTH_RADIUS *
                   (Math.PI / 2 - Math.asin(Math.sin(eLat) * Math.sin(sLat)
                                            + Math.cos(eLong - sLong) * Math.cos(eLat) * Math.cos(sLat)));
        return d.intValue();
    }

    public static String getDistanceInMilles(Float distance) {
        String distanceText;
        if (distance != null) {
            Float distanceInMilles = distance / 1852;
            distanceText = String.format("%.3f", distanceInMilles);

        } else {
            distanceText = "";
        }
        return distanceText;
    }

    private static DecimalFormatSymbols symbols;

    public static DecimalFormatSymbols getDecimalFormatSymbols() {
        if (symbols == null) {
            symbols = new DecimalFormatSymbols();
            symbols.setDecimalSeparator('.');
            symbols.setGroupingSeparator(' ');
        }
        return symbols;
    }

    private static DecimalFormat decimalFormat;

    public static DecimalFormat getDecimalFormat(int minDecimal, int maxDecimal) {
        if (decimalFormat == null) {
            decimalFormat = new DecimalFormat();
            decimalFormat.setDecimalFormatSymbols(getDecimalFormatSymbols());
            decimalFormat.setGroupingUsed(false);
        }
        decimalFormat.setMinimumFractionDigits(minDecimal);
        decimalFormat.setMaximumFractionDigits(maxDecimal);
        return decimalFormat;
    }

    public static String getWeightStringValue(Float weight) {
        String textValue;
        if (weight != null) {
            DecimalFormat weightDecimalFormat = getDecimalFormat(1, 3);
            textValue = weightDecimalFormat.format(weight);

        } else {
            textValue = "";
        }
        return textValue;
    }

    public static void openLink(URL url) {
        try {
            openLink(url.toURI());
        } catch (URISyntaxException e) {
            throw new TuttiTechnicalException(_("swing.error.cannot.open.link", url), e);
        }
    }

    public static Desktop getDesktopForBrowse() {

        if (!Desktop.isDesktopSupported()) {
            throw new TuttiTechnicalException(
                    _("swing.error.desktop.not.supported"));
        }

        Desktop desktop = Desktop.getDesktop();

        if (!desktop.isSupported(Desktop.Action.BROWSE)) {

            throw new TuttiTechnicalException(
                    _("swing.error.desktop.browse.not.supported"));
        }

        return desktop;
    }

    public static void openLink(URI uri) {

        Desktop desktop = getDesktopForBrowse();

        try {

            desktop.browse(uri);
        } catch (Exception e) {

            throw new TuttiTechnicalException(
                    _("swing.error.cannot.open.link", uri), e);
        }
    }

    public static Desktop getDesktopForMail() {

        if (!Desktop.isDesktopSupported()) {
            throw new TuttiTechnicalException(
                    _("swing.error.desktop.not.supported"));
        }

        Desktop desktop = Desktop.getDesktop();

        if (!desktop.isSupported(Desktop.Action.MAIL)) {

            throw new TuttiTechnicalException(
                    _("swing.error.desktop.mail.not.supported"));
        }

        return desktop;
    }

    public static void mail(String subject, String body) {

        Desktop desktop = getDesktopForMail();

        try {
            URI mailtoURI = new URI("mailto", null, null, "subject=" + subject + "&body=" + body, null);
            desktop.mail(mailtoURI);

        } catch (Exception e) {

            throw new TuttiTechnicalException(
                    _("swing.error.cannot.mail"), e);
        }
    }

    public static void tryToConnectToUpdateUrl(String urlAsString,
                                               String badUrlFormatI18nKey,
                                               String notReachI18nKey,
                                               String notFoundI18nKey) {
        URL url;
        // get url
        try {
            url = new URL(urlAsString);
        } catch (MalformedURLException e) {
            if (log.isDebugEnabled()) {
                log.debug("Bad url syntax at " + urlAsString, e);
            }
            throw new TuttiBusinessException(_(badUrlFormatI18nKey, urlAsString));
        }

        URLConnection urlConnection;
        // try to connect (fail if network or remote host does not exists)
        try {
            urlConnection = url.openConnection();
            urlConnection.setConnectTimeout(10000);
            urlConnection.connect();
        } catch (Exception e) {
            if (log.isDebugEnabled()) {
                log.debug("Could not connect to " + urlAsString, e);
            }
            throw new TuttiBusinessException(_(notReachI18nKey, urlAsString));
        }

        // try to open the resource (fail if resources does not exist)
        try {
            urlConnection.setReadTimeout(1000);
            InputStream inputStream = null;
            try {
                inputStream = urlConnection.getInputStream();
            } finally {
                IOUtils.closeQuietly(inputStream);
            }
        } catch (Exception e) {
            if (log.isDebugEnabled()) {
                log.debug("Could not found file at to " + urlAsString, e);
            }
            throw new TuttiBusinessException(_(notFoundI18nKey, urlAsString));
        }
    }

    public static float getRoundedLengthStep(float lengthStep, boolean aroundUp) {
        int intValue = (int) ((lengthStep + (aroundUp ? 0.001f : 0f)) * 10);
        float result = intValue / 10f;
        return result;
    }

    public static <E> TuttiDecorator<E> getSpeciesColumnDecorator(TableColumnExt tableColumn) {
        TuttiDecorator<E> decorator = (TuttiDecorator<E>)
                tableColumn.getClientProperty(SPECIES_DECORATOR);
        return decorator;
    }

    public static <R extends Serializable, T extends AbstractTuttiTableModel<R>> void installSpeciesColumnComparatorPopup(JXTable table,
                                                                                                                          TableColumnExt speciesColumn,
                                                                                                                          String... tips) {

        ButtonGroup buttonGroup = new ButtonGroup();

        SpeciesDecoratorListener<R, T> speciesDecoratorListener =
                new SpeciesDecoratorListener<R, T>(table, buttonGroup);

        TuttiDecorator<Species> decorator =
                getSpeciesColumnDecorator(speciesColumn);

        JPopupMenu popup = new JPopupMenu();
        popup.add(new JLabel(_("tutti.ui.change.species.decorator")));
        popup.add(new JSeparator());

        for (int i = 0, nbContext = decorator.getNbContext(); i < nbContext; i++) {
            String property = decorator.getProperty(i);

            String i18nName = "tutti.property." + property;
            speciesColumn.putClientProperty(i18nName, tips[i]);
            JRadioButtonMenuItem item = new JRadioButtonMenuItem(tips[i]);
            item.putClientProperty(TuttiUIUtil.SPECIES_DECORATOR_INDEX, i);
            item.addActionListener(speciesDecoratorListener);
            if (i == 0) {
                // select the first property (as it is the
                item.setSelected(true);
            }
            buttonGroup.add(item);
            popup.add(item);
        }

        // recompute the header tip using the decorator
        speciesDecoratorListener.recomputeSpeciesColumnTip();

        // listen when to show popup menu on cell header
        table.getTableHeader().addMouseListener(
                new ShowSpeciesDecoratorPopupListener(popup));

    }

    public static void selectFirstCellOnFirstRowAndStopEditing(JXTable table) {

        // select first cell
        doSelectCell(table, 0, 0);

        if (table.isEditing()) {

            // but no edit it
            table.getCellEditor().stopCellEditing();
        }
    }

    public static void selectFirstCellOnLastRow(JXTable table) {

        // select first cell
        doSelectCell(table, table.getRowCount() - 1, 0);
    }

    public static void selectFirstCellOnRow(JXTable table, int row, boolean stopEdit) {

        // select first cell
        doSelectCell(table, row, 0);

        if (stopEdit && table.isEditing()) {

            table.getCellEditor().stopCellEditing();
        }
    }

    public static void doSelectCell(JTable table,
                                    int rowIndex,
                                    int columnIndex) {

        int rowCount = table.getRowCount();
        if (rowCount == 0) {

            // no row, can not selected any cell
            if (log.isWarnEnabled()) {
                log.warn("No row in table, can not select any cell");
            }
            return;
        }
        int columnCount = table.getColumnCount();
        if (columnCount == 0) {

            // no column, can not selected any cell
            if (log.isWarnEnabled()) {
                log.warn("No column in table, can not select any cell");
            }
            return;
        }
        if (columnIndex > columnCount) {
            if (log.isWarnEnabled()) {
                log.warn(String.format("ColumnIndex: %s is more than columnCount %s", columnIndex, columnCount));
            }
            columnIndex = columnCount - 1;
        }
        if (columnIndex < 0) {
            columnIndex = 0;
        }
        if (rowIndex >= rowCount) {
            if (log.isWarnEnabled()) {
                log.warn(String.format("RowIndex: %s is more than rowCount %s", rowIndex, rowCount));
            }
            rowIndex = rowCount - 1;
        }
        if (rowIndex < 0) {
            rowIndex = 0;
        }

        table.setColumnSelectionInterval(columnIndex, columnIndex);
        table.setRowSelectionInterval(rowIndex, rowIndex);
        table.editCellAt(rowIndex, columnIndex);
    }

    protected static class SpeciesDecoratorListener<R extends Serializable, T extends AbstractTuttiTableModel<R>> implements ActionListener {

        protected final JXTable table;

        protected final ButtonGroup buttonGroup;

        protected final TuttiDecorator<Species> decorator;

        protected final TableColumnExt column;

        public SpeciesDecoratorListener(JXTable table,
                                        ButtonGroup buttonGroup) {
            this.table = table;
            this.buttonGroup = buttonGroup;
            this.column = (TableColumnExt) table.getColumn(0);
            this.decorator = TuttiUIUtil.getSpeciesColumnDecorator(column);
        }

        @Override
        public void actionPerformed(ActionEvent e) {

            JRadioButtonMenuItem source = (JRadioButtonMenuItem) e.getSource();
            buttonGroup.setSelected(source.getModel(), true);

            Integer index =
                    (Integer) source.getClientProperty(TuttiUIUtil.SPECIES_DECORATOR_INDEX);

            if (log.isInfoEnabled()) {
                log.info("Selected decorator context index: " + index);
            }

            decorator.setContextIndex(index);

            column.setComparator(decorator.getCurrentComparator());

            // recompute the header tip
            recomputeSpeciesColumnTip();

            T tableModel = (T) table.getModel();

            // keep selected rows
            List<R> rowsToReSelect = Lists.newArrayList();
            for (int rowIndex : SwingUtil.getSelectedModelRows(table)) {
                R row = tableModel.getEntry(rowIndex);
                rowsToReSelect.add(row);
            }

            // fire model (will reload the comparator)
            tableModel.fireTableDataChanged();

            // reselect rows
            for (R row : rowsToReSelect) {
                int modelRowIndex = tableModel.getRowIndex(row);
                SwingUtil.addRowSelectionInterval(table, modelRowIndex);
            }
        }

        public void recomputeSpeciesColumnTip() {
            List<String> tips = Lists.newArrayList();
            for (int i = 0, nbContext = decorator.getNbContext(); i < nbContext; i++) {
                String property = decorator.getProperty(i);

                String i18nName = "tutti.property." + property;
                String tip = (String) column.getClientProperty(i18nName);
                tips.add(tip);
            }
            String tip = Joiner.on(" - ").join(tips);
            column.setToolTipText(tip);
        }
    }

    protected static class ShowSpeciesDecoratorPopupListener extends MouseAdapter {

        private final JPopupMenu popup;

        public ShowSpeciesDecoratorPopupListener(JPopupMenu popup) {
            this.popup = popup;
        }

        @Override
        public void mouseClicked(MouseEvent e) {
            JTableHeader source = (JTableHeader) e.getSource();
            Point point = e.getPoint();
            int columnIndex = source.columnAtPoint(point);

            boolean rightClick = SwingUtilities.isRightMouseButton(e);
            if (columnIndex == 0 && rightClick) {
                e.consume();
                popup.show(source, e.getX(), e.getY());
            }
        }
    }
}
