package fr.ifremer.tutti.ui.swing.util;

/*
 * #%L
 * Tutti :: UI
 * $Id: AbstractTuttiUIHandler.java 1260 2013-10-01 12:24:29Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-2.6.1/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/util/AbstractTuttiUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import fr.ifremer.tutti.LabelAware;
import fr.ifremer.tutti.persistence.entities.TuttiEntity;
import fr.ifremer.tutti.service.DecoratorService;
import fr.ifremer.tutti.service.TuttiDataContext;
import fr.ifremer.tutti.service.WeightUnit;
import fr.ifremer.tutti.ui.swing.TuttiUIContext;
import fr.ifremer.tutti.ui.swing.config.TuttiApplicationConfig;
import fr.ifremer.tutti.ui.swing.content.MainUI;
import fr.ifremer.tutti.ui.swing.content.MainUIHandler;
import fr.ifremer.tutti.ui.swing.util.action.TuttiActionHelper;
import fr.ifremer.tutti.ui.swing.util.action.TuttiActionUI;
import fr.ifremer.tutti.ui.swing.util.attachment.ButtonAttachment;
import fr.ifremer.tutti.ui.swing.util.editor.SimpleTimeEditor;
import fr.ifremer.tutti.ui.swing.util.editor.TuttiComputedOrNotDataEditor;
import fr.ifremer.tutti.ui.swing.util.table.ColumnIdentifier;
import jaxx.runtime.JAXXUtil;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.swing.JAXXWidgetUtil;
import jaxx.runtime.swing.editor.NumberEditor;
import jaxx.runtime.swing.editor.bean.BeanDoubleList;
import jaxx.runtime.swing.editor.bean.BeanFilterableComboBox;
import jaxx.runtime.swing.editor.bean.BeanUIUtil;
import jaxx.runtime.swing.editor.cell.NumberCellEditor;
import jaxx.runtime.swing.renderer.DecoratorListCellRenderer;
import jaxx.runtime.swing.renderer.DecoratorTableCellRenderer;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXDatePicker;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.autocomplete.ComboBoxCellEditor;
import org.jdesktop.swingx.autocomplete.ObjectToStringConverter;
import org.jdesktop.swingx.table.TableColumnExt;
import org.nuiton.decorator.Decorator;
import org.nuiton.decorator.JXPathDecorator;
import org.nuiton.validator.bean.simple.SimpleBeanValidator;

import javax.swing.AbstractAction;
import javax.swing.AbstractButton;
import javax.swing.Action;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JPopupMenu;
import javax.swing.JRootPane;
import javax.swing.JScrollPane;
import javax.swing.JSpinner;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.JViewport;
import javax.swing.KeyStroke;
import javax.swing.ListCellRenderer;
import javax.swing.ListSelectionModel;
import javax.swing.ScrollPaneConstants;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.border.LineBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumnModel;
import javax.swing.text.JTextComponent;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dialog;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Frame;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.HierarchyEvent;
import java.awt.event.HierarchyListener;
import java.awt.event.ItemEvent;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.Serializable;
import java.text.ParseException;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static org.nuiton.i18n.I18n._;

/**
 * Contract of any UI handler.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public abstract class AbstractTuttiUIHandler<M, UI extends TuttiUI<M, ?>> implements UIMessageNotifier {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(AbstractTuttiUIHandler.class);

    public abstract void beforeInitUI();

    public abstract void afterInitUI();

    public abstract void onCloseUI();

    public abstract SwingValidator<M> getValidator();

    /**
     * Global application context.
     *
     * @since 0.1
     */
    protected final TuttiUIContext context;

    /**
     * UI handled.
     *
     * @since 0.1
     */
    protected final UI ui;

    protected AbstractTuttiUIHandler(TuttiUIContext context, UI ui) {
        this.context = context;
        this.ui = ui;
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    public DefaultComboBoxModel newComboModel(Object... items) {
        return new DefaultComboBoxModel(items);
    }

    public final M getModel() {
        return ui.getModel();
    }

    public final UI getUI() {
        return ui;
    }

    @Override
    public void showInformationMessage(String message) {
        context.showInformationMessage(message);
    }

    public TuttiUIContext getContext() {
        return context;
    }

    public TuttiDataContext getDataContext() {
        return getContext().getDataContext();
    }

    public TuttiApplicationConfig getConfig() {
        return context.getConfig();
    }

    public void setText(KeyEvent event, String property) {
        JTextComponent field = (JTextComponent) event.getSource();
        String value = field.getText();
        TuttiUIUtil.setProperty(getModel(), property, value);
    }

    public void setBoolean(ItemEvent event, String property) {
        boolean value = event.getStateChange() == ItemEvent.SELECTED;
        TuttiUIUtil.setProperty(getModel(), property, value);
    }

    public void setDate(ActionEvent event, String property) {
        JXDatePicker field = (JXDatePicker) event.getSource();
        Date value = field.getDate();
        TuttiUIUtil.setProperty(getModel(), property, value);
    }

    public void selectListData(ListSelectionEvent event, String property) {
        if (!event.getValueIsAdjusting()) {
            JList list = (JList) event.getSource();
            ListSelectionModel selectionModel = list.getSelectionModel();

            selectionModel.setValueIsAdjusting(true);
            try {
                List selectedList = Lists.newLinkedList();

                for (int index : list.getSelectedIndices()) {
                    Object o = list.getModel().getElementAt(index);
                    selectedList.add(o);
                }
                TuttiUIUtil.setProperty(getModel(), property, selectedList);
            } finally {
                selectionModel.setValueIsAdjusting(false);
            }
        }
    }

    public void openDialog(TuttiUI dialogContent,
                           String title, Dimension dim) {
        Component topestUI = getTopestUI();

        JDialog result;
        if (topestUI instanceof Frame) {
            result = new JDialog((Frame) topestUI, title, true);
        } else {
            result = new JDialog((Dialog) topestUI, title, true);
        }

        result.add((Component) dialogContent);
        result.setResizable(true);

        result.setSize(dim);

        final AbstractTuttiUIHandler handler = dialogContent.getHandler();

        if (handler instanceof Cancelable) {

            // add a auto-close action
            JRootPane rootPane = result.getRootPane();

            KeyStroke shortcutClosePopup = getConfig().getShortcutClosePopup();

            rootPane.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(
                    shortcutClosePopup, "close");
            rootPane.getActionMap().put("close", new AbstractAction() {
                private static final long serialVersionUID = 1L;

                @Override
                public void actionPerformed(ActionEvent e) {
                    ((Cancelable) handler).cancel();
                }
            });
        }

        result.addWindowListener(new WindowAdapter() {

            @Override
            public void windowClosed(WindowEvent e) {
                Component ui = (Component) e.getSource();
                if (log.isDebugEnabled()) {
                    log.debug("Destroy ui " + ui);
                }
                JAXXUtil.destroy(ui);
            }
        });
        SwingUtil.center(getContext().getMainUI(), result);
        result.setVisible(true);
    }

    public void closeDialog(TuttiUI ui) {
        SwingUtil.getParentContainer(ui, JDialog.class).setVisible(false);
    }

    public static final String CONFIRMATION_FORMAT = "<html>%s<hr/><br/>%s</html>";

    public int askSaveBeforeLeaving(String message) {
        String htmlMessage = String.format(
                CONFIRMATION_FORMAT,
                message,
                _("tutti.common.askSaveBeforeLeaving.help"));
        int result = JOptionPane.showConfirmDialog(
                getTopestUI(),
                htmlMessage,
                _("tutti.common.askSaveBeforeLeaving.title"),
                JOptionPane.YES_NO_CANCEL_OPTION,
                JOptionPane.QUESTION_MESSAGE);
        return result;
    }

    public boolean askCancelEditBeforeLeaving(String message) {
        String htmlMessage = String.format(
                CONFIRMATION_FORMAT,
                message,
                _("tutti.common.askCancelEditBeforeLeaving.help"));
        int i = JOptionPane.showConfirmDialog(
                getTopestUI(),
                htmlMessage,
                _("tutti.common.askCancelEditBeforeLeaving.title"),
                JOptionPane.OK_CANCEL_OPTION,
                JOptionPane.QUESTION_MESSAGE);

        boolean result = i == JOptionPane.OK_OPTION;
        return result;
    }

    public void showSuccessMessage(String title, String message) {

        Component topestUI = getTopestUI();
        boolean alwaysOnTop = false;

        if (topestUI instanceof JDialog) {
            alwaysOnTop = ((JDialog) topestUI).isAlwaysOnTop();
        }

        if (alwaysOnTop) {
            ((JDialog) topestUI).setAlwaysOnTop(false);
        }
        try {

            JOptionPane.showMessageDialog(
                    topestUI,
                    message,
                    title,
                    JOptionPane.INFORMATION_MESSAGE,
                    UIManager.getIcon("info")
            );
        } finally {
            if (alwaysOnTop) {
                ((JDialog) topestUI).setAlwaysOnTop(true);
            }
        }
    }

    public Component getTopestUI() {
        Component result;
        TuttiActionUI actionUI = getContext().getActionUI();
//        if (actionUI.isVisible()) {
        result = actionUI;
//        } else {
//            result = getContext().getMainUI();
//        }
        return result;
    }

    public void clearValidators() {
        MainUI main = context.getMainUI();
        Preconditions.checkNotNull(
                main, "No mainUI registred in application context");
        MainUIHandler handler = main.getHandler();
        handler.clearValidators();
    }

    public <O> Decorator<O> getDecorator(Class<O> type, String name) {
        DecoratorService decoratorService =
                context.getDecoratorService();

        Preconditions.checkNotNull(type);

        Decorator decorator = decoratorService.getDecoratorByType(type, name);
        if (decorator == null) {

            if (LabelAware.class.isAssignableFrom(type)) {
                decorator = getDecorator(LabelAware.class, null);
            }
        }
        Preconditions.checkNotNull(decorator);
        return decorator;
    }

    public void autoSelectRowInTable(MouseEvent e, JPopupMenu popup) {

        boolean rightClick = SwingUtilities.isRightMouseButton(e);

        if (rightClick || SwingUtilities.isLeftMouseButton(e)) {

            // get the coordinates of the mouse click
            Point p = e.getPoint();

            JXTable source = (JXTable) e.getSource();

            int[] selectedRows = source.getSelectedRows();
            int[] selectedColumns = source.getSelectedColumns();

            // get the row index at this point
            int rowIndex = source.rowAtPoint(p);

            // get the column index at this point
            int columnIndex = source.columnAtPoint(p);

            if (log.isDebugEnabled()) {
                log.debug("At point [" + p + "] found Row " + rowIndex + ", Column " + columnIndex);
            }

            boolean canContinue = true;

            if (source.isEditing()) {

                // stop editing
                boolean stopEdit = source.getCellEditor().stopCellEditing();
                if (!stopEdit) {
                    if (log.isWarnEnabled()) {
                        log.warn("Could not stop edit cell...");
                    }
                    canContinue = false;
                }
            }

            if (canContinue) {

                // select row (could empty selection)
                if (rowIndex == -1) {
                    source.clearSelection();
                } else if (!ArrayUtils.contains(selectedRows, rowIndex)) {
                    source.setRowSelectionInterval(rowIndex, rowIndex);
                }

                // select column (could empty selection)
                if (columnIndex == -1) {
                    source.clearSelection();
                } else if (!ArrayUtils.contains(selectedColumns, columnIndex)) {
                    source.setColumnSelectionInterval(columnIndex, columnIndex);
                }

                if (rightClick) {

                    beforeOpenPopup(rowIndex, columnIndex);

                    // on right click show popup
                    popup.show(source, e.getX(), e.getY());
                }
            }
        }
    }

    public void openRowMenu(KeyEvent e, JPopupMenu popup) {

        if (e.getKeyCode() == KeyEvent.VK_CONTEXT_MENU) {

            JXTable source = (JXTable) e.getSource();

            // get the lowest selected row
            int[] selectedRows = source.getSelectedRows();
            int lowestRow = -1;
            for (int row : selectedRows) {
                lowestRow = Math.max(lowestRow, row);
            }
            // get the selected column
            int selectedColumn = source.getSelectedColumn();
            Rectangle r = source.getCellRect(lowestRow, selectedColumn, true);

            // get the point in the middle lower of the cell
            Point p = new Point(r.x + r.width / 2, r.y + r.height);

            if (log.isDebugEnabled()) {
                log.debug("Row " + lowestRow + " found t point [" + p + "]");
            }

            boolean canContinue = true;

            if (source.isEditing()) {

                // stop editing
                boolean stopEdit = source.getCellEditor().stopCellEditing();
                if (!stopEdit) {
                    if (log.isWarnEnabled()) {
                        log.warn("Could not stop edit cell...");
                    }
                    canContinue = false;
                }
            }

            if (canContinue) {

                beforeOpenPopup(lowestRow, selectedColumn);

                popup.show(source, p.x, p.y);
            }
        }
    }

    //------------------------------------------------------------------------//
    //-- Init methods                                                       --//
    //------------------------------------------------------------------------//

    protected void initUI(TuttiUI ui) {

        for (Map.Entry<String, Object> entry : ui.get$objectMap().entrySet()) {
            Object component = entry.getValue();
            if (component instanceof NumberEditor) {

                initNumberEditor((NumberEditor) component);
            } else if (component instanceof JXDatePicker) {

                initDatePicker((JXDatePicker) component);
            } else if (component instanceof SimpleTimeEditor) {

                initTimeEditor((SimpleTimeEditor) component);
            } else if (component instanceof ButtonAttachment) {

                initButtonAttachment((ButtonAttachment) component);
            } else if (component instanceof JLabel) {

                initLabel((JLabel) component);
            } else if (component instanceof JTextField) {

                initTextField((JTextField) component);
            } else if (component instanceof AbstractButton) {

                initButton((AbstractButton) component);
            } else if (component instanceof JScrollPane) {

                initScrollPane((JScrollPane) component);
            }
        }

        ((Component) ui).addHierarchyListener(new HierarchyListener() {
            @Override
            public void hierarchyChanged(HierarchyEvent e) {
                JComponent component = getComponentToFocus();
                if ((e.getChangeFlags() & HierarchyEvent.SHOWING_CHANGED) > 0
                    && e.getChanged().isShowing()
                    && component != null) {
                    component.requestFocus();
                }
            }
        });
    }

    protected abstract JComponent getComponentToFocus();

    protected void initTextField(JTextField jTextField) {
        Boolean computed = (Boolean) jTextField.getClientProperty("computed");
        if (computed != null && computed) {
            Font font = jTextField.getFont().deriveFont(Font.ITALIC);
            jTextField.setFont(font);
            jTextField.setEditable(!computed);
            jTextField.setEnabled(!computed);
            jTextField.setDisabledTextColor(getConfig().getColorComputedWeights());
        }
        if (isAutoSelectOnFocus(jTextField)) {
            addAutoSelectOnFocus(jTextField);
        }
    }

    protected void initLabel(JLabel jLabel) {
        Boolean strongStyle = (Boolean) jLabel.getClientProperty("strongStyle");
        Boolean italicStyle = (Boolean) jLabel.getClientProperty("italicStyle");
        Font font = jLabel.getFont();
        int style = font.getStyle();
        if (strongStyle != null && strongStyle) {
            style |= Font.BOLD;
        }
        if (italicStyle != null && italicStyle) {
            style |= Font.ITALIC;
        }
        jLabel.setFont(font.deriveFont(style));

        WeightUnit weightUnit = (WeightUnit) jLabel.getClientProperty("addWeightUnit");
        if (weightUnit != null) {
            String text = weightUnit.decorateLabel(jLabel.getText());
            jLabel.setText(text);

            String tip = weightUnit.decorateTip(jLabel.getToolTipText());
            jLabel.setToolTipText(tip);

            Component labelFor = jLabel.getLabelFor();
            if (labelFor instanceof TuttiComputedOrNotDataEditor) {

                // set also the number of digits (4 for kg, 1 for g)
                TuttiComputedOrNotDataEditor editor = (TuttiComputedOrNotDataEditor) labelFor;
                editor.setNumberPattern(weightUnit.getNumberEditorPattern());
                editor.setDecimalNumber(weightUnit.getNumberDigits());
            } else if (labelFor instanceof NumberEditor) {

                // set also the number of digits (4 for kg, 1 for g)
                NumberEditor editor = (NumberEditor) labelFor;
                editor.setNumberPattern(weightUnit.getNumberEditorPattern());
            }
        }
    }

    protected void initButtonAttachment(ButtonAttachment component) {

        component.init();
    }

    protected void initButton(AbstractButton abstractButton) {

        Class actionName = (Class) abstractButton.getClientProperty("tuttiAction");
        if (actionName != null) {
            Action action = TuttiActionHelper.createUIAction(this, abstractButton, actionName);
            abstractButton.setAction(action);
        }
    }

    /**
     * Prépare un component de choix d'entités pour un type d'entité donné et
     * pour un service de persistance donné.
     *
     * @param comboBox le component graphique à initialiser
     */
    protected <E extends Serializable> void initBeanFilterableComboBox(
            BeanFilterableComboBox<E> comboBox,
            List<E> data,
            E selectedData) {

        initBeanFilterableComboBox(comboBox, data, selectedData, null);
    }

    protected <E extends Serializable> void initBeanFilterableComboBox(
            BeanFilterableComboBox<E> comboBox,
            List<E> data,
            E selectedData,
            String decoratorContext) {

        Preconditions.checkNotNull(comboBox, "No comboBox!");

        Class<E> beanType = comboBox.getBeanType();

        Preconditions.checkNotNull(beanType, "No beanType on the combobox!");

        Decorator<E> decorator = getDecorator(beanType, decoratorContext);

        if (data == null) {
            data = Lists.newArrayList();
        }

        if (log.isDebugEnabled()) {
            log.debug("entity comboBox list [" + beanType.getName() + "] : " +
                      (data == null ? 0 : data.size()));
        }

        comboBox.setI18nPrefix("tutti.property.");

        // add data list to combo box
        comboBox.init((JXPathDecorator<E>) decorator, data);

        comboBox.setSelectedItem(selectedData);

        if (log.isDebugEnabled()) {
            log.debug("combo [" + beanType.getName() + "] : " +
                      comboBox.getData().size());
        }
    }

    /**
     * Prépare un component de choix d'entités pour un type d'entité donné et
     * pour un service de persistance donné.
     *
     * @param list         le component graphique à initialiser
     * @param data         la liste des données à mettre dans la liste de gauche
     * @param selectedData la liste des données à mettre dans la liste de droite
     */
    protected <E extends TuttiEntity> void initBeanList(
            BeanDoubleList<E> list,
            List<E> data,
            List<E> selectedData,
            Decorator<E> selectedDecorator) {

        Preconditions.checkNotNull(list, "No list!");

        Class<E> beanType = list.getBeanType();
        Preconditions.checkNotNull(beanType, "No beanType on the double list!");

        DecoratorService decoratorService =
                context.getDecoratorService();
        Decorator<E> decorator = decoratorService.getDecoratorByType(beanType);

        if (log.isDebugEnabled()) {
            log.debug("entity list [" + beanType.getName() + "] : " +
                      (data == null ? 0 : data.size()));
        }

        list.setI18nPrefix("tutti.property.");

        // add data list to combo box
        list.init((JXPathDecorator<E>) decorator,
                  (JXPathDecorator<E>) selectedDecorator,
                  data,
                  selectedData);

        if (log.isDebugEnabled()) {
            log.debug("Jlist [" + beanType.getName() + "] : " +
                      list.getUniverseList().getModel().getSize());
        }
    }

    /**
     * Prépare un component de choix d'entités pour un type d'entité donné et
     * pour un service de persistance donné.
     *
     * @param list         le component graphique à initialiser
     * @param data         la liste des données à mettre dans la liste de gauche
     * @param selectedData la liste des données à mettre dans la liste de droite
     */
    protected <E extends TuttiEntity> void initBeanList(
            BeanDoubleList<E> list,
            List<E> data,
            List<E> selectedData) {

        initBeanList(list, data, selectedData, null);
    }

    protected void initNumberEditor(NumberEditor editor) {
        if (log.isDebugEnabled()) {
            log.debug("init number editor " + editor.getName());
        }
        editor.init();

        // Force binding if value is already in model
        Number model = editor.getModel();
        if (model != null) {
            editor.setModel(null);
            editor.setModel(model);
        }

        if (isAutoSelectOnFocus(editor)) {

            addAutoSelectOnFocus(editor.getTextField());
        }
    }

    protected void initTimeEditor(SimpleTimeEditor editor) {
        if (log.isDebugEnabled()) {
            log.debug("init time editor " + editor.getName() +
                      " for property " + editor.getModel().getProperty());
        }
        editor.init();

        if (isAutoSelectOnFocus(editor)) {
            addAutoSelectOnFocus(((JSpinner.DefaultEditor) editor.getHour().getEditor()).getTextField());
            addAutoSelectOnFocus(((JSpinner.DefaultEditor) editor.getMinute().getEditor()).getTextField());
        }
    }

    protected void initDatePicker(final JXDatePicker picker) {

        if (log.isDebugEnabled()) {
            log.debug("disable JXDatePicker editor" + picker.getName());
        }
        String dateFormat = getConfig().getDateFormat();
        picker.setFormats(dateFormat);
        picker.setToolTipText(_("tutti.common.datefield.tip", dateFormat));
        picker.getEditor().addFocusListener(new FocusAdapter() {

            @Override
            public void focusLost(FocusEvent e) {
                try {
                    picker.commitEdit();

                } catch (ParseException ex) {
                    if (log.isDebugEnabled()) {
                        log.debug("format error", ex);
                    }
                    picker.requestFocus();
                }
            }
        });

        if (isAutoSelectOnFocus(picker)) {
            addAutoSelectOnFocus(picker.getEditor());
        }
    }

    protected void initScrollPane(JScrollPane scrollPane) {
        Boolean onlyVerticalScrollable = (Boolean) scrollPane.getClientProperty("onlyVerticalScrollable");
        if (onlyVerticalScrollable != null && onlyVerticalScrollable) {
            scrollPane.setHorizontalScrollBarPolicy(ScrollPaneConstants.HORIZONTAL_SCROLLBAR_NEVER);
            scrollPane.setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED);

            final JViewport viewport = scrollPane.getViewport();
            viewport.addChangeListener(new ChangeListener() {
                @Override
                public void stateChanged(ChangeEvent e) {
                    Dimension newDimension = new Dimension(viewport.getExtentSize().width,
                                                           viewport.getViewSize().height);
                    viewport.setViewSize(newDimension);
                }
            });
        }
    }

    protected boolean isAutoSelectOnFocus(JComponent comp) {
        Object selectOnFocus = comp.getClientProperty("selectOnFocus");
        return selectOnFocus != null && Boolean.valueOf(selectOnFocus.toString());
    }

    protected void addAutoSelectOnFocus(JTextField jTextField) {
        jTextField.addFocusListener(new FocusAdapter() {
            @Override
            public void focusGained(final FocusEvent e) {
                SwingUtilities.invokeLater(new Runnable() {
                    @Override
                    public void run() {
                        JTextField source = (JTextField) e.getSource();
                        source.selectAll();
                    }
                });

            }
        });
    }

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected boolean quitScreen(boolean modelIsValid,
                                 boolean modelIsModify,
                                 String askGiveUpMessage,
                                 String askSaveMessage,
                                 Action saveAction) {
        boolean result;

        if (!modelIsValid) {

            // model is not valid
            // ask user to qui or not
            result = askCancelEditBeforeLeaving(askGiveUpMessage);

        } else if (modelIsModify) {

            // something is modify ask user what to do
            int answer = askSaveBeforeLeaving(askSaveMessage);
            switch (answer) {
                case JOptionPane.YES_OPTION:

                    // ok save
                    saveAction.actionPerformed(null);
                    result = true;
                    break;
                case JOptionPane.NO_OPTION:

                    // do not save but can still quit the screen (so nothing to do)
                    result = true;
                    break;
                default:
                    // do not save and stay here (so nothing to do)
                    result = false;

            }
        } else {

            // model is valid and not modify, can safely quit screen
            result = true;
        }
        return result;
    }

    protected void registerValidators(SwingValidator... validators) {
        MainUI main = context.getMainUI();
        Preconditions.checkNotNull(
                main, "No mainUI registred in application context");
        MainUIHandler handler = main.getHandler();
        handler.clearValidators();
        for (SwingValidator validator : validators) {
            handler.registerValidator(validator);
        }
    }

    protected String decorate(Object object) {
        return decorate(object, null);
    }

    protected String decorate(Object object, String context) {
        String result = "";
        if (object != null) {
            result = getDecorator(object.getClass(), context).toString(object);
        }
        return result;
    }

    protected <O> ListCellRenderer newListCellRender(Class<O> type) {

        return newListCellRender(type, null);
    }

    protected <O> ListCellRenderer newListCellRender(Class<O> type, String name) {

        Decorator<O> decorator = getDecorator(type, name);
        return newListCellRender(decorator);
    }

    protected <O> ListCellRenderer newListCellRender(Decorator<O> decorator) {

        Preconditions.checkNotNull(decorator);

        ListCellRenderer result = new DecoratorListCellRenderer(decorator);
        return result;
    }

    protected void listenValidatorValid(SimpleBeanValidator validator,
                                        final AbstractTuttiBeanUIModel model) {
        validator.addPropertyChangeListener(SimpleBeanValidator.VALID_PROPERTY, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                if (log.isDebugEnabled()) {
                    log.debug("Model [" + model +
                              "] pass to valid state [" +
                              evt.getNewValue() + "]");
                }
                model.setValid((Boolean) evt.getNewValue());
            }
        });
    }

    protected void listenValidationTableHasNoFatalError(final SimpleBeanValidator validator,
                                                        final AbstractTuttiBeanUIModel model) {
        getContext().getMainUI().getValidatorMessageWidget().addTableModelListener(new TableModelListener() {
            @Override
            public void tableChanged(TableModelEvent e) {
                boolean valid = !validator.hasFatalErrors();
                if (log.isDebugEnabled()) {
                    log.debug("Model [" + model +
                              "] pass to valid state [" + valid + "]");
                }
                model.setValid(valid);
            }
        });
    }

    protected void listModelIsModify(AbstractTuttiBeanUIModel model) {
        model.addPropertyChangeListener(new PropertyChangeListener() {

            final Set<String> excludeProperties = getPropertiesToIgnore();

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                if (!excludeProperties.contains(evt.getPropertyName())) {
                    ((AbstractTuttiBeanUIModel) evt.getSource()).setModify(true);
                }
            }
        });
    }

    protected Set<String> getPropertiesToIgnore() {
        return Sets.newHashSet(
                AbstractTuttiBeanUIModel.PROPERTY_MODIFY,
                AbstractTuttiBeanUIModel.PROPERTY_VALID);
    }

    protected void closeUI(TuttiUI ui) {
        ui.getHandler().onCloseUI();
    }

    protected <B> void changeValidatorContext(String newContext,
                                              SwingValidator<B> validator) {
        B bean = validator.getBean();
        validator.setContext(newContext);
        validator.setBean(bean);
    }

    protected <R extends AbstractTuttiBeanUIModel> void addColumnToModel(TableColumnModel model,
                                                                         TableCellEditor editor,
                                                                         TableCellRenderer renderer,
                                                                         ColumnIdentifier<R> identifier) {

        addColumnToModel(model, editor, renderer, identifier, null);
    }


    protected <R extends AbstractTuttiBeanUIModel> void addColumnToModel(TableColumnModel model,
                                                                         TableCellEditor editor,
                                                                         TableCellRenderer renderer,
                                                                         ColumnIdentifier<R> identifier,
                                                                         WeightUnit weightUnit) {

        TableColumnExt col = new TableColumnExt(model.getColumnCount());
        col.setCellEditor(editor);
        col.setCellRenderer(renderer);
        String label = _(identifier.getHeaderI18nKey());
        if (weightUnit != null) {
            label = weightUnit.decorateLabel(label);
        }
        col.setHeaderValue(label);
        String tip = _(identifier.getHeaderTipI18nKey());
        if (weightUnit != null) {
            tip = weightUnit.decorateTip(tip);
        }
        col.setToolTipText(tip);

        col.setIdentifier(identifier);
        model.addColumn(col);
    }

    protected <R extends AbstractTuttiBeanUIModel> void addColumnToModel(TableColumnModel model,
                                                                         ColumnIdentifier<R> identifier) {

        addColumnToModel(model, null, null, identifier, null);
    }

    protected <R extends AbstractTuttiBeanUIModel> void addFloatColumnToModel(TableColumnModel model,
                                                                              ColumnIdentifier<R> identifier,
                                                                              String numberPattern) {

        NumberCellEditor<Float> editor =
                JAXXWidgetUtil.newNumberTableCellEditor(Float.class, false);
        editor.getNumberEditor().setSelectAllTextOnError(true);
        editor.getNumberEditor().getTextField().setBorder(new LineBorder(Color.GRAY, 2));
        editor.getNumberEditor().setNumberPattern(numberPattern);

        addColumnToModel(model, editor, null, identifier, null);
    }

//    protected <R extends AbstractTuttiBeanUIModel> void addFloatColumnToModel(TableColumnModel model,
//                                                                              ColumnIdentifier<R> identifier,
//                                                                              String numberPattern,
//                                                                              WeightUnit weightUnit) {
//
//        NumberCellEditor<Float> editor =
//                JAXXWidgetUtil.newNumberTableCellEditor(Float.class, false);
//        editor.getNumberEditor().setSelectAllTextOnError(true);
//        editor.getNumberEditor().getTextField().setBorder(new LineBorder(Color.GRAY, 2));
//        editor.getNumberEditor().setNumberPattern(numberPattern);
//
//        addColumnToModel(model, editor, null, identifier, weightUnit);
//    }

    protected <R extends AbstractTuttiBeanUIModel> void addFloatColumnToModel(TableColumnModel model,
                                                                              ColumnIdentifier<R> identifier,
                                                                              WeightUnit weightUnit) {

        Preconditions.checkNotNull(weightUnit);
        NumberCellEditor<Float> editor =
                JAXXWidgetUtil.newNumberTableCellEditor(Float.class, false);
        editor.getNumberEditor().setSelectAllTextOnError(true);
        editor.getNumberEditor().getTextField().setBorder(new LineBorder(Color.GRAY, 2));
        editor.getNumberEditor().setNumberPattern(weightUnit.getNumberEditorPattern());

        addColumnToModel(model, editor, null, identifier, weightUnit);
    }

    protected <R extends AbstractTuttiBeanUIModel> void addIntegerColumnToModel(TableColumnModel model,
                                                                                ColumnIdentifier<R> identifier,
                                                                                String numberPattern) {

        NumberCellEditor<Integer> editor =
                JAXXWidgetUtil.newNumberTableCellEditor(Integer.class, false);
        editor.getNumberEditor().setSelectAllTextOnError(true);
        editor.getNumberEditor().getTextField().setBorder(new LineBorder(Color.GRAY, 2));
        editor.getNumberEditor().setNumberPattern(numberPattern);

        addColumnToModel(model, editor, null, identifier, null);
    }

    protected <R extends AbstractTuttiBeanUIModel> void addBooleanColumnToModel(TableColumnModel model,
                                                                                ColumnIdentifier<R> identifier,
                                                                                JTable table) {

        addColumnToModel(model,
                         table.getDefaultEditor(Boolean.class),
                         table.getDefaultRenderer(Boolean.class),
                         identifier,
                         null);
    }

    protected <R extends AbstractTuttiBeanUIModel, B> void addComboDataColumnToModel(TableColumnModel model,
                                                                                     ColumnIdentifier<R> identifier,
                                                                                     Decorator<B> decorator,
                                                                                     List<B> data) {
        JComboBox comboBox = new JComboBox();
        comboBox.setRenderer(newListCellRender(decorator));

        List<B> dataToList = Lists.newArrayList(data);

        // add a null value at first position
        if (!dataToList.isEmpty() && dataToList.get(0) != null) {
            dataToList.add(0, null);
        }
        SwingUtil.fillComboBox(comboBox, dataToList, null);

        ObjectToStringConverter converter = BeanUIUtil.newDecoratedObjectToStringConverter(decorator);
        BeanUIUtil.decorate(comboBox, converter);
        ComboBoxCellEditor editor = new ComboBoxCellEditor(comboBox);

        addColumnToModel(model,
                         editor,
                         newTableCellRender(decorator),
                         identifier,
                         null);
    }

    protected <O> TableCellRenderer newTableCellRender(Class<O> type) {

        return newTableCellRender(type, null);
    }

    protected <O> TableCellRenderer newTableCellRender(Class<O> type, String name) {

        Decorator<O> decorator = getDecorator(type, name);

        TableCellRenderer result = newTableCellRender(decorator);
        return result;
    }

    protected <O> TableCellRenderer newTableCellRender(Decorator<O> decorator) {

        Preconditions.checkNotNull(decorator);

        DecoratorTableCellRenderer result = new DecoratorTableCellRenderer(decorator, true);
        return result;
    }

    /**
     * Hook to prepare popup just before showing it.
     * <p/>
     * The right place to update actions accessibility; a quite better design
     * than trying to update each time something change in the table...
     *
     * @param rowIndex    selected row index (or lowest selected one)
     * @param columnIndex selected column index
     * @since 2.6
     */
    protected void beforeOpenPopup(int rowIndex, int columnIndex) {

    }

}
