package fr.ifremer.tutti.ui.swing.content.operation.catches.benthos;

/*
 * #%L
 * Tutti :: UI
 * $Id: BenthosBatchRowModel.java 1203 2013-09-23 09:39:50Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-2.5.1/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/catches/benthos/BenthosBatchRowModel.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import fr.ifremer.tutti.persistence.entities.TuttiBeanFactory;
import fr.ifremer.tutti.persistence.entities.data.AttachementObjectTypeEnum;
import fr.ifremer.tutti.persistence.entities.data.Attachment;
import fr.ifremer.tutti.persistence.entities.data.BenthosBatch;
import fr.ifremer.tutti.persistence.entities.data.BenthosBatchFrequency;
import fr.ifremer.tutti.persistence.entities.data.FishingOperation;
import fr.ifremer.tutti.persistence.entities.data.SampleCategory;
import fr.ifremer.tutti.persistence.entities.data.SampleCategoryModel;
import fr.ifremer.tutti.persistence.entities.data.SampleCategoryModelEntry;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.ui.swing.content.operation.catches.SampleCategoryAble;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.frequency.BenthosFrequencyRowModel;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiBeanUIModel;
import fr.ifremer.tutti.ui.swing.util.TuttiComputedOrNotData;
import fr.ifremer.tutti.ui.swing.util.TuttiUIUtil;
import fr.ifremer.tutti.ui.swing.util.WeightUnit;
import fr.ifremer.tutti.ui.swing.util.attachment.AttachmentModelAware;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

/**
 * Represents a species batch (i.e a row in the batch table).
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.2
 */
public class BenthosBatchRowModel extends AbstractTuttiBeanUIModel<BenthosBatch, BenthosBatchRowModel> implements BenthosBatch, AttachmentModelAware, SampleCategoryAble<BenthosBatchRowModel> {

    private static final long serialVersionUID = 1L;

    public static final String PROPERTY_SPECIES_TO_CONFIRM = "speciesToConfirm";

    public static final String PROPERTY_SPECIES = "species";

    public static final String PROPERTY_SAMPLE_CATEGORY = "sampleCategory";

    public static final String PROPERTY_SAMPLE_CATEGORY_VALUE = "sampleCategoryValue";

    public static final String PROPERTY_SAMPLE_CATEGORY_WEIGHT = "sampleCategoryWeight";

    public static final String PROPERTY_SAMPLE_CATEGORY_COMPUTED_WEIGHT = "sampleCategoryComputedWeight";

    public static final String PROPERTY_WEIGHT = "weight";

    public static final String PROPERTY_NUMBER = "number";

    public static final String PROPERTY_FREQUENCY = "frequency";

    public static final String PROPERTY_COMPUTED_NUMBER = "computedOrNotNumber";

    public static final String PROPERTY_COMPUTED_WEIGHT = "computedOrNotWeight";

    public static final String PROPERTY_PARENT_BATCH = "parentBatch";

    public static final String PROPERTY_CHILD_BATCH = "childBatch";

    public static final String PROPERTY_BATCH_LEAF = "batchLeaf";

    public static final String PROPERTY_BATCH_ROOT = "batchRoot";

    /**
     * Species.
     *
     * @since 0.3
     */
    protected Species species;

    /**
     * All categories(can not be null).
     *
     * @since 2.4
     */
    protected final SampleCategory<?>[] categories;

    /**
     * Is the species need to be confirmed?.
     *
     * @since 0.2
     */
    protected boolean speciesToConfirm;

    /**
     * Observed weight.
     *
     * @since 0.2
     */
    protected TuttiComputedOrNotData<Float> computedOrNotWeight =
            new TuttiComputedOrNotData<Float>();

    /**
     * Total computed number (from frequencies).
     *
     * @since 0.2
     */
    protected TuttiComputedOrNotData<Integer> computedOrNotNumber =
            new TuttiComputedOrNotData<Integer>();

    /**
     * Comment.
     *
     * @since 0.2
     */
    protected String comment;

    /**
     * Attachments (should never be null).
     *
     * @since 0.2
     */
    protected final List<Attachment> attachment = Lists.newArrayList();

    /**
     * List of frequencies observed for this batch.
     *
     * @since 0.2
     */
    protected List<BenthosFrequencyRowModel> frequency = Lists.newArrayList();

    /**
     * Parent of this batch (can be null if batch is root).
     *
     * @see #isBatchRoot()
     * @since 0.3
     */
    protected BenthosBatchRowModel parentBatch;

    /**
     * List of child batches (can be null or empty if batch is a leaf).
     *
     * @see #isBatchLeaf()
     * @since 0.3
     */
    protected List<BenthosBatchRowModel> childBatch;

    /**
     * Model of sample categories.
     *
     * @since 2.4
     */
    protected final SampleCategoryModel sampleCategoryModel;

    /**
     * Weight unit.
     *
     * @since 2.5
     */
    protected final WeightUnit weightUnit;

    protected static final Binder<BenthosBatch, BenthosBatchRowModel> fromBeanBinder =
            BinderFactory.newBinder(BenthosBatch.class,
                                    BenthosBatchRowModel.class);

    protected static final Binder<BenthosBatchRowModel, BenthosBatch> toBeanBinder =
            BinderFactory.newBinder(BenthosBatchRowModel.class,
                                    BenthosBatch.class);

    public BenthosBatchRowModel(WeightUnit weightUnit,
                                SampleCategoryModel sampleCategoryModel) {
        super(BenthosBatch.class, fromBeanBinder, toBeanBinder);
        this.weightUnit = weightUnit;
        this.sampleCategoryModel = sampleCategoryModel;

        categories = new SampleCategory[sampleCategoryModel.getNbSampling()];

        for (int i = 0; i < categories.length; i++) {
            SampleCategoryModelEntry entry = sampleCategoryModel.getCategoryByIndex(i);
            SampleCategory<?> category = SampleCategory.newSample(entry);
            categories[i] = category;
        }

        computedOrNotWeight.addPropagateListener(PROPERTY_WEIGHT, this);
        computedOrNotWeight.addPropagateListener(PROPERTY_COMPUTED_WEIGHT, this);
        computedOrNotNumber.addPropagateListener(PROPERTY_NUMBER, this);
        computedOrNotNumber.addPropagateListener(PROPERTY_COMPUTED_NUMBER, this);
    }


    public BenthosBatchRowModel(WeightUnit weightUnit,
                                SampleCategoryModel sampleCategoryModel,
                                BenthosBatch entity,
                                List<BenthosBatchFrequency> frequencies) {
        this(weightUnit, sampleCategoryModel);

        fromEntity(entity);

        List<BenthosFrequencyRowModel> frequencyRows =
                BenthosFrequencyRowModel.fromEntity(weightUnit,
                                                    frequencies);
        frequency.addAll(frequencyRows);
        Collections.sort(frequency);
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiBeanUIModel                                           --//
    //------------------------------------------------------------------------//

    @Override
    protected BenthosBatch newEntity() {
        return TuttiBeanFactory.newBenthosBatch();
    }

    @Override
    public void fromEntity(BenthosBatch entity) {
        super.fromEntity(entity);

        // convert weight
        setWeight(weightUnit.fromEntity(getWeight()));
    }

    @Override
    public BenthosBatch toEntity() {
        BenthosBatch result = super.toEntity();

        // convert weight
        result.setWeight(weightUnit.toEntity(getWeight()));

        SampleCategory<?> sampleCategory = getFinestCategory();
        Preconditions.checkNotNull(sampleCategory);
        Preconditions.checkNotNull(sampleCategory.getCategoryId());
        Preconditions.checkNotNull(sampleCategory.getCategoryValue());

        // apply sample category
        result.setSampleCategoryId(sampleCategory.getCategoryId());
        result.setSampleCategoryValue(sampleCategory.getCategoryValue());

        // convert sample category weight
        Float categoryWeight = sampleCategory.getCategoryWeight();
        result.setSampleCategoryWeight(weightUnit.toEntity(categoryWeight));
        return result;
    }

    //------------------------------------------------------------------------//
    //-- SampleCategoryAble                                                 --//
    //------------------------------------------------------------------------//

    @Override
    public Integer getCategoryIndex(Integer id) {
        Integer result = null;
        for (SampleCategory<?> category : categories) {
            if (category != null && id.equals(category.getCategoryId())) {
                result = category.getCategoryDef().getOrder();
                break;
            }
        }
        return result;
    }

    @Override
    public void setSampleCategory(SampleCategory sampleCategory) {
        int index = sampleCategory.getCategoryDef().getOrder();
        SampleCategory<?> oldCategory = categories[index];
        Object oldValue = oldCategory.getCategoryValue();
        Object oldWeight = oldCategory.getCategoryWeight();
        Object oldComputedWeight = oldCategory.getComputedWeight();
        categories[index] = sampleCategory;
        //FIXME (indexed)
        fireIndexedPropertyChange(PROPERTY_SAMPLE_CATEGORY, index, oldCategory, sampleCategory);
        fireIndexedPropertyChange(PROPERTY_SAMPLE_CATEGORY_VALUE, index, oldValue, sampleCategory.getCategoryValue());
        fireIndexedPropertyChange(PROPERTY_SAMPLE_CATEGORY_WEIGHT, index, oldWeight, sampleCategory.getCategoryWeight());
        fireIndexedPropertyChange(PROPERTY_SAMPLE_CATEGORY_COMPUTED_WEIGHT, index, oldComputedWeight, sampleCategory.getComputedWeight());
    }

    @Override
    public SampleCategory<?> getFirstSampleCategory() {
        return categories[0];
    }

    @Override
    public SampleCategory getFinestCategory() {
        SampleCategory result = null;
        for (int i = categories.length - 1; i > -1; i--) {
            SampleCategory<?> category = categories[i];
            if (category != null && category.isValid()) {
                result = category;
                break;
            }
        }
        return result;
    }

    @Override
    public SampleCategory<?> getSampleCategoryById(Integer sampleCategoryId) {
        Integer index = getCategoryIndex(sampleCategoryId);
        SampleCategory<?> result = index == null ? null : categories[index];
        return result;
    }

    @Override
    public void setSampleCategoryWeight(Integer sampleCategoryId, Object value) {
        SampleCategory<?> sampleCategory =
                getSampleCategoryById(sampleCategoryId);
        TuttiUIUtil.setProperty(sampleCategory,
                                SampleCategory.PROPERTY_CATEGORY_WEIGHT, value);
        firePropertyChange(PROPERTY_SAMPLE_CATEGORY_WEIGHT, null, sampleCategory);
    }

    @Override
    public BenthosBatchRowModel getFirstAncestor(SampleCategory<?> entrySampleCategory) {
        BenthosBatchRowModel result = this;
        if (getParentBatch() != null) {
            BenthosBatchRowModel parentBatch = getParentBatch();
            SampleCategory<?> parentSampleCategory = parentBatch.getSampleCategoryById(entrySampleCategory.getCategoryId());
            if (ObjectUtils.equals(entrySampleCategory, parentSampleCategory)) {

                result = parentBatch.getFirstAncestor(entrySampleCategory);
            }
        }
        return result;
    }

    @Override
    public Iterator<SampleCategory<?>> iterator() {
        return Arrays.asList(categories).iterator();
    }

    //------------------------------------------------------------------------//
    //-- Species category                                                   --//
    //------------------------------------------------------------------------//

    @Override
    public Species getSpecies() {
        return species;
    }

    @Override
    public void setSpecies(Species species) {
        Object oldCategory = getSpecies();
        this.species = species;
        firePropertyChange(PROPERTY_SPECIES, oldCategory, species);
    }

    //------------------------------------------------------------------------//
    //-- Navigation properties                                              --//
    //------------------------------------------------------------------------//

    @Override
    public BenthosBatchRowModel getParentBatch() {
        return parentBatch;
    }

    public void setParentBatch(BenthosBatchRowModel parentBatch) {
        Object oldValue = getParentBatch();
        this.parentBatch = parentBatch;
        firePropertyChange(PROPERTY_PARENT_BATCH, oldValue, parentBatch);
        firePropertyChange(PROPERTY_BATCH_ROOT, null, isBatchRoot());
    }

    public List<BenthosBatchRowModel> getChildBatch() {
        return childBatch;
    }

    public void setChildBatch(List<BenthosBatchRowModel> childBatch) {
        this.childBatch = childBatch;
        // force to propagate child changes
        firePropertyChange(PROPERTY_CHILD_BATCH, null, childBatch);
        firePropertyChange(PROPERTY_BATCH_LEAF, null, isBatchLeaf());
    }

    public boolean isBatchLeaf() {
        return CollectionUtils.isEmpty(childBatch);
    }

    public boolean isBatchRoot() {
        return parentBatch == null;
    }

    //------------------------------------------------------------------------//
    //-- CommentAware                                                        --//
    //------------------------------------------------------------------------//

    @Override
    public String getComment() {
        return comment;
    }

    @Override
    public void setComment(String comment) {
        Object oldValue = getComment();
        this.comment = comment;
        firePropertyChange(PROPERTY_COMMENT, oldValue, comment);
    }

    //------------------------------------------------------------------------//
    //-- AttachmentModelAware                                               --//
    //------------------------------------------------------------------------//

    @Override
    public AttachementObjectTypeEnum getObjectType() {
        return AttachementObjectTypeEnum.BATCH;
    }

    @Override
    public Integer getObjectId() {
        return getIdAsInt();
    }

    @Override
    public List<Attachment> getAttachment() {
        return attachment;
    }

    @Override
    public void addAllAttachment(Collection<Attachment> attachments) {
        this.attachment.addAll(attachments);
        firePropertyChange(PROPERTY_ATTACHMENT, null, getAttachment());
    }

    @Override
    public void addAttachment(Attachment attachment) {
        this.attachment.add(attachment);
        firePropertyChange(PROPERTY_ATTACHMENT, null, getAttachment());
    }

    @Override
    public void removeAllAttachment(Collection<Attachment> attachments) {
        this.attachment.removeAll(attachments);
        firePropertyChange(PROPERTY_ATTACHMENT, null, getAttachment());
    }

    @Override
    public void removeAttachment(Attachment attachment) {
        this.attachment.remove(attachment);
        firePropertyChange(PROPERTY_ATTACHMENT, null, getAttachment());
    }

    //------------------------------------------------------------------------//
    //-- Other properties                                                   --//
    //------------------------------------------------------------------------//

    @Override
    public boolean isSpeciesToConfirm() {
        return speciesToConfirm;
    }

    @Override
    public void setSpeciesToConfirm(boolean speciesToConfirm) {
        Object oldValue = isSpeciesToConfirm();
        this.speciesToConfirm = speciesToConfirm;
        firePropertyChange(PROPERTY_SPECIES_TO_CONFIRM, oldValue, speciesToConfirm);
    }

    @Override
    public Float getWeight() {
        return computedOrNotWeight.getData();
    }

    @Override
    public void setWeight(Float weight) {
        this.computedOrNotWeight.setData(weight);
    }

    @Override
    public Integer getNumber() {
        return computedOrNotNumber.getData();
    }

    @Override
    public void setNumber(Integer number) {
        computedOrNotNumber.setData(number);
    }

    public List<BenthosFrequencyRowModel> getFrequency() {
        return frequency;
    }

    public void setFrequency(List<BenthosFrequencyRowModel> frequency) {
        this.frequency = frequency;
        // force to propagate frequencies changes
        firePropertyChange(PROPERTY_FREQUENCY, null, frequency);
    }

    @Override
    public Integer getComputedNumber() {
        return computedOrNotNumber.getComputedData();
    }

    @Override
    public void setComputedNumber(Integer computedNumber) {
        computedOrNotNumber.setComputedData(computedNumber);
    }

    @Override
    public Float getComputedWeight() {
        return computedOrNotWeight.getComputedData();
    }

    @Override
    public void setComputedWeight(Float computedWeight) {
        computedOrNotWeight.setComputedData(computedWeight);
    }

    public TuttiComputedOrNotData<Integer> getComputedOrNotNumber() {
        return computedOrNotNumber;
    }

    public void setComputedOrNotNumber(TuttiComputedOrNotData<Integer> computedOrNotNumber) {
        this.computedOrNotNumber = computedOrNotNumber;
    }

    public TuttiComputedOrNotData<Float> getComputedOrNotWeight() {
        return computedOrNotWeight;
    }

    public void setComputedOrNotWeight(TuttiComputedOrNotData<Float> computedOrNotWeight) {
        this.computedOrNotWeight = computedOrNotWeight;
    }

    @Override
    public Integer getSampleCategoryId() {
        return null;
    }

    @Override
    public void setSampleCategoryId(Integer sampleCategoryId) {
    }

    @Override
    public Serializable getSampleCategoryValue() {
        return null;
    }

    @Override
    public void setSampleCategoryValue(Serializable sampleCategoryValue) {
    }

    @Override
    public Float getSampleCategoryWeight() {
        return null;
    }

    @Override
    public void setSampleCategoryWeight(Float sampleCategoryWeight) {
    }

    @Override
    public Float getSampleCategoryComputedWeight() {
        return null;
    }

    @Override
    public void setSampleCategoryComputedWeight(Float sampleCategoryComputedWeight) {
    }

    @Override
    public BenthosBatch getChildBatchs(int index) {
        return null;
    }

    @Override
    public boolean isChildBatchsEmpty() {
        return false;
    }

    @Override
    public int sizeChildBatchs() {
        return 0;
    }

    @Override
    public void addChildBatchs(BenthosBatch childBatchs) {
    }

    @Override
    public void addAllChildBatchs(Collection<BenthosBatch> childBatchs) {
    }

    @Override
    public boolean removeChildBatchs(BenthosBatch childBatchs) {
        return false;
    }

    @Override
    public boolean removeAllChildBatchs(Collection<BenthosBatch> childBatchs) {
        return false;
    }

    @Override
    public boolean containsChildBatchs(BenthosBatch childBatchs) {
        return false;
    }

    @Override
    public boolean containsAllChildBatchs(Collection<BenthosBatch> childBatchs) {
        return false;
    }

    @Override
    public List<BenthosBatch> getChildBatchs() {
        return null;
    }

    @Override
    public void setChildBatchs(List<BenthosBatch> childBatchs) {
    }

    @Override
    public void setParentBatch(BenthosBatch parentBatch) {
    }

    @Override
    public FishingOperation getFishingOperation() {
        return null;
    }

    @Override
    public void setFishingOperation(FishingOperation fishingOperation) {
    }

    public void collectShell(Set<BenthosBatchRowModel> collectedRows) {

        if (!isBatchLeaf()) {

            for (BenthosBatchRowModel batchChild : getChildBatch()) {
                collectedRows.add(batchChild);
                batchChild.collectShell(collectedRows);
            }
        }
    }

    @Override
    public Integer getRankOrder() {
        return null;
    }

    @Override
    public void setRankOrder(Integer rankOrder) {
    }
}
