package fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.frequency;

/*
 * #%L
 * Tutti :: UI
 * $Id: BenthosFrequencyTableModel.java 666 2013-03-23 18:20:58Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-2.1/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/catches/benthos/frequency/BenthosFrequencyTableModel.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import fr.ifremer.tutti.ui.swing.util.table.AbstractTuttiTableModel;
import fr.ifremer.tutti.ui.swing.util.table.ColumnIdentifier;
import org.jdesktop.swingx.table.TableColumnModelExt;

import java.util.List;
import java.util.Map;

import static org.nuiton.i18n.I18n.n_;

/**
 * Model of the species frequency table.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.2
 */
public class BenthosFrequencyTableModel extends AbstractTuttiTableModel<BenthosFrequencyRowModel> {

    private static final long serialVersionUID = 1L;

    public static final ColumnIdentifier<BenthosFrequencyRowModel> LENGTH_STEP = ColumnIdentifier.newId(
            BenthosFrequencyRowModel.PROPERTY_LENGTH_STEP,
            n_("tutti.editSpeciesFrequencies.table.header.lengthStep"),
            n_("tutti.editSpeciesFrequencies.table.header.lengthStep"));

    public static final ColumnIdentifier<BenthosFrequencyRowModel> NUMBER = ColumnIdentifier.newId(
            BenthosFrequencyRowModel.PROPERTY_NUMBER,
            n_("tutti.editSpeciesFrequencies.table.header.number"),
            n_("tutti.editSpeciesFrequencies.table.header.number"));

    public static final ColumnIdentifier<BenthosFrequencyRowModel> WEIGHT = ColumnIdentifier.newId(
            BenthosFrequencyRowModel.PROPERTY_WEIGHT,
            n_("tutti.editSpeciesFrequencies.table.header.weight"),
            n_("tutti.editSpeciesFrequencies.table.header.weight"));

    private final BenthosFrequencyUIModel uiModel;

    private final Map<Float, BenthosFrequencyRowModel> rowCache;

    public BenthosFrequencyTableModel(TableColumnModelExt columnModel,
                                      BenthosFrequencyUIModel uiModel) {
        super(columnModel, true, true);
        this.uiModel = uiModel;
        this.rowCache = Maps.newTreeMap();
        setNoneEditableCols();
    }

    @Override
    public BenthosFrequencyRowModel createNewRow() {
        Float defaultStep = null;

        int rowCount = getRowCount();
        if (rowCount > 0) {

            BenthosFrequencyRowModel rowModel = getEntry(rowCount - 1);
            Float lengthStep = rowModel.getLengthStep();
            if (lengthStep != null) {
                defaultStep = uiModel.getLengthStep(
                        lengthStep + uiModel.getStep());
            }
        }
        BenthosFrequencyRowModel result = new BenthosFrequencyRowModel();
        result.setLengthStep(defaultStep);
        result.setValid(false);
        return result;
    }

    @Override
    public void setValueAt(Object aValue,
                           int rowIndex,
                           int columnIndex,
                           ColumnIdentifier<BenthosFrequencyRowModel> propertyName,
                           BenthosFrequencyRowModel entry) {
        super.setValueAt(aValue, rowIndex, columnIndex, propertyName, entry);
        // TODO Rebuild the computedWeight if possible...
    }

    public Map<Float, BenthosFrequencyRowModel> getRowCache() {
        return rowCache;
    }

    @Override
    protected void onRowsChanged(List<BenthosFrequencyRowModel> data) {

        // rebuild row cache
        rowCache.clear();

        for (BenthosFrequencyRowModel row : data) {
            Float lengthStep = row.getLengthStep();
            if (lengthStep != null) {
                rowCache.put(lengthStep, row);
            }
        }
    }

    @Override
    protected void onRowAdded(int rowIndex, BenthosFrequencyRowModel newValue) {

        Preconditions.checkNotNull(newValue, "can't add a null row");

        newValue.setLengthStepCaracteristic(uiModel.getLengthStepCaracteristic());

        // add new row to cache
        Float lengthStep = newValue.getLengthStep();

        Preconditions.checkNotNull(lengthStep,
                                   "can't add a null lengthStep row");

        float roundLenghtValue = uiModel.getLengthStep(lengthStep);

        if (!rowCache.containsKey(roundLenghtValue)) {

            rowCache.put(roundLenghtValue, newValue);
        }
    }
}