package fr.ifremer.tutti.ui.swing;

/*
 * #%L
 * Tutti :: UI
 * $Id: TuttiApplicationUpdaterCallBack.java 907 2013-05-01 14:27:16Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-2.1/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/TuttiApplicationUpdaterCallBack.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import fr.ifremer.tutti.LabelAware;
import fr.ifremer.tutti.TuttiIOUtil;
import fr.ifremer.tutti.TuttiTechnicalException;
import fr.ifremer.tutti.persistence.ProgressionModel;
import fr.ifremer.tutti.persistence.config.TuttiPersistenceConfig;
import fr.ifremer.tutti.persistence.service.synchro.ReferentialSynchronizeResult;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.service.referential.TuttiReferentialSynchronizeService;
import fr.ifremer.tutti.ui.swing.util.action.AbstractTuttiAction;
import fr.ifremer.tutti.ui.swing.util.action.TuttiActionException;
import fr.ifremer.tutti.ui.swing.util.action.TuttiActionUI;
import org.apache.commons.io.FileUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.updater.ApplicationInfo;
import org.nuiton.util.updater.ApplicationUpdater;
import org.nuiton.util.updater.ApplicationUpdaterCallback;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Map;

import static org.nuiton.i18n.I18n._;
import static org.nuiton.i18n.I18n.n_;

/**
 * CallBack to update jre, application, i18n or db.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.0
 */
public class TuttiApplicationUpdaterCallBack implements ApplicationUpdaterCallback {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(TuttiApplicationUpdaterCallBack.class);

    public enum UpdateType implements LabelAware {
        JRE(n_("tutti.update.jre")),
        TUTTI(n_("tutti.update.tutti")),
        I18N(n_("tutti.update.i18n")),
        HELP(n_("tutti.update.help")),
        DB(n_("tutti.update.db"));

        private final String i18nKey;

        private UpdateType(String i18nKey) {
            this.i18nKey = i18nKey;
        }

        @Override
        public String getLabel() {
            return _(i18nKey);
        }
    }

    protected final TuttiUIContext context;

    protected List<UpdateType> types;

    protected ProgressionModel progressionModel;

    protected boolean applicationUpdated;

    protected boolean dbInstalled;

    protected boolean dbUpdated;

    protected final AbstractTuttiAction action;

    public TuttiApplicationUpdaterCallBack(AbstractTuttiAction action,
                                           ProgressionModel progressionModel) {
        this.action = action;
        this.context = action.getContext();
        this.progressionModel = progressionModel;
    }

    public void setTypes(UpdateType... types) {
        this.types = Lists.newArrayList(types);
    }

    public boolean isApplicationUpdated() {
        return applicationUpdated;
    }

    public boolean isDbUpdated() {
        return dbUpdated;
    }

    public boolean isDbInstalled() {
        return dbInstalled;
    }

    @Override
    public Map<String, ApplicationInfo> updateToDo(Map<String, ApplicationInfo> appToUpdate) {
        Map<String, ApplicationInfo> result = Maps.newHashMap();

        for (UpdateType type : types) {
            ApplicationInfo info = getInfo(type, appToUpdate);
            if (info != null) {
                result.put(info.name, info);
                if (UpdateType.DB.equals(type)) {
                    // add auth
                    info.setAuthentication(
                            "t" + "u" + "t" + "t" + "i" + "d" + "b",
                            new char[]{'3', 'R', 'A', 'C', 'e', 'v', '2', 'N', 'w', '8', 'u', 'E', 'D', 'E', 'v', 's', 'y', '0', 'v', 'J'}
                    );
                }
            }
        }
        if (types.contains(UpdateType.DB)) {

            ApplicationInfo info = getInfo(UpdateType.DB, result);

            if (info == null) {
                dbInstalled = false;
                dbUpdated = false;
            } else {
                if (context.isDbExist()) {

                    // when db exists always an update
                    dbUpdated = true;
                } else {

                    // when no db, then always install
                    dbInstalled = true;
                }
            }
        }
        return result;
    }

    @Override
    public void startUpdate(ApplicationInfo info) {
        if (UpdateType.DB.name().toLowerCase().equals(info.name)) {

            if (dbInstalled) {

                progressionModel.setMessage(_("tutti.applicationUpdater.startUpdate.db.installation", info.newVersion));
            } else if (dbUpdated) {
                progressionModel.setMessage(_("tutti.applicationUpdater.startUpdate.db.update", info.newVersion));
            }
        }
        if (UpdateType.JRE.name().toLowerCase().equals(info.name)) {
            progressionModel.setMessage(_("tutti.applicationUpdater.startUpdate.jre", info.newVersion));
        }

        if (UpdateType.TUTTI.name().toLowerCase().equals(info.name)) {
            progressionModel.setMessage(_("tutti.applicationUpdater.startUpdate.tutti", info.newVersion));
        }

        if (UpdateType.I18N.name().toLowerCase().equals(info.name)) {
            progressionModel.setMessage(_("tutti.applicationUpdater.startUpdate.i18n", info.newVersion));
        }
        if (UpdateType.HELP.name().toLowerCase().equals(info.name)) {
            progressionModel.setMessage(_("tutti.applicationUpdater.startUpdate.help", info.newVersion));
        }
    }

    @Override
    public void updateDone(Map<String, ApplicationInfo> appToUpdate,
                           Map<String, Exception> appUpdateError) {

        boolean updateJRE = updateDoneJre(appToUpdate, appUpdateError);
        boolean updateTutti = updateDoneTutti(appToUpdate, appUpdateError);
        boolean updateI18n = updateDoneI18n(appToUpdate, appUpdateError);
        boolean updateHelp = updateDoneHelp(appToUpdate, appUpdateError);

        boolean doRestart = updateJRE || updateTutti || updateI18n || updateHelp;

        updateDoneDb(appToUpdate, appUpdateError);

        if (doRestart) {

            applicationUpdated = true;
        }
    }

    @Override
    public void aborted(String propertiesURL, Exception eee) {
        if (log.isErrorEnabled()) {
            log.error("Could not update from " + propertiesURL, eee);
        }
        throw TuttiActionException.propagateError(action, eee);
    }

    protected boolean updateDoneJre(Map<String, ApplicationInfo> appToUpdate,
                                    Map<String, Exception> appUpdateError) {
        boolean doRestart = false;
        Exception error = getError(UpdateType.JRE, appUpdateError);
        if (error != null) {

            // something bad while updating jre
            if (log.isErrorEnabled()) {
                log.error("Could not update jre", error);
            }
        } else {
            ApplicationInfo info = getInfo(UpdateType.JRE, appToUpdate);
            if (info != null) {

                if (log.isInfoEnabled()) {
                    log.info(String.format(
                            "A jre update was downloaded (oldVersion: %s, newVersion: %s), will restart application to use it",
                            info.oldVersion, info.newVersion));
                }
                doRestart = true;
            }
        }
        return doRestart;
    }

    protected boolean updateDoneTutti(Map<String, ApplicationInfo> appToUpdate,
                                      Map<String, Exception> appUpdateError) {
        boolean doRestart = false;
        Exception error = getError(UpdateType.TUTTI, appUpdateError);
        if (error != null) {

            // something bad while updating application
            if (log.isErrorEnabled()) {
                log.error("Could not update tutti", error);
            }
        } else {
            ApplicationInfo info = getInfo(UpdateType.TUTTI, appToUpdate);
            if (info != null) {

                if (log.isInfoEnabled()) {
                    log.info(String.format(
                            "A tutti update was downloaded (oldVersion: %s, newVersion: %s), will restart application to use it",
                            info.oldVersion, info.newVersion));
                }
                doRestart = true;

                TuttiPersistenceConfig persistenceConfig =
                        context.getConfig().getServiceConfig().getPersistenceConfig();

                // must remove the enumeration file at exit
                File enumerationPath = persistenceConfig.getDbEnumerationPath();
                TuttiIOUtil.forceDeleteOnExit(
                        enumerationPath,
                        _("tutti.applicationUpdater.updateDone.deleteDirectory.enum.error", enumerationPath)
                );

                // must remove the db conf file at exit
                File dbConfPath = persistenceConfig.getDbConfigurationPath();
                TuttiIOUtil.forceDeleteOnExit(
                        dbConfPath,
                        _("tutti.applicationUpdater.updateDone.deleteDirectory.dbConf.error", dbConfPath)
                );

                //FIXME-check this is necessary: i18n is no more generated ?
                // must also remove i18n directory
                File i18nDirectory = context.getConfig().getI18nDirectory();
                TuttiIOUtil.forceDeleteOnExit(
                        i18nDirectory,
                        _("tutti.applicationUpdater.updateDone.deleteDirectory.i18n.error", i18nDirectory)
                );
            }
        }
        return doRestart;
    }

    protected boolean updateDoneI18n(Map<String, ApplicationInfo> appToUpdate,
                                     Map<String, Exception> appUpdateError) {
        boolean doRestart = false;
        Exception error = getError(UpdateType.I18N, appUpdateError);
        if (error != null) {

            // something bad while updating i18n
            if (log.isErrorEnabled()) {
                log.error("Could not update i18n", error);
            }
        } else {
            ApplicationInfo info = getInfo(UpdateType.I18N, appToUpdate);
            if (info != null) {

                if (log.isInfoEnabled()) {
                    log.info(String.format(
                            "A i18n update was downloaded (oldVersion: %s, newVersion: %s), will restart application to use it",
                            info.oldVersion, info.newVersion));
                }
                doRestart = true;
            }
        }
        return doRestart;
    }

    protected boolean updateDoneHelp(Map<String, ApplicationInfo> appToUpdate,
                                     Map<String, Exception> appUpdateError) {
        boolean doRestart = false;
        Exception error = getError(UpdateType.HELP, appUpdateError);
        if (error != null) {

            // something bad while updating i18n
            if (log.isErrorEnabled()) {
                log.error("Could not update help", error);
            }
        } else {
            ApplicationInfo info = getInfo(UpdateType.HELP, appToUpdate);
            if (info != null) {

                if (log.isInfoEnabled()) {
                    log.info(String.format(
                            "A help update was downloaded (oldVersion: %s, newVersion: %s), will restart application to use it",
                            info.oldVersion, info.newVersion));
                }
                doRestart = true;
            }
        }
        return doRestart;
    }

    protected void updateDoneDb(Map<String, ApplicationInfo> appToUpdate,
                                Map<String, Exception> appUpdateError) {
        Exception error = getError(UpdateType.DB, appUpdateError);
        if (error != null) {

            // something bad while updating db
            if (log.isErrorEnabled()) {
                log.error("Could not update db", error);
            }
            throw new TuttiActionException(action, error);
        } else {
            ApplicationInfo info = getInfo(UpdateType.DB, appToUpdate);
            if (info != null) {

                if (log.isInfoEnabled()) {
                    log.info(String.format(
                            "A db update was downloaded (oldVersion: %s, newVersion: %s), will process it.",
                            info.oldVersion, info.newVersion));
                }

                // before install or update, regenerate db configuration files

                TuttiPersistenceConfig persistenceConfig = regenerateDbConf();

                persistenceConfig.generateExternalDbFiles(true);

                if (dbInstalled) {

                    // first database, just copy it to correct directory

                    prepareFirstDatabase(info);
                } else if (dbUpdated) {

                    // launch a referential synchronize operation
                    synchronizetDatabase(info);
                }
            }
        }
    }

    protected TuttiPersistenceConfig regenerateDbConf() {

        TuttiPersistenceConfig persistenceConfig =
                context.getConfig().getServiceConfig().getPersistenceConfig();

        context.showInformationMessage(
                "Regénérer les fichiers de configuration de la base");
        return persistenceConfig;
    }

    protected ApplicationInfo getInfo(UpdateType type,
                                      Map<String, ApplicationInfo> appToUpdate) {
        return appToUpdate.get(type.name().toLowerCase());
    }

    protected Exception getError(UpdateType type,
                                 Map<String, Exception> appUpdateError) {
        return appUpdateError.get(type.name().toLowerCase());
    }

    protected File getDbDirectory(ApplicationInfo info) {
        File[] sources = info.destDir.listFiles();
        Preconditions.checkState(
                sources != null && sources.length == 1,
                "Downloaded db should contains one directory at " + info.destDir);
        File result = sources[0];
        return result;
    }

    protected void prepareFirstDatabase(ApplicationInfo info) {
        if (log.isInfoEnabled()) {
            log.info("First time database was downloaded at version: " + info.newVersion);
        }
        File source = getDbDirectory(info);
        File target = context.getConfig().getServiceConfig().getPersistenceConfig().getDbDirectory();
        if (log.isInfoEnabled()) {
            log.info("Copy from " + source + " to " + target);
        }
        try {
            FileUtils.copyDirectory(source, target);
        } catch (IOException e) {
            throw new TuttiTechnicalException(_("tutti.applicationUpdater.prepareFirstDB.copyDirectory.error", source, target), e);
        }
        try {
            FileUtils.deleteDirectory(source.getParentFile());
        } catch (IOException e) {
            throw new TuttiTechnicalException(_("tutti.applicationUpdater.prepareFirstDB.deleteDirectory.error", target), e);
        }
    }

    protected void synchronizetDatabase(ApplicationInfo info) {
        if (log.isInfoEnabled()) {
            log.info(String.format("A database update was downloaded (oldVersion: %s, newVersion: %s), will launch a referential synchronize operation ", info.oldVersion, info.newVersion));
        }
        TuttiReferentialSynchronizeService service = context.getTuttiReferentialSynchronizeService();
        ReferentialSynchronizeResult result = new ReferentialSynchronizeResult();
        File dbDirectory = getDbDirectory(info);

        TuttiActionUI actionUI = context.getActionUI();
        actionUI.getModel().setProgressionModel(result.getProgressionModel());
        service.prepare(dbDirectory, result);

        if (!result.isSuccess()) {
            throw new TuttiTechnicalException(_("tutti.applicationUpdater.synchroDB.prepare.error"), result.getError());
        }

        service.synchronize(dbDirectory, result);

        if (!result.isSuccess()) {
            throw new TuttiTechnicalException(_("tutti.applicationUpdater.synchroDB.synchro.error"), result.getError());
        }

        // reset cache
        if (log.isInfoEnabled()) {
            log.info("Reset all caches.");
        }
        PersistenceService persistence = context.getPersistenceService();
        persistence.clearAllCaches();

        // replace the version.appup file content
        File target = context.getConfig().getServiceConfig().getPersistenceConfig().getDbDirectory();
        File versionFile = ApplicationUpdater.getVersionFile(target);
        if (log.isInfoEnabled()) {
            log.info("Replace content of file " + versionFile + " with " + info.newVersion);
        }
        try {
            ApplicationUpdater.storeVersionFile(target, info.newVersion);
        } catch (IOException e) {
            throw new TuttiTechnicalException(_("tutti.applicationUpdater.synchroDB.writeVersion.error", versionFile));
        }
    }
}
