package fr.ifremer.tutti.ui.swing.content.protocol;

/*
 * #%L
 * Tutti :: UI
 * $Id: EditProtocolUIHandler.java 778 2013-04-15 15:38:46Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-2.0/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/protocol/EditProtocolUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.protocol.SpeciesProtocol;
import fr.ifremer.tutti.persistence.entities.protocol.TuttiProtocol;
import fr.ifremer.tutti.persistence.entities.referential.Caracteristic;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.ui.swing.content.home.CloneProtocolAction;
import fr.ifremer.tutti.ui.swing.content.home.ImportProtocolAction;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiUIHandler;
import fr.ifremer.tutti.ui.swing.util.CloseableUI;
import fr.ifremer.tutti.ui.swing.util.TuttiUI;
import fr.ifremer.tutti.ui.swing.util.TuttiUIUtil;
import fr.ifremer.tutti.ui.swing.util.species.SelectSpeciesUI;
import fr.ifremer.tutti.ui.swing.util.species.SelectSpeciesUIModel;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.swing.editor.bean.BeanDoubleList;
import jaxx.runtime.swing.editor.bean.BeanDoubleListModel;
import jaxx.runtime.swing.editor.bean.BeanFilterableComboBox;
import jaxx.runtime.swing.editor.bean.BeanUIUtil;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.autocomplete.ComboBoxCellEditor;
import org.jdesktop.swingx.autocomplete.ObjectToStringConverter;
import org.jdesktop.swingx.decorator.ComponentAdapter;
import org.jdesktop.swingx.decorator.HighlightPredicate;
import org.jdesktop.swingx.decorator.Highlighter;
import org.jdesktop.swingx.table.DefaultTableColumnModelExt;
import org.nuiton.util.decorator.Decorator;

import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JTabbedPane;
import javax.swing.ListSelectionModel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.TableColumnModel;
import java.awt.Component;
import java.awt.Dimension;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import static org.nuiton.i18n.I18n._;


/**
 * @author tchemit <chemit@codelutin.com>
 * @since 0.3
 */
public class EditProtocolUIHandler extends AbstractTuttiUIHandler<EditProtocolUIModel, EditProtocolUI> implements CloseableUI {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(EditProtocolUIHandler.class);

    public static String getTitle(boolean exist) {

        String result;
        if (exist) {
            result = _("tutti.editProtocol.title.edit.protocol");
        } else {
            result = _("tutti.editProtocol.title.create.protocol");
        }
        return result;
    }

    protected SelectSpeciesUI dialog;

    /**
     * Persistence service.
     *
     * @since 0.2
     */
    protected final PersistenceService persistenceService;

    public EditProtocolUIHandler(TuttiUI parentUi, EditProtocolUI ui) {
        super(parentUi.getHandler().getContext(), ui);
        this.persistenceService = context.getPersistenceService();
    }

    public JXTable getSpeciesTable() {
        return ui.getSpeciesTable();
    }

    public JXTable getBenthosTable() {
        return ui.getBenthosTable();
    }

    public EditProtocolSpeciesTableModel getSpeciesTableModel() {
        return (EditProtocolSpeciesTableModel) getSpeciesTable().getModel();
    }

    public EditProtocolSpeciesTableModel getBenthosTableModel() {
        return (EditProtocolSpeciesTableModel) getBenthosTable().getModel();
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    public SwingValidator<EditProtocolUIModel> getValidator() {
        return ui.getValidator();
    }

    @Override
    public void beforeInitUI() {

        getDataContext().resetValidationDataContext();

        EditProtocolUIModel model = new EditProtocolUIModel();

        // load cache data

        List<Species> allSpecies = Lists.newArrayList(getDataContext().getSpecies());
        model.setAllSpecies(allSpecies);

        Multimap<String, Species> allSpeciesByTaxonId =
                TuttiEntities.splitByReferenceTaxonId(allSpecies);
        model.setAllSpeciesByTaxonId(allSpeciesByTaxonId);

        Map<String, Species> allReferentSpeciesByTaxonId = TuttiEntities.splitByTaxonId(
                getDataContext().getReferentSpecies());
        model.setAllReferentSpeciesByTaxonId(allReferentSpeciesByTaxonId);

        List<Caracteristic> caracteristics = Lists.newArrayList(getDataContext().getCaracteristics());
        model.setCaracteristics(caracteristics);

        Map<String, Caracteristic> allCaracteristic = TuttiEntities.splitById(caracteristics);
        model.setAllCaracteristic(allCaracteristic);

        // can't load directly model from database here, since we want to
        // fill only the model with rows (transformed from speciesProtocol)
        // As we still don't have the table model at this point, wait the
        // afterUI method to fill model

        listModelIsModify(model);
        ui.setContextValue(model);
    }

    @Override
    public void afterInitUI() {

        initUI(ui);

        EditProtocolUIModel model = getModel();

        TuttiProtocol protocol;

        if (context.isProtocolFilled()) {

            // load existing protocol

            protocol = getDataContext().getProtocol();

            model.fromBean(protocol);

        } else if ((protocol = ImportProtocolAction.IMPORT_PROTOCOL_ENTRY.getContextValue(ui)) != null) {

            // import protocol

            ImportProtocolAction.IMPORT_PROTOCOL_ENTRY.removeContextValue(ui);

            model.fromBean(protocol);
            model.setImported(true);
            ui.getSaveWarning().setText(_("tutti.editProtocol.warn.import"));

        } else if ((protocol = CloneProtocolAction.CLONE_PROTOCOL_ENTRY.getContextValue(ui)) != null) {

            // clone protocol

            CloneProtocolAction.CLONE_PROTOCOL_ENTRY.removeContextValue(ui);

            model.fromBean(protocol);
            model.setCloned(true);
            ui.getSaveWarning().setText(_("tutti.editProtocol.warn.clone"));

        } else {

            // create new protocol

            if (log.isDebugEnabled()) {
                log.debug("Will create a new protocol");
            }
        }

        SwingValidator validator = ui.getValidator();
        listenValidatorValid(validator, model);

        registerValidators(validator);

        Collection<Species> referents =
                model.getAllReferentSpeciesByTaxonId().values();

        initBeanFilterableComboBox(ui.getSpeciesComboBox(),
                                   Lists.newArrayList(referents), null);
        initBeanFilterableComboBox(ui.getBenthosComboBox(),
                                   Lists.newArrayList(referents), null);

        List<EditProtocolSpeciesRowModel> speciesRows;
        List<EditProtocolSpeciesRowModel> benthosRows;

        // build species and benthos rows
        if (protocol == null) {
            speciesRows = Lists.newArrayList();
            benthosRows = Lists.newArrayList();
        } else {

            speciesRows = toRows(protocol.getSpecies());
            benthosRows = toRows(protocol.getBenthos());

            if (log.isDebugEnabled()) {
                log.debug("Will edit protocol with " +
                          speciesRows.size() + " species and " +
                          benthosRows.size() + " benthos declared.");
            }
        }

        // set to model (will propagate to tableModel)
        model.setSpeciesRow(speciesRows);
        model.setBenthosRow(benthosRows);

        ui.getSpeciesComboBox().getHandler().reset();
        ui.getBenthosComboBox().getHandler().reset();

        {
            // create species table model

            JXTable table = getSpeciesTable();

            DefaultTableColumnModelExt columnModel = new DefaultTableColumnModelExt();

            addColumnToModel(columnModel,
                             null,
                             newTableCellRender(Species.class),
                             EditProtocolSpeciesTableModel.SPECIES_ID);

            addColumnToModel(columnModel,
                             null,
                             null,
                             EditProtocolSpeciesTableModel.SURVEY_CODE_ID);

            addLengthClassesColumnToModel(columnModel, model.getLengthClassesPmfmId());

            addBooleanColumnToModel(columnModel, EditProtocolSpeciesTableModel.WEIGHT_ENABLED, table);
            addBooleanColumnToModel(columnModel, EditProtocolSpeciesTableModel.COUNT_IF_NO_FREQUENCY_ENABLED, table);

            addBooleanColumnToModel(columnModel, EditProtocolSpeciesTableModel.SIZE_ENABLED, table);
            addBooleanColumnToModel(columnModel, EditProtocolSpeciesTableModel.SEX_ENABLED, table);
            addBooleanColumnToModel(columnModel, EditProtocolSpeciesTableModel.MATURITY_ENABLED, table);
            addBooleanColumnToModel(columnModel, EditProtocolSpeciesTableModel.AGE_ENABLED, table);

            addBooleanColumnToModel(columnModel, EditProtocolSpeciesTableModel.CALCIFY_SAMPLE_ENABLED, table);

            EditProtocolSpeciesTableModel tableModel =
                    new EditProtocolSpeciesTableModel(columnModel);
            table.setModel(tableModel);
            table.setColumnModel(columnModel);

            initTable(table);

            table.getSelectionModel().addListSelectionListener(new ListSelectionListener() {
                @Override
                public void valueChanged(ListSelectionEvent e) {
                    ListSelectionModel source = (ListSelectionModel) e.getSource();
                    ui.getRemoveSpeciesProtocolButton().setEnabled(
                            !source.isSelectionEmpty());
                }
            });

            tableModel.setRows(speciesRows);
        }

        {
            // create benthos table model

            JXTable table = getBenthosTable();

            DefaultTableColumnModelExt columnModel = new DefaultTableColumnModelExt();

            addColumnToModel(columnModel,
                             null,
                             newTableCellRender(Species.class),
                             EditProtocolSpeciesTableModel.SPECIES_ID);

            addColumnToModel(columnModel,
                             null,
                             null,
                             EditProtocolSpeciesTableModel.SURVEY_CODE_ID);

            addLengthClassesColumnToModel(columnModel, model.getLengthClassesPmfmId());

            addBooleanColumnToModel(columnModel, EditProtocolSpeciesTableModel.WEIGHT_ENABLED, table);
            addBooleanColumnToModel(columnModel, EditProtocolSpeciesTableModel.COUNT_IF_NO_FREQUENCY_ENABLED, table);

            addBooleanColumnToModel(columnModel, EditProtocolSpeciesTableModel.SIZE_ENABLED, table);
            addBooleanColumnToModel(columnModel, EditProtocolSpeciesTableModel.SEX_ENABLED, table);
            addBooleanColumnToModel(columnModel, EditProtocolSpeciesTableModel.MATURITY_ENABLED, table);
            addBooleanColumnToModel(columnModel, EditProtocolSpeciesTableModel.AGE_ENABLED, table);

            addBooleanColumnToModel(columnModel, EditProtocolSpeciesTableModel.CALCIFY_SAMPLE_ENABLED, table);

            EditProtocolSpeciesTableModel tableModel =
                    new EditProtocolSpeciesTableModel(columnModel);
            table.setModel(tableModel);
            table.setColumnModel(columnModel);

            initTable(table);

            table.getSelectionModel().addListSelectionListener(new ListSelectionListener() {
                @Override
                public void valueChanged(ListSelectionEvent e) {
                    ListSelectionModel source = (ListSelectionModel) e.getSource();
                    ui.getRemoveBenthosProtocolButton().setEnabled(
                            !source.isSelectionEmpty());
                }
            });

            tableModel.setRows(benthosRows);
        }

        initDoubleList(EditProtocolUIModel.PROPERTY_LENGTH_CLASSES_PMFM_ID,
                       ui.getLengthClassesList(),
                       Lists.newArrayList(model.getCaracteristics()),
                       model.getLengthClassesPmfmId(),
                       new Predicate<Caracteristic>() {

                           public boolean apply(Caracteristic input) {
                               boolean result = !ui.getGearUseFeatureList().getModel().getSelected().contains(input);
                               result &= !ui.getVesselUseFeatureList().getModel().getSelected().contains(input);
                               return result;
                           }
                       });

        initDoubleList(EditProtocolUIModel.PROPERTY_GEAR_USE_FEATURE_PMFM_ID,
                       ui.getGearUseFeatureList(),
                       Lists.newArrayList(model.getCaracteristics()),
                       model.getGearUseFeaturePmfmId(),
                       new Predicate<Caracteristic>() {

                           public boolean apply(Caracteristic input) {
                               boolean result = !ui.getLengthClassesList().getModel().getSelected().contains(input);
                               result &= !ui.getVesselUseFeatureList().getModel().getSelected().contains(input);
                               return result;
                           }
                       });

        initDoubleList(EditProtocolUIModel.PROPERTY_VESSEL_USE_FEATURE_PMFM_ID,
                       ui.getVesselUseFeatureList(),
                       Lists.newArrayList(model.getCaracteristics()),
                       model.getVesselUseFeaturePmfmId(),
                       new Predicate<Caracteristic>() {

                           public boolean apply(Caracteristic input) {
                               boolean result = !ui.getGearUseFeatureList().getModel().getSelected().contains(input);
                               result &= !ui.getLengthClassesList().getModel().getSelected().contains(input);
                               return result;
                           }
                       });

        // if new protocol can already cancel his creation
        model.setModify(model.isCreate());

        ui.getCaracteristicPane().addChangeListener(new ChangeListener() {

            public void stateChanged(ChangeEvent e) {
                JTabbedPane tabPanel = (JTabbedPane) e.getSource();
                int selectedIndex = tabPanel.getSelectedIndex();
                BeanDoubleList<Caracteristic> selectedDoubleList;
                log.debug("selected " + selectedIndex);
                switch (selectedIndex) {
                    case 0:
                        selectedDoubleList = ui.getLengthClassesList();
                        break;

                    case 1:
                        selectedDoubleList = ui.getGearUseFeatureList();
                        break;

                    case 2:
                        selectedDoubleList = ui.getVesselUseFeatureList();
                        break;

                    default:
                        selectedDoubleList = null;
                }
                if (selectedDoubleList != null) {
                    selectedDoubleList.getHandler().refreshFilteredElements();
                }
            }
        });

        dialog = new SelectSpeciesUI(ui);
    }

    @Override
    protected JComponent getComponentToFocus() {
        return getUI().getNameField();
    }

    public List<EditProtocolSpeciesRowModel> toRows(List<SpeciesProtocol> speciesProtocols) {
        BeanFilterableComboBox<Species> speciesComboBox = ui.getSpeciesComboBox();
        Preconditions.checkNotNull(speciesComboBox.getData());
        BeanFilterableComboBox<Species> benthosComboBox = ui.getBenthosComboBox();
        Preconditions.checkNotNull(benthosComboBox.getData());

        EditProtocolUIModel model = getModel();

        Map<String, Species> allReferentSpeciesByTaxonId = model.getAllReferentSpeciesByTaxonId();
        Map<String, Caracteristic> allCaracteristic = model.getAllCaracteristic();

        List<EditProtocolSpeciesRowModel> result = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(speciesProtocols)) {
            for (SpeciesProtocol speciesProtocol : speciesProtocols) {
                Integer taxonId = speciesProtocol.getSpeciesReferenceTaxonId();
                String taxonIdStr = String.valueOf(taxonId);

                // remove all synonyms from available synonym list
                Collection<Species> allSynonyms = model.getAllSynonyms(taxonIdStr);
                model.getAllSynonyms().removeAll(allSynonyms);

                // get species referent taxon
                Species species = allReferentSpeciesByTaxonId.get(taxonIdStr);

                // remove it from the combo box
                speciesComboBox.removeItem(species);
                benthosComboBox.removeItem(species);

                EditProtocolSpeciesRowModel row = EditProtocolSpeciesTableModel.newRow();
                row.setSpecies(species);
                row.setLengthStepPmfm(allCaracteristic.get(speciesProtocol.getLengthStepPmfmId()));
                row.fromBean(speciesProtocol);
                result.add(row);
            }
        }
        return result;
    }

    @Override
    public void onCloseUI() {
        if (log.isDebugEnabled()) {
            log.debug("closing: " + ui);
        }
        clearValidators();
        ui.getTabPanel().setSelectedIndex(0);
    }

    @Override
    public boolean quitUI() {
        boolean result = quitScreen(
                getModel().isValid(),
                getModel().isModify(),
                _("tutti.editProtocol.askCancelEditBeforeLeaving.cancelSaveProtocol"),
                _("tutti.editProtocol.askSaveBeforeLeaving.saveProtocol"),
                ui.getSaveButton().getAction()
        );
        return result;
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    public void addDoubleListListeners() {
        String id;
        UpdateSelectedList updateListener;
        EditProtocolUIModel model = getModel();

        id = (String) ui.getLengthClassesList().getClientProperty("_updateListenerId");
        updateListener = (UpdateSelectedList) ui.getLengthClassesList().getClientProperty("_updateListener");
        model.addPropertyChangeListener(id, updateListener);

        id = (String) ui.getVesselUseFeatureList().getClientProperty("_updateListenerId");
        updateListener = (UpdateSelectedList) ui.getVesselUseFeatureList().getClientProperty("_updateListener");
        model.addPropertyChangeListener(id, updateListener);

        id = (String) ui.getGearUseFeatureList().getClientProperty("_updateListenerId");
        updateListener = (UpdateSelectedList) ui.getGearUseFeatureList().getClientProperty("_updateListener");
        model.addPropertyChangeListener(id, updateListener);
    }

    public void removeDoubleListListeners() {
        String id;
        UpdateSelectedList updateListener;
        EditProtocolUIModel model = getModel();

        id = (String) ui.getLengthClassesList().getClientProperty("_updateListenerId");
        updateListener = (UpdateSelectedList) ui.getLengthClassesList().getClientProperty("_updateListener");

        model.removePropertyChangeListener(id, updateListener);

        id = (String) ui.getVesselUseFeatureList().getClientProperty("_updateListenerId");
        updateListener = (UpdateSelectedList) ui.getVesselUseFeatureList().getClientProperty("_updateListener");
        model.removePropertyChangeListener(id, updateListener);

        id = (String) ui.getGearUseFeatureList().getClientProperty("_updateListenerId");
        updateListener = (UpdateSelectedList) ui.getGearUseFeatureList().getClientProperty("_updateListener");
        model.removePropertyChangeListener(id, updateListener);
    }

    public Species openSelectOtherSpeciesDialog(String title, List<Species> species) {
        SelectSpeciesUIModel model = dialog.getModel();
        model.setSpecies(species);
        model.setSelectedSpecies(null);

        openDialog(dialog, title, new Dimension(400, 130));

        return model.getSelectedSpecies();
    }

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected void initDoubleList(String propertyId,
                                  BeanDoubleList<Caracteristic> widget,
                                  List<Caracteristic> availableCaracteristics,
                                  List<String> selectedCaracteristics,
                                  Predicate<Caracteristic> filter) {

        initBeanList(widget, availableCaracteristics,
                     Lists.<Caracteristic>newArrayList());

        UpdateSelectedList listener = new UpdateSelectedList(
                widget,
                getModel().getAllCaracteristic());
        widget.putClientProperty("_updateListener", listener);
        widget.putClientProperty("_updateListenerId", propertyId);
        listener.select(selectedCaracteristics);

        widget.getHandler().addFilter(filter);
    }

    protected void selectLengthClasses(List<String> ids, JComboBox comboBox) {

        Map<String, Caracteristic> allCaracteristic = getModel().getAllCaracteristic();
        List<Caracteristic> selection = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(ids)) {
            for (String id : ids) {
                selection.add(allCaracteristic.get(id));
            }
        }

        List<Caracteristic> dataToList = Lists.newArrayList(selection);

        // add a null value at first position
        if (!dataToList.isEmpty() && dataToList.get(0) != null) {
            dataToList.add(0, null);
        }
        SwingUtil.fillComboBox(comboBox, dataToList, null);
    }

    protected void addLengthClassesColumnToModel(TableColumnModel model,
                                                 List<String> selectedIds) {

        Decorator<Caracteristic> decorator =
                getDecorator(Caracteristic.class, null);

        final JComboBox comboBox = new JComboBox();

        getModel().addPropertyChangeListener(EditProtocolUIModel.PROPERTY_LENGTH_CLASSES_PMFM_ID, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                List<String> ids = (List<String>) evt.getNewValue();
                selectLengthClasses(ids, comboBox);
            }
        });
        comboBox.setRenderer(newListCellRender(decorator));

        selectLengthClasses(selectedIds, comboBox);
        ObjectToStringConverter converter = BeanUIUtil.newDecoratedObjectToStringConverter(decorator);
        BeanUIUtil.decorate(comboBox, converter);
        ComboBoxCellEditor editor = new ComboBoxCellEditor(comboBox);

        addColumnToModel(model,
                         editor,
                         newTableCellRender(decorator),
                         EditProtocolSpeciesTableModel.LENGTH_STEP_PMFM_ID);
    }

    protected void initTable(JXTable table) {

        // by default do not authorize to change column orders
        table.getTableHeader().setReorderingAllowed(false);

        addHighlighters(table);

        // always scroll to selected row
        SwingUtil.scrollToTableSelection(table);

        table.getModel().addTableModelListener(new TableModelListener() {
            @Override
            public void tableChanged(TableModelEvent e) {
                getModel().setModify(true);
            }
        });
    }

    protected void addHighlighters(final JXTable table) {
        // paint in a special color for read only cells
        Highlighter readOnlyHighlighter = TuttiUIUtil.newBackgroundColorHighlighter(
                HighlightPredicate.READ_ONLY, getConfig().getColorRowReadOnly());
        table.addHighlighter(readOnlyHighlighter);

        // paint in a special color inValid rows
        Highlighter validHighlighter = TuttiUIUtil.newBackgroundColorHighlighter(
                new HighlightPredicate.AndHighlightPredicate(HighlightPredicate.EDITABLE, new HighlightPredicate() {
                    @Override
                    public boolean isHighlighted(Component renderer, ComponentAdapter adapter) {

                        boolean result = false;
                        if (adapter.isEditable()) {
                            EditProtocolSpeciesTableModel model = (EditProtocolSpeciesTableModel) table.getModel();
                            EditProtocolSpeciesRowModel row = model.getEntry(adapter.row);
                            result = !row.isValid();
                        }
                        return result;
                    }
                }), getConfig().getColorRowInvalid());
        table.addHighlighter(validHighlighter);
    }

    protected static class UpdateSelectedList implements PropertyChangeListener {

        private final BeanDoubleListModel<Caracteristic> model;

        private final Map<String, Caracteristic> caracteristicMap;

        public UpdateSelectedList(BeanDoubleList<Caracteristic> doubleList,
                                  Map<String, Caracteristic> caracteristicMap) {
            this.model = doubleList.getModel();
            this.caracteristicMap = caracteristicMap;
        }

        private boolean valueIsAdjusting;

        @Override
        public void propertyChange(PropertyChangeEvent evt) {

            if (!valueIsAdjusting) {

                valueIsAdjusting = true;

                try {
                    List<String> selectedIds = (List<String>) evt.getNewValue();
                    if (log.isInfoEnabled()) {
                        log.info("[" + evt.getPropertyName() + "] selected ids: " + selectedIds);
                    }

                    select(selectedIds);
                } finally {
                    valueIsAdjusting = false;
                }
            }
        }

        public void select(List<String> selectedIds) {

            List<Caracteristic> selection = Lists.newArrayList();
            if (CollectionUtils.isNotEmpty(selectedIds)) {
                for (String selectedId : selectedIds) {
                    selection.add(caracteristicMap.get(selectedId));
                }
            }
            model.setSelected(selection);
        }
    }
}
