package fr.ifremer.tutti.ui.swing.content.operation.catches.plankton;

/*
 * #%L
 * Tutti :: UI
 * $Id: PlanktonBatchUIHandler.java 773 2013-04-13 20:09:01Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.4/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/catches/plankton/PlanktonBatchUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Lists;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.data.Attachment;
import fr.ifremer.tutti.persistence.entities.data.FishingOperation;
import fr.ifremer.tutti.persistence.entities.data.PlanktonBatch;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.ui.swing.content.operation.AbstractTuttiBatchTableUIHandler;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUIModel;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import fr.ifremer.tutti.ui.swing.util.TuttiUI;
import fr.ifremer.tutti.ui.swing.util.attachment.AttachmentCellEditor;
import fr.ifremer.tutti.ui.swing.util.attachment.AttachmentCellRenderer;
import fr.ifremer.tutti.ui.swing.util.comment.CommentCellEditor;
import fr.ifremer.tutti.ui.swing.util.comment.CommentCellRenderer;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.table.DefaultTableColumnModelExt;
import org.nuiton.util.decorator.Decorator;

import javax.swing.JComponent;
import java.util.List;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 0.2
 */
public class PlanktonBatchUIHandler extends AbstractTuttiBatchTableUIHandler<PlanktonBatchRowModel, PlanktonBatchUIModel, PlanktonBatchUI> {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(PlanktonBatchUIHandler.class);

    public PlanktonBatchUIHandler(TuttiUI<?, ?> parentUi, PlanktonBatchUI ui) {
        super(parentUi, ui,
              PlanktonBatchRowModel.PROPERTY_SPECIES_TO_CONFIRM,
              PlanktonBatchRowModel.PROPERTY_SPECIES,
              PlanktonBatchRowModel.PROPERTY_WEIGHT,
              PlanktonBatchRowModel.PROPERTY_COMMENT);
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiBatchTableUIHandler methods                           --//
    //------------------------------------------------------------------------//

    @Override
    public void selectFishingOperation(FishingOperation bean) {

        boolean empty = bean == null;

        PlanktonBatchUIModel model = getModel();

        List<PlanktonBatchRowModel> rows;

        if (empty) {
            rows = null;
        } else {


            if (log.isInfoEnabled()) {
                log.info("Get plankton batch for fishingOperation: " +
                         bean.getId());
            }
            rows = Lists.newArrayList();

            if (!TuttiEntities.isNew(bean)) {
                List<PlanktonBatch> catches =
                        persistenceService.getAllPlanktonBatch(bean.getId());
                for (PlanktonBatch aBatch : catches) {
                    PlanktonBatchRowModel entry =
                            new PlanktonBatchRowModel(aBatch);
                    rows.add(entry);
                }
            }
        }
        model.setRows(rows);
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiTableUIHandler methods                                --//
    //------------------------------------------------------------------------//

    @Override
    public PlanktonBatchTableModel getTableModel() {
        return (PlanktonBatchTableModel) getTable().getModel();
    }

    @Override
    public JXTable getTable() {
        return ui.getTable();
    }

    @Override
    protected boolean isRowValid(PlanktonBatchRowModel row) {
        boolean result = row.getSpecies() != null && row.getWeight() != null;
        return result;
    }

    @Override
    protected void onRowModified(int rowIndex,
                                 PlanktonBatchRowModel row,
                                 String propertyName,
                                 Object oldValue,
                                 Object newValue) {
        recomputeRowValidState(row);
    }

    @Override
    protected void saveSelectedRowIfRequired(TuttiBeanMonitor<PlanktonBatchRowModel> rowMonitor,
                                             PlanktonBatchRowModel row) {

        if (row.isValid()) {
            // there is a valid bean attached to the monitor

            if (rowMonitor.wasModified()) {

                // monitored bean was modified, save it
                if (log.isInfoEnabled()) {
                    log.info("Row " + row + " was modified, will save it");
                }

                saveRow(row);

                // clear modified flag on the monitor
                rowMonitor.clearModified();
            }
        } else {

            // row is not valid can not save it

            PlanktonBatch catchBean = row.toBean();

            if (!TuttiEntities.isNew(catchBean)) {

                // remove this
                persistenceService.deletePlanktonBatch(catchBean.getId());
            }
        }
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    public SwingValidator<PlanktonBatchUIModel> getValidator() {
        return ui.getValidator();
    }

    @Override
    public void beforeInitUI() {

        if (log.isInfoEnabled()) {
            log.info("beforeInit: " + ui);
        }

        EditCatchesUIModel catchesUIModel =
                ui.getContextValue(EditCatchesUIModel.class);

        PlanktonBatchUIModel model = new PlanktonBatchUIModel(catchesUIModel);
        ui.setContextValue(model);
    }

    @Override
    public void afterInitUI() {

        if (log.isInfoEnabled()) {
            log.info("afterInit: " + ui);
        }

        initUI(ui);

        JXTable table = getTable();

        // create table column model
        DefaultTableColumnModelExt columnModel =
                new DefaultTableColumnModelExt();

        List<Species> allSpecies = persistenceService.getAllSpecies();

        { // Species (by code) column

            Decorator<Species> decorator = getDecorator(Species.class, null);

            addComboDataColumnToModel(columnModel,
                                      PlanktonBatchTableModel.SPECIES_BY_CODE,
                                      decorator, allSpecies);
        }

        { // Species (by genusCode) column

            Decorator<Species> decorator = getDecorator(Species.class, null);

            addComboDataColumnToModel(columnModel,
                                      PlanktonBatchTableModel.SPECIES_BY_GENUS_CODE,
                                      decorator, allSpecies);

        }

        { // Weight column

            addFloatColumnToModel(columnModel,
                                  PlanktonBatchTableModel.WEIGHT,
                                  TuttiUI.DECIMAL3_PATTERN);
        }

        { // Sample weight column

            addColumnToModel(columnModel,
                             PlanktonBatchTableModel.SAMPLE_WEIGHT);
        }

        { // Comment column

            addColumnToModel(columnModel,
                             CommentCellEditor.newEditor(ui),
                             CommentCellRenderer.newRender(),
                             PlanktonBatchTableModel.COMMENT);
        }

        { // File column

            addColumnToModel(columnModel,
                             AttachmentCellEditor.newEditor(ui),
                             AttachmentCellRenderer.newRender(getDecorator(Attachment.class, null)),
                             PlanktonBatchTableModel.ATTACHMENT);
        }

        { // Species to confirm column

            addBooleanColumnToModel(columnModel,
                                    PlanktonBatchTableModel.SPECIES_TO_CONFIRM,
                                    getTable());
        }

        // create table model
        PlanktonBatchTableModel tableModel =
                new PlanktonBatchTableModel(columnModel);

        table.setModel(tableModel);
        table.setColumnModel(columnModel);

        initBatchTable(table, columnModel, tableModel);
    }

    @Override
    protected JComponent getComponentToFocus() {
        return getUI().getTable();
    }

    @Override
    public void onCloseUI() {
        if (log.isDebugEnabled()) {
            log.debug("closing: " + ui);
        }
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected void saveRow(PlanktonBatchRowModel row) {

        PlanktonBatch catchBean = row.toBean();

        FishingOperation fishingOperation = getModel().getFishingOperation();
        catchBean.setFishingOperation(fishingOperation);
        if (log.isInfoEnabled()) {
            log.info("Selected fishingOperation: " + fishingOperation.getId());
        }

        if (TuttiEntities.isNew(catchBean)) {

            catchBean = persistenceService.createPlanktonBatch(catchBean);
            row.setId(catchBean.getId());
        } else {
            persistenceService.savePlanktonBatch(catchBean);
        }
    }
}
