package fr.ifremer.tutti.ui.swing.content.operation.catches.benthos;

/*
 * #%L
 * Tutti :: UI
 * $Id: BenthosBatchRowModel.java 692 2013-03-27 21:44:37Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.2/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/catches/benthos/BenthosBatchRowModel.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Lists;
import fr.ifremer.tutti.persistence.entities.data.AttachementObjectTypeEnum;
import fr.ifremer.tutti.persistence.entities.data.Attachment;
import fr.ifremer.tutti.persistence.entities.data.SampleCategoryEnum;
import fr.ifremer.tutti.persistence.entities.data.SpeciesBatch;
import fr.ifremer.tutti.persistence.entities.data.SpeciesBatchFrequency;
import fr.ifremer.tutti.persistence.entities.referential.CaracteristicQualitativeValue;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.ui.swing.content.operation.catches.SampleCategory;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.frequency.BenthosFrequencyRowModel;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiBeanUIModel;
import fr.ifremer.tutti.ui.swing.util.TuttiComputedOrNotData;
import fr.ifremer.tutti.ui.swing.util.attachment.AttachmentModelAware;
import fr.ifremer.tutti.ui.swing.util.comment.CommentModelAware;
import org.apache.commons.collections.CollectionUtils;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.util.Collection;
import java.util.Collections;
import java.util.List;

/**
 * Represents a species batch (i.e a row in the batch table).
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.2
 */
public class BenthosBatchRowModel extends AbstractTuttiBeanUIModel<SpeciesBatch, BenthosBatchRowModel> implements AttachmentModelAware, CommentModelAware {

    private static final long serialVersionUID = 1L;

    public static final String PROPERTY_SPECIES_TO_CONFIRM = "speciesToConfirm";

    public static final String PROPERTY_SPECIES = "species";

    public static final String PROPERTY_SORTED_UNSORTED_CATEGORY = "sortedUnsortedCategory";

    public static final String PROPERTY_SORTED_UNSORTED_CATEGORY_VALUE = "sortedUnsortedCategoryValue";

    public static final String PROPERTY_SORTED_UNSORTED_CATEGORY_WEIGHT = "sortedUnsortedCategoryWeight";

    public static final String PROPERTY_SORTED_UNSORTED_COMPUTED_WEIGHT = "sortedUnsortedComputedWeight";

    public static final String PROPERTY_SIZE_CATEGORY = "sizeCategory";

    public static final String PROPERTY_SIZE_CATEGORY_VALUE = "sizeCategoryValue";

    public static final String PROPERTY_SIZE_CATEGORY_WEIGHT = "sizeCategoryWeight";

    public static final String PROPERTY_SEX_CATEGORY = "sexCategory";

    public static final String PROPERTY_SEX_CATEGORY_VALUE = "sexCategoryValue";

    public static final String PROPERTY_SEX_CATEGORY_WEIGHT = "sexCategoryWeight";

    public static final String PROPERTY_MATURITY_CATEGORY = "maturityCategory";

    public static final String PROPERTY_MATURITY_CATEGORY_VALUE = "maturityCategoryValue";

    public static final String PROPERTY_MATURITY_CATEGORY_WEIGHT = "maturityCategoryWeight";

    public static final String PROPERTY_AGE_CATEGORY = "ageCategory";

    public static final String PROPERTY_AGE_CATEGORY_VALUE = "ageCategoryValue";

    public static final String PROPERTY_AGE_CATEGORY_WEIGHT = "ageCategoryWeight";

    public static final String PROPERTY_WEIGHT = "weight";

    public static final String PROPERTY_NUMBER = "number";

    public static final String PROPERTY_FREQUENCY = "frequency";

    public static final String PROPERTY_COMPUTED_NUMBER = "computedOrNotNumber";

    public static final String PROPERTY_COMPUTED_WEIGHT = "computedOrNotWeight";

    public static final String PROPERTY_PARENT_BATCH = "parentBatch";

    public static final String PROPERTY_CHILD_BATCH = "childBatch";

    public static final String PROPERTY_BATCH_LEAF = "batchLeaf";

    public static final String PROPERTY_BATCH_ROOT = "batchRoot";

    public static final String PROPERTY_SAMPLE_CATEGORY = "sampleCategory";

    /**
     * Species.
     *
     * @since 0.3
     */
    protected Species species;

    /**
     * Is catch is vrac or  horsVrac?.
     *
     * @since 0.2
     */
    protected SampleCategory<CaracteristicQualitativeValue> sortedUnsortedCategory;

    /**
     * Weight category (can be null).
     *
     * @since 0.2
     */
    protected SampleCategory<CaracteristicQualitativeValue> sizeCategory;

    /**
     * Sex (can be null).
     *
     * @since 0.2
     */
    protected SampleCategory<CaracteristicQualitativeValue> sexCategory;

    /**
     * Maturity (can be null).
     *
     * @since 0.2
     */
    protected SampleCategory<CaracteristicQualitativeValue> maturityCategory;

    /**
     * Age (can be null).
     *
     * @since 0.2
     */
    protected SampleCategory<Float> ageCategory;

    /**
     * Sample category of this batch.
     * <p/>
     * This is one of the sample category befined above, used to know if the
     * table row which sample category affects this row exactly.
     *
     * @since 0.3
     */
    private SampleCategory<?> sampleCategory;

    /**
     * Is the species need to be confirmed?.
     *
     * @since 0.2
     */
    protected boolean speciesToConfirm;

    /**
     * Observed weight.
     *
     * @since 0.2
     */
    protected TuttiComputedOrNotData<Float> computedOrNotWeight =
            new TuttiComputedOrNotData<Float>();

    /**
     * Total computed number (from frequencies).
     *
     * @since 0.2
     */
    protected TuttiComputedOrNotData<Integer> computedOrNotNumber =
            new TuttiComputedOrNotData<Integer>();

    /**
     * Comment.
     *
     * @since 0.2
     */
    protected String comment;

    /**
     * Attachments (should never be null).
     *
     * @since 0.2
     */
    protected final List<Attachment> attachment = Lists.newArrayList();

    /**
     * List of frequencies observed for this batch.
     *
     * @since 0.2
     */
    protected List<BenthosFrequencyRowModel> frequency = Lists.newArrayList();

    /**
     * Parent of this batch (can be null if batch is root).
     *
     * @see #isBatchRoot()
     * @since 0.3
     */
    protected BenthosBatchRowModel parentBatch;

    /**
     * List of child batches (can be null or empty if batch is a leaf).
     *
     * @see #isBatchLeaf()
     * @since 0.3
     */
    protected List<BenthosBatchRowModel> childBatch;

    protected static final Binder<SpeciesBatch, BenthosBatchRowModel> fromBeanBinder =
            BinderFactory.newBinder(SpeciesBatch.class,
                                    BenthosBatchRowModel.class);

    protected static final Binder<BenthosBatchRowModel, SpeciesBatch> toBeanBinder =
            BinderFactory.newBinder(BenthosBatchRowModel.class,
                                    SpeciesBatch.class);

    public BenthosBatchRowModel() {
        super(SpeciesBatch.class, fromBeanBinder, toBeanBinder);
        sortedUnsortedCategory = SampleCategory.newSample(SampleCategoryEnum.sortedUnsorted);
        sizeCategory = SampleCategory.newSample(SampleCategoryEnum.size);
        sexCategory = SampleCategory.newSample(SampleCategoryEnum.sex);
        maturityCategory = SampleCategory.newSample(SampleCategoryEnum.maturity);
        ageCategory = SampleCategory.newSample(SampleCategoryEnum.age);

        computedOrNotWeight.addPropagateListener(PROPERTY_WEIGHT, this);
        computedOrNotWeight.addPropagateListener(PROPERTY_COMPUTED_WEIGHT, this);
        computedOrNotNumber.addPropagateListener(PROPERTY_NUMBER, this);
        computedOrNotNumber.addPropagateListener(PROPERTY_COMPUTED_NUMBER, this);
    }

    public BenthosBatchRowModel(SpeciesBatch aBatch,
                                List<SpeciesBatchFrequency> frequencies,
                                List<Attachment> attachments) {
        this();
        fromBean(aBatch);
        List<BenthosFrequencyRowModel> frequencyRows =
                BenthosFrequencyRowModel.fromBeans(frequencies);
        frequency.addAll(frequencyRows);
        this.attachment.addAll(attachments);
        Collections.sort(frequency);
    }

    //------------------------------------------------------------------------//
    //-- Sample category                                                    --//
    //------------------------------------------------------------------------//

    public SampleCategory<?> getSampleCategory() {
        return sampleCategory;
    }

    public void setSampleCategory(SampleCategory<?> sampleCategory) {
        Object oldValue = getSampleCategory();
        this.sampleCategory = sampleCategory;
        firePropertyChange(PROPERTY_SAMPLE_CATEGORY, oldValue, sampleCategory);
    }

    public SampleCategory<?> getSampleCategory(SampleCategoryEnum sampleCategoryEnum) {
        switch (sampleCategoryEnum) {
            case sortedUnsorted:
                sampleCategory = getSortedUnsortedCategory();
                break;
            case size:
                sampleCategory = getSizeCategory();
                break;
            case sex:
                sampleCategory = getSexCategory();
                break;
            case maturity:
                sampleCategory = getMaturityCategory();
                break;
            case age:
                sampleCategory = getAgeCategory();
                break;
            default:
        }
        return sampleCategory;
    }

    public SampleCategory getFinestCategory() {
        SampleCategory finestCategory;
        if (ageCategory.isValid()) {
            finestCategory = ageCategory;

        } else if (maturityCategory.isValid()) {
            finestCategory = maturityCategory;

        } else if (sexCategory.isValid()) {
            finestCategory = sexCategory;

        } else if (sizeCategory.isValid()) {
            finestCategory = sizeCategory;

        } else {
            finestCategory = sortedUnsortedCategory;
        }
        return finestCategory;
    }

    //------------------------------------------------------------------------//
    //-- Species category                                                   --//
    //------------------------------------------------------------------------//

    public Species getSpecies() {
        return species;
    }

    public void setSpecies(Species species) {
        Object oldCategory = getSpecies();
        this.species = species;
        firePropertyChange(PROPERTY_SPECIES, oldCategory, species);
    }

    //------------------------------------------------------------------------//
    //-- SortedUnsorted category                                            --//
    //------------------------------------------------------------------------//

    public SampleCategory<CaracteristicQualitativeValue> getSortedUnsortedCategory() {
        return sortedUnsortedCategory;
    }

    public void setSortedUnsortedCategory(SampleCategory<CaracteristicQualitativeValue> sortedUnsortedCategory) {
        Object oldCategory = getSortedUnsortedCategory();
        Object oldValue = getSortedUnsortedCategoryValue();
        Object oldWeight = getSortedUnsortedCategoryWeight();
        this.sortedUnsortedCategory = sortedUnsortedCategory;
        firePropertyChange(PROPERTY_SORTED_UNSORTED_CATEGORY, oldCategory, sortedUnsortedCategory);
        firePropertyChange(PROPERTY_SORTED_UNSORTED_CATEGORY_VALUE, oldValue, getSortedUnsortedCategoryValue());
        firePropertyChange(PROPERTY_SORTED_UNSORTED_CATEGORY_WEIGHT, oldWeight, getSortedUnsortedCategoryWeight());
    }

    public CaracteristicQualitativeValue getSortedUnsortedCategoryValue() {
        return sortedUnsortedCategory.getCategoryValue();
    }

    public Float getSortedUnsortedCategoryWeight() {
        return sortedUnsortedCategory.getCategoryWeight();
    }

    public void setSortedUnsortedCategoryWeight(Float sortedUnsortedCategoryWeight) {
        Object oldValue = getSortedUnsortedCategoryWeight();
        sortedUnsortedCategory.setCategoryWeight(sortedUnsortedCategoryWeight);
        firePropertyChange(PROPERTY_SORTED_UNSORTED_CATEGORY_WEIGHT, oldValue, sortedUnsortedCategoryWeight);
    }

    //------------------------------------------------------------------------//
    //-- Size category                                                      --//
    //------------------------------------------------------------------------//

    public SampleCategory<CaracteristicQualitativeValue> getSizeCategory() {
        return sizeCategory;
    }

    public void setSizeCategory(SampleCategory<CaracteristicQualitativeValue> sizeCategory) {
        Object oldCategory = getSizeCategory();
        Object oldValue = getSizeCategoryValue();
        Object oldWeight = getSizeCategoryWeight();
        this.sizeCategory = sizeCategory;
        firePropertyChange(PROPERTY_SIZE_CATEGORY, oldCategory, sizeCategory);
        firePropertyChange(PROPERTY_SIZE_CATEGORY_VALUE, oldValue, getSizeCategoryValue());
        firePropertyChange(PROPERTY_SIZE_CATEGORY_WEIGHT, oldWeight, getSizeCategoryWeight());
    }

    public CaracteristicQualitativeValue getSizeCategoryValue() {
        return sizeCategory.getCategoryValue();
    }

    public Float getSizeCategoryWeight() {
        return sizeCategory.getCategoryWeight();
    }

    public void setSizeCategoryWeight(Float sizeCategoryWeight) {
        Object oldValue = getSizeCategoryWeight();
        this.sizeCategory.setCategoryWeight(sizeCategoryWeight);
        firePropertyChange(PROPERTY_SIZE_CATEGORY_WEIGHT, oldValue, sizeCategoryWeight);
    }

    //------------------------------------------------------------------------//
    //-- Sex category                                                       --//
    //------------------------------------------------------------------------//

    public SampleCategory<CaracteristicQualitativeValue> getSexCategory() {
        return sexCategory;
    }

    public void setSexCategory(SampleCategory<CaracteristicQualitativeValue> sexCategory) {
        Object oldCategory = getSexCategory();
        Object oldValue = getSexCategoryValue();
        Object oldWeight = getSexCategoryWeight();
        this.sexCategory = sexCategory;
        firePropertyChange(PROPERTY_SEX_CATEGORY, oldCategory, sexCategory);
        firePropertyChange(PROPERTY_SEX_CATEGORY_VALUE, oldValue, getSexCategoryValue());
        firePropertyChange(PROPERTY_SEX_CATEGORY_WEIGHT, oldWeight, getSexCategoryWeight());
    }

    public CaracteristicQualitativeValue getSexCategoryValue() {
        return sexCategory.getCategoryValue();
    }

    public Float getSexCategoryWeight() {
        return sexCategory.getCategoryWeight();
    }

    public void setSexCategoryWeight(Float sexCategoryWeight) {
        Object oldValue = getSexCategoryWeight();
        this.sexCategory.setCategoryWeight(sexCategoryWeight);
        firePropertyChange(PROPERTY_SEX_CATEGORY_WEIGHT, oldValue, sexCategoryWeight);
    }

    //------------------------------------------------------------------------//
    //-- Maturity category                                                  --//
    //------------------------------------------------------------------------//

    public SampleCategory<CaracteristicQualitativeValue> getMaturityCategory() {
        return maturityCategory;
    }

    public void setMaturityCategory(SampleCategory<CaracteristicQualitativeValue> maturityCategory) {
        Object oldCategory = getMaturityCategory();
        Object oldValue = getMaturityCategoryValue();
        Object oldWeight = getMaturityCategoryWeight();
        this.maturityCategory = maturityCategory;
        firePropertyChange(PROPERTY_MATURITY_CATEGORY, oldCategory, maturityCategory);
        firePropertyChange(PROPERTY_MATURITY_CATEGORY_VALUE, oldValue, getMaturityCategoryValue());
        firePropertyChange(PROPERTY_MATURITY_CATEGORY_WEIGHT, oldWeight, getMaturityCategoryWeight());
    }

    public CaracteristicQualitativeValue getMaturityCategoryValue() {
        return maturityCategory.getCategoryValue();
    }

    public Float getMaturityCategoryWeight() {
        return maturityCategory.getCategoryWeight();
    }

    public void setMaturityCategoryWeight(Float maturityCategoryWeight) {
        Object oldValue = getMaturityCategoryWeight();
        this.maturityCategory.setCategoryWeight(maturityCategoryWeight);
        firePropertyChange(PROPERTY_MATURITY_CATEGORY_WEIGHT, oldValue, maturityCategoryWeight);
    }

    //------------------------------------------------------------------------//
    //-- Age category                                                       --//
    //------------------------------------------------------------------------//

    public SampleCategory<Float> getAgeCategory() {
        return ageCategory;
    }

    public void setAgeCategory(SampleCategory<Float> ageCategory) {
        Object oldCategory = getAgeCategory();
        Object oldValue = getAgeCategoryValue();
        Object oldWeight = getAgeCategoryWeight();
        this.ageCategory = ageCategory;

        firePropertyChange(PROPERTY_AGE_CATEGORY, oldCategory, ageCategory);
        firePropertyChange(PROPERTY_AGE_CATEGORY_VALUE, oldValue, getAgeCategoryValue());
        firePropertyChange(PROPERTY_AGE_CATEGORY_WEIGHT, oldWeight, getAgeCategoryWeight());
    }

    public Float getAgeCategoryValue() {
        return ageCategory.getCategoryValue();
    }

    public Float getAgeCategoryWeight() {
        return ageCategory.getCategoryWeight();
    }

    public void setAgeCategoryWeight(Float ageCategoryWeight) {
        Object oldValue = getAgeCategoryWeight();
        ageCategory.setCategoryWeight(ageCategoryWeight);
        firePropertyChange(PROPERTY_AGE_CATEGORY_WEIGHT, oldValue, ageCategoryWeight);
    }

    //------------------------------------------------------------------------//
    //-- Navigation properties                                              --//
    //------------------------------------------------------------------------//

    public BenthosBatchRowModel getParentBatch() {
        return parentBatch;
    }

    public void setParentBatch(BenthosBatchRowModel parentBatch) {
        Object oldValue = getParentBatch();
        this.parentBatch = parentBatch;
        firePropertyChange(PROPERTY_PARENT_BATCH, oldValue, parentBatch);
        firePropertyChange(PROPERTY_BATCH_ROOT, null, isBatchRoot());
    }

    public List<BenthosBatchRowModel> getChildBatch() {
        return childBatch;
    }

    public void setChildBatch(List<BenthosBatchRowModel> childBatch) {
        this.childBatch = childBatch;
        // force to propagate child changes
        firePropertyChange(PROPERTY_CHILD_BATCH, null, childBatch);
        firePropertyChange(PROPERTY_BATCH_LEAF, null, isBatchLeaf());
    }

    public boolean isBatchLeaf() {
        return CollectionUtils.isEmpty(childBatch);
    }

    public boolean isBatchRoot() {
        return parentBatch == null;
    }

    //------------------------------------------------------------------------//
    //-- CommentModelAware                                                  --//
    //------------------------------------------------------------------------//

    @Override
    public String getComment() {
        return comment;
    }

    @Override
    public void setComment(String comment) {
        Object oldValue = getComment();
        this.comment = comment;
        firePropertyChange(PROPERTY_COMMENT, oldValue, comment);
    }

    //------------------------------------------------------------------------//
    //-- AttachmentModelAware                                               --//
    //------------------------------------------------------------------------//

    @Override
    public AttachementObjectTypeEnum getObjectType() {
        return AttachementObjectTypeEnum.SAMPLE;
    }

    @Override
    public String getObjectId() {
        return getId();
    }

    @Override
    public List<Attachment> getAttachment() {
        return attachment;
    }

    @Override
    public void addAllAttachment(Collection<Attachment> attachments) {
        this.attachment.addAll(attachments);
        firePropertyChange(PROPERTY_ATTACHMENT, null, getAttachment());
    }

    @Override
    public void addAttachment(Attachment attachment) {
        this.attachment.add(attachment);
        firePropertyChange(PROPERTY_ATTACHMENT, null, getAttachment());
    }

    @Override
    public void removeAllAttachment(Collection<Attachment> attachments) {
        this.attachment.removeAll(attachments);
        firePropertyChange(PROPERTY_ATTACHMENT, null, getAttachment());
    }

    @Override
    public void removeAttachment(Attachment attachment) {
        this.attachment.remove(attachment);
        firePropertyChange(PROPERTY_ATTACHMENT, null, getAttachment());
    }

    //------------------------------------------------------------------------//
    //-- Other properties                                                   --//
    //------------------------------------------------------------------------//

    public Boolean getSpeciesToConfirm() {
        return speciesToConfirm;
    }

    public void setSpeciesToConfirm(Boolean speciesToConfirm) {
        Object oldValue = getSpeciesToConfirm();
        this.speciesToConfirm = speciesToConfirm;
        firePropertyChange(PROPERTY_SPECIES_TO_CONFIRM, oldValue, speciesToConfirm);
    }

    public Float getWeight() {
        return computedOrNotWeight.getData();
    }

    public void setWeight(Float weight) {
        this.computedOrNotWeight.setData(weight);
    }

    public Integer getNumber() {
        return computedOrNotNumber.getData();
    }

    public void setNumber(Integer number) {
        computedOrNotNumber.setData(number);
    }

    public List<BenthosFrequencyRowModel> getFrequency() {
        return frequency;
    }

    public void setFrequency(List<BenthosFrequencyRowModel> frequency) {
        this.frequency = frequency;
        // force to propagate frequencies changes
        firePropertyChange(PROPERTY_FREQUENCY, null, frequency);
    }

    public Integer getComputedNumber() {
        return computedOrNotNumber.getComputedData();
    }

    public void setComputedNumber(Integer computedNumber) {
        computedOrNotNumber.setComputedData(computedNumber);
    }

    public Float getComputedWeight() {
        return computedOrNotWeight.getComputedData();
    }

    public void setComputedWeight(Float computedWeight) {
        computedOrNotWeight.setComputedData(computedWeight);
    }

    public TuttiComputedOrNotData<Integer> getComputedOrNotNumber() {
        return computedOrNotNumber;
    }

    public void setComputedOrNotNumber(TuttiComputedOrNotData<Integer> computedOrNotNumber) {
        this.computedOrNotNumber = computedOrNotNumber;
    }

    public TuttiComputedOrNotData<Float> getComputedOrNotWeight() {
        return computedOrNotWeight;
    }

    public void setComputedOrNotWeight(TuttiComputedOrNotData<Float> computedOrNotWeight) {
        this.computedOrNotWeight = computedOrNotWeight;
    }

}
