package fr.ifremer.tutti.ui.swing.util;

/*
 * #%L
 * Tutti :: UI
 * $Id: AbstractTuttiUIHandler.java 630 2013-03-15 10:20:17Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.1/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/util/AbstractTuttiUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import fr.ifremer.tutti.TuttiTechnicalException;
import fr.ifremer.tutti.persistence.entities.IdAware;
import fr.ifremer.tutti.service.DecoratorService;
import fr.ifremer.tutti.ui.swing.TuttiDataContext;
import fr.ifremer.tutti.ui.swing.TuttiUIContext;
import fr.ifremer.tutti.ui.swing.config.TuttiApplicationConfig;
import fr.ifremer.tutti.ui.swing.content.AbstractMainUITuttiAction;
import fr.ifremer.tutti.ui.swing.content.MainUI;
import fr.ifremer.tutti.ui.swing.content.MainUIHandler;
import fr.ifremer.tutti.ui.swing.util.action.AbstractTuttiAction;
import fr.ifremer.tutti.ui.swing.util.action.TuttiActionUI;
import fr.ifremer.tutti.ui.swing.util.action.TuttiUIAction;
import fr.ifremer.tutti.ui.swing.util.attachment.ButtonAttachment;
import fr.ifremer.tutti.ui.swing.util.editor.SimpleTimeEditor;
import jaxx.runtime.JAXXUtil;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.swing.editor.NumberEditor;
import jaxx.runtime.swing.editor.bean.BeanComboBox;
import jaxx.runtime.swing.editor.bean.BeanDoubleList;
import jaxx.runtime.swing.editor.bean.BeanFilterableComboBox;
import jaxx.runtime.swing.renderer.DecoratorListCellRenderer;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.lang3.reflect.ConstructorUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXDatePicker;
import org.nuiton.util.decorator.Decorator;
import org.nuiton.util.decorator.JXPathDecorator;
import org.nuiton.validator.bean.simple.SimpleBeanValidator;

import javax.swing.AbstractAction;
import javax.swing.AbstractButton;
import javax.swing.Action;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JRootPane;
import javax.swing.JTextField;
import javax.swing.KeyStroke;
import javax.swing.ListCellRenderer;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.text.JTextComponent;
import java.awt.Component;
import java.awt.Dialog;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Frame;
import java.awt.event.ActionEvent;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.ItemEvent;
import java.awt.event.KeyEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.io.Serializable;
import java.text.ParseException;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static org.nuiton.i18n.I18n._;

/**
 * Contract of any UI handler.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public abstract class AbstractTuttiUIHandler<M, UI extends TuttiUI<M, ?>> implements UIMessageNotifier {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(AbstractTuttiUIHandler.class);

    public abstract void beforeInitUI();

    public abstract void afterInitUI();

    public abstract void onCloseUI();

    public abstract SwingValidator<M> getValidator();

    /**
     * Global application context.
     *
     * @since 0.1
     */
    protected final TuttiUIContext context;

    /**
     * UI handled.
     *
     * @since 0.1
     */
    protected final UI ui;

    protected AbstractTuttiUIHandler(TuttiUIContext context, UI ui) {
        this.context = context;
        this.ui = ui;
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    public DefaultComboBoxModel newComboModel(Object... items) {
        return new DefaultComboBoxModel(items);
    }

    public final M getModel() {
        return ui.getModel();
    }

    public final UI getUI() {
        return ui;
    }

    @Override
    public void showInformationMessage(String message) {
        context.showInformationMessage(message);
    }

    public TuttiUIContext getContext() {
        return context;
    }

    public TuttiDataContext getDataContext() {
        return getContext().getDataContext();
    }

    public TuttiApplicationConfig getConfig() {
        return context.getConfig();
    }

    public void setText(KeyEvent event, String property) {
        JTextComponent field = (JTextComponent) event.getSource();
        String value = field.getText();
        TuttiUIUtil.setProperty(getModel(), property, value);
    }

    public void setBoolean(ItemEvent event, String property) {
        boolean value = event.getStateChange() == ItemEvent.SELECTED;
        TuttiUIUtil.setProperty(getModel(), property, value);
    }

    public void setDate(ActionEvent event, String property) {
        JXDatePicker field = (JXDatePicker) event.getSource();
        Date value = field.getDate();
        TuttiUIUtil.setProperty(getModel(), property, value);
    }

    public void selectListData(ListSelectionEvent event, String property) {
        if (!event.getValueIsAdjusting()) {
            JList list = (JList) event.getSource();
            ListSelectionModel selectionModel = list.getSelectionModel();

            selectionModel.setValueIsAdjusting(true);
            try {
                List selectedList = Lists.newLinkedList();

                for (int index : list.getSelectedIndices()) {
                    Object o = list.getModel().getElementAt(index);
                    selectedList.add(o);
                }
                TuttiUIUtil.setProperty(getModel(), property, selectedList);
            } finally {
                selectionModel.setValueIsAdjusting(false);
            }
        }
    }

    public void openDialog(TuttiUI dialogContent,
                           String title, Dimension dim) {
        Component topestUI = getTopestUI();

        JDialog result;
        if (topestUI instanceof Frame) {
            result = new JDialog((Frame) topestUI, title, true);
        } else {
            result = new JDialog((Dialog) topestUI, title, true);
        }

        result.add((Component) dialogContent);
        result.setResizable(true);

        result.setSize(dim);

        final AbstractTuttiUIHandler handler = dialogContent.getHandler();

        if (handler instanceof Cancelable) {

            // add a auto-close action
            JRootPane rootPane = result.getRootPane();

            KeyStroke shortcutClosePopup = getConfig().getShortcutClosePopup();

            rootPane.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(
                    shortcutClosePopup, "close");
            rootPane.getActionMap().put("close", new AbstractAction() {
                private static final long serialVersionUID = 1L;

                @Override
                public void actionPerformed(ActionEvent e) {
                    ((Cancelable) handler).cancel();
                }
            });
        }

        result.addWindowListener(new WindowAdapter() {

            @Override
            public void windowClosed(WindowEvent e) {
                Component ui = (Component) e.getSource();
                if (log.isInfoEnabled()) {
                    log.info("Destroy ui " + ui);
                }
                JAXXUtil.destroy(ui);
            }
        });
        SwingUtil.center(getContext().getMainUI(), result);
        result.setVisible(true);
    }

    public void closeDialog(TuttiUI ui) {
        SwingUtil.getParentContainer(ui, JDialog.class).setVisible(false);
    }

    public static final String CONFIRMATION_FORMAT = "<html>%s<hr/><br/>%s</html>";

    public static final String WARNING_FORMAT = "<html>%s<hr/></html>";

    public int askSaveBeforeLeaving(String message) {
        String htmlMessage = String.format(
                CONFIRMATION_FORMAT,
                message,
                _("tutti.common.askSaveBeforeLeaving.help"));
        int result = JOptionPane.showConfirmDialog(
                getTopestUI(),
                htmlMessage,
                _("tutti.common.askSaveBeforeLeaving.title"),
                JOptionPane.YES_NO_CANCEL_OPTION,
                JOptionPane.QUESTION_MESSAGE);
        return result;
    }

    public boolean askCancelEditBeforeLeaving(String message) {
        String htmlMessage = String.format(
                CONFIRMATION_FORMAT,
                message,
                _("tutti.common.askCancelEditBeforeLeaving.help"));
        int i = JOptionPane.showConfirmDialog(
                getTopestUI(),
                htmlMessage,
                _("tutti.common.askCancelEditBeforeLeaving.title"),
                JOptionPane.OK_CANCEL_OPTION,
                JOptionPane.QUESTION_MESSAGE);

        boolean result = i == JOptionPane.OK_OPTION;
        return result;
    }

    public boolean askOverwriteFile(File file) {
        boolean result;
        if (file.exists()) {

            // file exists ask user to overwrite
            String htmlMessage = String.format(
                    CONFIRMATION_FORMAT,
                    _("tutti.common.askOverwriteFile.message", file),
                    _("tutti.common.askOverwriteFile.help"));

            result = JOptionPane.showConfirmDialog(
                    getTopestUI(),
                    htmlMessage,
                    _("tutti.common.askOverwriteFile.title"),
                    JOptionPane.OK_CANCEL_OPTION,
                    JOptionPane.QUESTION_MESSAGE) == JOptionPane.OK_OPTION;
        } else {

            // file does not exist
            result = true;
        }
        return result;
    }

    protected Component getTopestUI() {
        Component result;
        TuttiActionUI actionUI = getContext().getActionUI();
        if (actionUI.isVisible()) {
            result = actionUI;
        } else {
            result = getContext().getMainUI();
        }
        return result;
    }
    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected void initUI(TuttiUI ui) {

        for (Map.Entry<String, Object> entry : ui.get$objectMap().entrySet()) {
            Object component = entry.getValue();
            if (component instanceof NumberEditor) {
                initNumberEditor((NumberEditor) component);

            } else if (component instanceof JXDatePicker) {
                initDatePicker((JXDatePicker) component);

            } else if (component instanceof SimpleTimeEditor) {
                initTimeEditor((SimpleTimeEditor) component);

            } else if (component instanceof ButtonAttachment) {
                initButtonAttachment((ButtonAttachment) component);

            } else if (component instanceof JLabel) {
                JLabel jLabel = (JLabel) component;
                Boolean strongStyle = (Boolean) jLabel.getClientProperty("strongStyle");
                Boolean italicStyle = (Boolean) jLabel.getClientProperty("italicStyle");
                boolean addHtml = strongStyle != null && strongStyle || italicStyle != null && italicStyle;
                if (addHtml) {
                    String text = jLabel.getText();
                    if (strongStyle != null && strongStyle) {
                        text = "<strong>" + text + "</strong>";
                    }
                    if (italicStyle != null && italicStyle) {
                        text = "<em>" + text + "</em>";
                    }
                    jLabel.setText("<html>" + text + "</html>");
                }

            } else if (component instanceof JTextField) {
                JTextField jTextField = (JTextField) component;
                Boolean computed = (Boolean) jTextField.getClientProperty("computed");
                if (computed != null && computed) {
                    Font font = jTextField.getFont().deriveFont(Font.ITALIC);
                    jTextField.setFont(font);
                    jTextField.setEditable(!computed);
                    jTextField.setEnabled(!computed);
                    jTextField.setDisabledTextColor(getConfig().getColorComputedWeights());
                }

            } else if (component instanceof AbstractButton) {
                AbstractButton abstractButton = (AbstractButton) component;
                Class<AbstractTuttiAction> actionName = (Class<AbstractTuttiAction>) abstractButton.getClientProperty("tuttiAction");
                if (actionName != null) {
                    initAction(abstractButton, actionName);
                }
            }
        }
    }

    protected void initButtonAttachment(ButtonAttachment component) {

        component.init();
    }

    protected <A extends AbstractTuttiAction> void initAction(AbstractButton abstractButton,
                                                              Class<A> actionName) {

        Action action = createUIAction(abstractButton, actionName);
        abstractButton.setAction(action);
    }

    public <A extends AbstractTuttiAction> TuttiUIAction<A> createUIAction(AbstractButton abstractButton,
                                                                           Class<A> actionName) {
        try {

            // create logic action
            A logicAction = createLogicAction(actionName);

            // create ui action
            TuttiUIAction<A> result = new TuttiUIAction<A>(abstractButton,
                                                           logicAction);
            return result;
        } catch (Exception e) {
            throw new TuttiTechnicalException(
                    "Could not instanciate action " + actionName, e);
        }

    }

    public <A extends AbstractTuttiAction> A createLogicAction(Class<A> actionName) {
        try {

            AbstractTuttiUIHandler handler = this;

            if (AbstractMainUITuttiAction.class.isAssignableFrom(actionName) &&
                getContext().getMainUI() != null) {
                handler = getContext().getMainUI().getHandler();
            }

            // create action
            A result = ConstructorUtils.invokeConstructor(actionName, handler);
            return result;
        } catch (Exception e) {
            throw new RuntimeException(
                    "Could not instanciate action " + actionName, e);
        }
    }

    public <A extends AbstractTuttiAction> A getLogicAction(AbstractButton b) {
        Action action = b.getAction();
        Preconditions.checkNotNull(action);
        Preconditions.checkState(action instanceof TuttiUIAction);
        return ((TuttiUIAction<A>) action).getLogicAction();
    }

    protected void doAction(AbstractButton button, ActionEvent event) {
        button.getAction().actionPerformed(event);
    }

    protected boolean quitScreen(boolean modelIsValid,
                                 boolean modelIsModify,
                                 String askGiveUpMessage,
                                 String askSaveMessage,
                                 Action saveAction) {
        boolean result;

        if (!modelIsValid) {

            // model is not valid
            // ask user to qui or not
            result = askCancelEditBeforeLeaving(askGiveUpMessage);

        } else if (modelIsModify) {

            // something is modify ask user what to do
            int answer = askSaveBeforeLeaving(askSaveMessage);
            switch (answer) {
                case JOptionPane.YES_OPTION:

                    // ok save
                    saveAction.actionPerformed(null);
                    result = true;
                    break;
                case JOptionPane.NO_OPTION:

                    // do not save but can still quit the screen (so nothing to do)
                    result = true;
                    break;
                default:
                    // do not save and stay here (so nothing to do)
                    result = false;

            }
        } else {

            // model is valid and not modify, can safely quit screen
            result = true;
        }
        return result;
    }

    protected void registerValidators(SwingValidator... validators) {
        MainUI main = context.getMainUI();
        Preconditions.checkNotNull(
                main, "No mainUI registred in application context");
        MainUIHandler handler = main.getHandler();
        handler.clearValidators();
        for (SwingValidator validator : validators) {
            handler.registerValidator(validator);
        }
    }

    public void clearValidators() {
        MainUI main = context.getMainUI();
        Preconditions.checkNotNull(
                main, "No mainUI registred in application context");
        MainUIHandler handler = main.getHandler();
        handler.clearValidators();
    }

    /**
     * Prépare un component de choix d'entités pour un type d'entité donné et
     * pour un service de persistance donné.
     *
     * @param comboBox le component graphique à initialiser
     */
    protected <E extends Serializable> void initBeanFilterableComboBox(
            BeanFilterableComboBox<E> comboBox,
            List<E> data,
            E selectedData) {

        initBeanFilterableComboBox(comboBox, data, selectedData, null);
    }

    protected <E extends Serializable> void initBeanFilterableComboBox(
            BeanFilterableComboBox<E> comboBox,
            List<E> data,
            E selectedData,
            String decoratorContext) {

        Preconditions.checkNotNull(comboBox, "No comboBox!");

        Class<E> beanType = comboBox.getBeanType();

        Preconditions.checkNotNull(beanType, "No beanType on the combobox!");

        Decorator<E> decorator = getDecorator(beanType, decoratorContext);

        if (data == null) {
            data = Lists.newArrayList();
        }

        if (log.isInfoEnabled()) {
            log.info("entity comboBox list [" + beanType.getName() + "] : " +
                     (data == null ? 0 : data.size()));
        }

        // add data list to combo box
        comboBox.init((JXPathDecorator<E>) decorator, data);

        comboBox.setSelectedItem(selectedData);

        if (log.isDebugEnabled()) {
            log.debug("combo [" + beanType.getName() + "] : " +
                      comboBox.getData().size());
        }
    }

    /**
     * Prépare un component de choix d'entités pour un type d'entité donné et
     * pour un service de persistance donné.
     *
     * @param comboBox le component graphique à initialiser
     */
    protected <E extends Serializable> void initBeanComboBox(
            BeanComboBox<E> comboBox,
            List<E> data,
            E selectedData) {

        initBeanComboBox(comboBox, data, selectedData, null);
    }

    protected <E extends Serializable> void initBeanComboBox(
            BeanComboBox<E> comboBox,
            List<E> data,
            E selectedData,
            String decoratorContext) {

        Preconditions.checkNotNull(comboBox, "No comboBox!");

        Class<E> beanType = comboBox.getBeanType();

        Preconditions.checkNotNull(beanType, "No beanType on the combobox!");

        Decorator<E> decorator = getDecorator(beanType, decoratorContext);

        if (data == null) {
            data = Lists.newArrayList();
        }

        if (log.isInfoEnabled()) {
            log.info("entity comboBox list [" + beanType.getName() + "] : " +
                     (data == null ? 0 : data.size()));
        }

        // add data list to combo box
        comboBox.init((JXPathDecorator<E>) decorator, data);

        comboBox.setSelectedItem(selectedData);

        if (log.isDebugEnabled()) {
            log.debug("combo [" + beanType.getName() + "] : " +
                      comboBox.getData().size());
        }
    }

    /**
     * Prépare un component de choix d'entités pour un type d'entité donné et
     * pour un service de persistance donné.
     *
     * @param list         le component graphique à initialiser
     * @param data         la liste des données à mettre dans la liste de gauche
     * @param selectedData la liste des données à mettre dans la liste de droite
     */
    protected <E extends IdAware> void initBeanList(
            BeanDoubleList<E> list,
            List<E> data,
            List<E> selectedData) {

        Preconditions.checkNotNull(list, "No list!");

        Class<E> beanType = list.getBeanType();
        Preconditions.checkNotNull(beanType, "No beanType on the double list!");

        DecoratorService decoratorService =
                context.getDecoratorService();
        Decorator<E> decorator = decoratorService.getDecoratorByType(beanType);

        if (log.isInfoEnabled()) {
            log.info("entity list [" + beanType.getName() + "] : " +
                     (data == null ? 0 : data.size()));
        }

        // add data list to combo box
        list.init((JXPathDecorator<E>) decorator, data, selectedData);

        if (log.isDebugEnabled()) {
            log.debug("Jlist [" + beanType.getName() + "] : " +
                      list.getUniverseList().getModel().getSize());
        }
    }

    protected void initNumberEditor(NumberEditor editor) {
        if (log.isDebugEnabled()) {
            log.debug("init number editor " + editor.getName());
        }
        editor.init();

        // Force binding if value is already in model
        Number model = editor.getModel();
        if (model != null) {
            editor.setModel(null);
            editor.setModel(model);
        }
    }

    protected void initTimeEditor(SimpleTimeEditor editor) {
        if (log.isDebugEnabled()) {
            log.debug("init time editor " + editor.getName() +
                      " for property " + editor.getModel().getProperty());
        }
        editor.init();
    }

    protected void initDatePicker(final JXDatePicker picker) {

        if (log.isDebugEnabled()) {
            log.debug("disable JXDatePicker editor" + picker.getName());
        }
        String dateFormat = getConfig().getDateFormat();
        picker.setFormats(dateFormat);
        picker.getEditor().addFocusListener(new FocusAdapter() {

            @Override
            public void focusLost(FocusEvent e) {
                try {
                    picker.commitEdit();

                } catch (ParseException ex) {
                    if (log.isDebugEnabled()) {
                        log.debug("format error", ex);
                    }
                }
            }

        });
    }

    public <O> Decorator<O> getDecorator(Class<O> type, String name) {
        DecoratorService decoratorService =
                context.getDecoratorService();

        Preconditions.checkNotNull(type);

        Decorator decorator = decoratorService.getDecoratorByType(type, name);
        if (decorator == null) {

            if (DecoratorService.LabelAware.class.isAssignableFrom(type)) {
                decorator = getDecorator(DecoratorService.LabelAware.class, null);
            }
        }
        Preconditions.checkNotNull(decorator);
        return decorator;
    }

    protected String decorate(Object object) {
        return decorate(object, null);
    }

    protected String decorate(Object object, String context) {
        String result = "";
        if (object != null) {
            getDecorator(object.getClass(), context).toString(object);
        }
        return result;
    }

    protected <O> ListCellRenderer newListCellRender(Class<O> type) {

        return newListCellRender(type, null);
    }

    protected <O> ListCellRenderer newListCellRender(Class<O> type, String name) {

        Decorator<O> decorator = getDecorator(type, name);
        return newListCellRender(decorator);
    }

    protected <O> ListCellRenderer newListCellRender(Decorator<O> decorator) {

        Preconditions.checkNotNull(decorator);

        ListCellRenderer result = new DecoratorListCellRenderer(decorator);
        return result;
    }

    protected void listenValidatorValid(SimpleBeanValidator validator,
                                        final AbstractTuttiBeanUIModel model) {
        validator.addPropertyChangeListener(SimpleBeanValidator.VALID_PROPERTY, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                if (log.isDebugEnabled()) {
                    log.debug("Model [" + model +
                              "] pass to valid state [" +
                              evt.getNewValue() + "]");
                }
                model.setValid((Boolean) evt.getNewValue());
            }
        });
    }

    protected void listModelIsModify(AbstractTuttiBeanUIModel model) {
        model.addPropertyChangeListener(new PropertyChangeListener() {

            final Set<String> excludeProperties = getPropertiesToIgnore();

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                if (!excludeProperties.contains(evt.getPropertyName())) {
                    ((AbstractTuttiBeanUIModel) evt.getSource()).setModify(true);
                }
            }
        });
    }

    protected Set<String> getPropertiesToIgnore() {
        return Sets.newHashSet(
                AbstractTuttiBeanUIModel.PROPERTY_MODIFY,
                AbstractTuttiBeanUIModel.PROPERTY_VALID);
    }

//    public <B> void selectFirstInCombo(BeanFilterableComboBox<B> combo) {
//        List<B> data = combo.getData();
//        B selectedItem = null;
//        if (CollectionUtils.isNotEmpty(data)) {
//            selectedItem = data.get(0);
//        }
//        combo.setSelectedItem(selectedItem);
//    }

    protected void closeUI(TuttiUI ui) {
        ui.getHandler().onCloseUI();
    }

    protected <B> void changeValidatorContext(String newContext,
                                              SwingValidator<B> validator) {
        B bean = validator.getBean();
        validator.setContext(newContext);
        validator.setBean(bean);
    }
}
