package fr.ifremer.tutti.ui.swing.content.operation.catches.benthos;

/*
 * #%L
 * Tutti :: UI
 * $Id: BenthosBatchUIHandler.java 602 2013-03-13 09:20:00Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.1/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/catches/benthos/BenthosBatchUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Lists;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.data.Attachment;
import fr.ifremer.tutti.persistence.entities.data.FishingOperation;
import fr.ifremer.tutti.persistence.entities.data.SpeciesBatch;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.ui.swing.content.operation.AbstractTuttiBatchTableUIHandler;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUIModel;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import fr.ifremer.tutti.ui.swing.util.TuttiUI;
import fr.ifremer.tutti.ui.swing.util.attachment.AttachmentCellEditor;
import fr.ifremer.tutti.ui.swing.util.attachment.AttachmentCellRenderer;
import fr.ifremer.tutti.ui.swing.util.comment.CommentCellEditor;
import fr.ifremer.tutti.ui.swing.util.comment.CommentCellRenderer;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.table.DefaultTableColumnModelExt;
import org.nuiton.util.decorator.Decorator;

import java.util.List;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class BenthosBatchUIHandler extends AbstractTuttiBatchTableUIHandler<BenthosBatchRowModel, BenthosBatchUIModel, BenthosBatchUI> {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(BenthosBatchUIHandler.class);

    public BenthosBatchUIHandler(TuttiUI<?, ?> parentUi,
                                 BenthosBatchUI ui) {
        super(parentUi, ui,
              BenthosBatchRowModel.PROPERTY_SPECIES_TO_CONFIRM,
              BenthosBatchRowModel.PROPERTY_SPECIES,
              BenthosBatchRowModel.PROPERTY_WEIGHT,
              BenthosBatchRowModel.PROPERTY_NUMBER,
              BenthosBatchRowModel.PROPERTY_SAMPLE_WEIGHT,
              BenthosBatchRowModel.PROPERTY_COMMENT);
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiBatchTableUIHandler methods                           --//
    //------------------------------------------------------------------------//

    @Override
    public void selectFishingOperation(FishingOperation bean) {

        boolean empty = bean == null;

        BenthosBatchUIModel model = getModel();

        List<BenthosBatchRowModel> rows;

        if (empty) {
            rows = null;
        } else {

            if (log.isInfoEnabled()) {
                log.info("Get benthos batch for fishingOperation: " +
                         bean.getId());
            }
            rows = Lists.newArrayList();

            if (!TuttiEntities.isNew(bean)) {
                List<SpeciesBatch> catches =
                        persistenceService.getAllRootBenthosBatch(bean.getId());
                for (SpeciesBatch aBatch : catches) {
                    BenthosBatchRowModel entry =
                            new BenthosBatchRowModel(aBatch);
                    rows.add(entry);
                }
            }
        }
        model.setRows(rows);
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiTableUIHandler methods                                --//
    //------------------------------------------------------------------------//

    @Override
    public BenthosBatchTableModel getTableModel() {
        return (BenthosBatchTableModel) getTable().getModel();
    }

    @Override
    public JXTable getTable() {
        return ui.getTable();
    }

    @Override
    protected boolean isRowValid(BenthosBatchRowModel row) {
        boolean result = row.getSpecies() != null && row.getWeight() != null;
        return result;
    }

    @Override
    protected void onRowModified(int rowIndex,
                                 BenthosBatchRowModel row,
                                 String propertyName,
                                 Object oldValue,
                                 Object newValue) {
        recomputeRowValidState(row);
    }

    @Override
    protected void saveSelectedRowIfRequired(TuttiBeanMonitor<BenthosBatchRowModel> rowMonitor,
                                             BenthosBatchRowModel row) {

        if (row.isValid()) {
            // there is a valid bean attached to the monitor

            if (rowMonitor.wasModified()) {

                // monitored bean was modified, save it
                if (log.isInfoEnabled()) {
                    log.info("Row " + row + " was modified, will save it");
                }

                saveRow(row);

                // clear modified flag on the monitor
                rowMonitor.clearModified();
            }
        } else {

            // row is not valid can not save it

            SpeciesBatch catchBean = row.toBean();

            if (!TuttiEntities.isNew(catchBean)) {

                // remove this
                persistenceService.deleteBenthosBatch(catchBean.getId());
            }
        }
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    public SwingValidator<BenthosBatchUIModel> getValidator() {
        return ui.getValidator();
    }

    @Override
    public void beforeInitUI() {

        if (log.isInfoEnabled()) {
            log.info("beforeInit: " + ui);
        }

        EditCatchesUIModel catchesUIModel =
                ui.getContextValue(EditCatchesUIModel.class);

        BenthosBatchUIModel model = new BenthosBatchUIModel(catchesUIModel);
        ui.setContextValue(model);
    }

    @Override
    public void afterInitUI() {

        if (log.isInfoEnabled()) {
            log.info("afterInit: " + ui);
        }

        initUI(ui);

        JXTable table = getTable();

        // create table column model

        DefaultTableColumnModelExt columnModel =
                new DefaultTableColumnModelExt();

        List<Species> allSpecies = persistenceService.getAllSpecies();

        {
            // Species (by code) column

            Decorator<Species> decorator = getDecorator(
                    Species.class, null);

            addComboDataColumnToModel(columnModel,
                                      BenthosBatchTableModel.SPECIES_BY_CODE,
                                      decorator, allSpecies);
        }

        {
            // Species (by genusCode) column

            Decorator<Species> decorator = getDecorator(
                    Species.class, null);

            addComboDataColumnToModel(columnModel,
                                      BenthosBatchTableModel.SPECIES_BY_GENUS_CODE,
                                      decorator, allSpecies);

        }

        { // Weight column

            addFloatColumnToModel(columnModel,
                                  BenthosBatchTableModel.WEIGHT,
                                  TuttiUI.DECIMAL3_PATTERN);
        }

        { // Number column

            addIntegerColumnToModel(columnModel,
                                    BenthosBatchTableModel.NUMBER,
                                    TuttiUI.INT_3_DIGITS_PATTERN);
        }

        { // Sample weight column

            addFloatColumnToModel(columnModel,
                                  BenthosBatchTableModel.SAMPLE_WEIGHT,
                                  TuttiUI.DECIMAL3_PATTERN);
        }

        { // Comment column

            addColumnToModel(columnModel,
                             CommentCellEditor.newEditor(ui),
                             CommentCellRenderer.newRender(),
                             BenthosBatchTableModel.COMMENT);
        }

        { // File column

            addColumnToModel(columnModel,
                             AttachmentCellEditor.newEditor(ui),
                             AttachmentCellRenderer.newRender(getDecorator(Attachment.class, null)),
                             BenthosBatchTableModel.ATTACHMENT);
        }

        { // Species to confirm column

            addBooleanColumnToModel(columnModel,
                                    BenthosBatchTableModel.SPECIES_TO_CONFIRM,
                                    getTable());
        }

        // create table model
        BenthosBatchTableModel tableModel =
                new BenthosBatchTableModel(columnModel);

        table.setModel(tableModel);
        table.setColumnModel(columnModel);

        initBatchTable(table, columnModel, tableModel);
    }

    @Override
    public void onCloseUI() {
        if (log.isDebugEnabled()) {
            log.debug("closing: " + ui);
        }
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected void saveRow(BenthosBatchRowModel row) {

        SpeciesBatch catchBean = row.toBean();

        FishingOperation fishingOperation = getModel().getFishingOperation();
        catchBean.setFishingOperation(fishingOperation);
        if (log.isInfoEnabled()) {
            log.info("Selected fishingOperation: " + fishingOperation.getId());
        }

        if (TuttiEntities.isNew(catchBean)) {

            BenthosBatchRowModel batchParent = row.getParentBatch();
            String parentBatchId = null;

            if (batchParent != null) {
                parentBatchId = batchParent.getId();
            }

            if (log.isInfoEnabled()) {
                log.info("Persist new benthos batch with parentId: " +
                         parentBatchId);
            }
            catchBean = persistenceService.createBenthosBatch(catchBean, parentBatchId);
            row.setId(catchBean.getId());
        } else {
            persistenceService.saveBenthosBatch(catchBean);
        }
    }

}
