package fr.ifremer.tutti.ui.swing.content.operation.catches.benthos;

/*
 * #%L
 * Tutti :: UI
 * $Id: BenthosBatchRowModel.java 602 2013-03-13 09:20:00Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.1/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/catches/benthos/BenthosBatchRowModel.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Lists;
import fr.ifremer.tutti.persistence.entities.data.AttachementObjectTypeEnum;
import fr.ifremer.tutti.persistence.entities.data.Attachment;
import fr.ifremer.tutti.persistence.entities.data.SpeciesBatch;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiBeanUIModel;
import fr.ifremer.tutti.ui.swing.util.attachment.AttachmentModelAware;
import fr.ifremer.tutti.ui.swing.util.comment.CommentModelAware;
import org.apache.commons.collections.CollectionUtils;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.util.Collection;
import java.util.List;

/**
 * Define a benthos batch row.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.2
 */
public class BenthosBatchRowModel extends AbstractTuttiBeanUIModel<SpeciesBatch, BenthosBatchRowModel> implements AttachmentModelAware, CommentModelAware {

    private static final long serialVersionUID = 1L;

    public static final String PROPERTY_SPECIES = "species";

    public static final String PROPERTY_SPECIES_TO_CONFIRM = "speciesToConfirm";

    public static final String PROPERTY_WEIGHT = "weight";

    public static final String PROPERTY_NUMBER = "number";

    public static final String PROPERTY_SAMPLE_WEIGHT = "sampleWeight";

//    public static final String PROPERTY_ATTACHMENTS = "attachments";

    public static final String PROPERTY_PARENT_BATCH = "parentBatch";

    public static final String PROPERTY_CHILD_BATCH = "childBatch";

    public static final String PROPERTY_BATCH_LEAF = "batchLeaf";

    public static final String PROPERTY_BATCH_ROOT = "batchRoot";


    /**
     * Species observed.
     *
     * @since 0.2
     */
    protected Species species;

    /**
     * Is the species need to be confirmed?.
     *
     * @since 0.2
     */
    protected boolean speciesToConfirm;

    /**
     * Observed weight.
     *
     * @since 0.2
     */
    protected Float weight;

    /**
     * Observed number.
     *
     * @since 0.3
     */
    protected Integer number;

    /**
     * Sample weight.
     *
     * @since 0.2
     */
    protected Float sampleWeight;


    /**
     * Comment.
     *
     * @since 0.2
     */
    protected String comment;

    /**
     * Attachments (should never be null).
     *
     * @since 0.2
     */
    protected final List<Attachment> attachment = Lists.newArrayList();

    /**
     * Parent of this batch (can be null if batch is root).
     *
     * @see #isBatchRoot()
     * @since 0.3
     */
    protected BenthosBatchRowModel parentBatch;

    /**
     * List of child batches (can be null or empty if batch is a leaf).
     *
     * @see #isBatchLeaf()
     * @since 0.3
     */
    protected List<BenthosBatchRowModel> childBatch;

    protected static final Binder<SpeciesBatch, BenthosBatchRowModel> fromBeanBinder =
            BinderFactory.newBinder(SpeciesBatch.class,
                                    BenthosBatchRowModel.class);

    protected static final Binder<BenthosBatchRowModel, SpeciesBatch> toBeanBinder =
            BinderFactory.newBinder(BenthosBatchRowModel.class,
                                    SpeciesBatch.class);

    public BenthosBatchRowModel() {
        super(SpeciesBatch.class, fromBeanBinder, toBeanBinder);
    }

    public BenthosBatchRowModel(SpeciesBatch aBatch) {
        this();
        fromBean(aBatch);
    }

    public Species getSpecies() {
        return species;
    }

    public void setSpecies(Species species) {
        Object oldValue = getSpecies();
        this.species = species;
        firePropertyChange(PROPERTY_SPECIES, oldValue, species);
    }

    public Boolean getSpeciesToConfirm() {
        return speciesToConfirm;
    }

    public void setSpeciesToConfirm(Boolean speciesToConfirm) {
        Object oldValue = getSpeciesToConfirm();
        this.speciesToConfirm = speciesToConfirm;
        firePropertyChange(PROPERTY_SPECIES_TO_CONFIRM, oldValue, speciesToConfirm);
    }

    public Float getWeight() {
        return weight;
    }

    public void setWeight(Float weight) {
        Object oldValue = getWeight();
        this.weight = weight;
        firePropertyChange(PROPERTY_WEIGHT, oldValue, weight);
    }

    public Float getSampleWeight() {
        return sampleWeight;
    }

    public void setSampleWeight(Float sampleWeight) {
        Object oldValue = getSampleWeight();
        this.sampleWeight = sampleWeight;
        firePropertyChange(PROPERTY_SAMPLE_WEIGHT, oldValue, sampleWeight);
    }

    //------------------------------------------------------------------------//
    //-- Navigation properties                                              --//
    //------------------------------------------------------------------------//

    public BenthosBatchRowModel getParentBatch() {
        return parentBatch;
    }

    public void setParentBatch(BenthosBatchRowModel parentBatch) {
        Object oldValue = getParentBatch();
        this.parentBatch = parentBatch;
        firePropertyChange(PROPERTY_PARENT_BATCH, oldValue, parentBatch);
        firePropertyChange(PROPERTY_BATCH_ROOT, null, isBatchRoot());
    }

    public List<BenthosBatchRowModel> getChildBatch() {
        return childBatch;
    }

    public void setChildBatch(List<BenthosBatchRowModel> childBatch) {
        this.childBatch = childBatch;
        // force to propagate child changes
        firePropertyChange(PROPERTY_CHILD_BATCH, null, childBatch);
        firePropertyChange(PROPERTY_BATCH_LEAF, null, isBatchLeaf());
    }

    public boolean isBatchLeaf() {
        return CollectionUtils.isEmpty(childBatch);
    }

    public boolean isBatchRoot() {
        return parentBatch == null;
    }

    public Integer getNumber() {
        return number;
    }

    public void setNumber(Integer number) {
        Object oldValue = getNumber();
        this.number = number;
        firePropertyChange(PROPERTY_NUMBER, oldValue, number);
    }

    //------------------------------------------------------------------------//
    //-- CommentModelAware                                                  --//
    //------------------------------------------------------------------------//

    @Override
    public String getComment() {
        return comment;
    }

    @Override
    public void setComment(String comment) {
        Object oldValue = getComment();
        this.comment = comment;
        firePropertyChange(PROPERTY_COMMENT, oldValue, comment);
    }

    //------------------------------------------------------------------------//
    //-- AttachmentModelAware                                               --//
    //------------------------------------------------------------------------//

    @Override
    public AttachementObjectTypeEnum getObjectType() {
        return AttachementObjectTypeEnum.SAMPLE;
    }

    @Override
    public String getObjectId() {
        return getId();
    }

    @Override
    public List<Attachment> getAttachment() {
        return attachment;
    }

    @Override
    public void addAllAttachment(Collection<Attachment> attachments) {
        this.attachment.addAll(attachments);
        firePropertyChange(PROPERTY_ATTACHMENT, null, getAttachment());
    }

    @Override
    public void addAttachment(Attachment attachment) {
        this.attachment.add(attachment);
        firePropertyChange(PROPERTY_ATTACHMENT, null, getAttachment());
    }

    @Override
    public void removeAllAttachment(Collection<Attachment> attachments) {
        this.attachment.removeAll(attachments);
        firePropertyChange(PROPERTY_ATTACHMENT, null, getAttachment());
    }

    @Override
    public void removeAttachment(Attachment attachment) {
        this.attachment.remove(attachment);
        firePropertyChange(PROPERTY_ATTACHMENT, null, getAttachment());
    }

}
