package fr.ifremer.tutti.ui.swing.content.protocol;

/*
 * #%L
 * Tutti :: UI
 * $Id: EditProtocolUIHandler.java 438 2013-02-18 07:40:56Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.0/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/protocol/EditProtocolUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.ezware.oxbow.swingbits.util.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.protocol.SpeciesProtocol;
import fr.ifremer.tutti.persistence.entities.protocol.TuttiProtocol;
import fr.ifremer.tutti.persistence.entities.referential.Caracteristic;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.ui.swing.MainUIHandler;
import fr.ifremer.tutti.ui.swing.TuttiScreen;
import fr.ifremer.tutti.ui.swing.TuttiUI;
import fr.ifremer.tutti.ui.swing.content.home.ImportProtocolAction;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import fr.ifremer.tutti.ui.swing.util.table.AbstractTuttiTableUIHandler;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.swing.editor.bean.BeanDoubleList;
import jaxx.runtime.swing.editor.bean.BeanDoubleListModel;
import jaxx.runtime.swing.editor.bean.BeanUIUtil;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.autocomplete.ComboBoxCellEditor;
import org.jdesktop.swingx.autocomplete.ObjectToStringConverter;
import org.jdesktop.swingx.table.DefaultTableColumnModelExt;
import org.nuiton.util.decorator.Decorator;

import javax.swing.JComboBox;
import javax.swing.JOptionPane;
import javax.swing.table.TableColumnModel;
import java.awt.event.ActionEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import static org.nuiton.i18n.I18n._;


/**
 * @author tchemit <chemit@codelutin.com>
 * @since 0.3
 */
public class EditProtocolUIHandler extends AbstractTuttiTableUIHandler<EditProtocolSpeciesRowModel, EditProtocolUIModel, EditProtocolUI> {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(EditProtocolUIHandler.class);

    public EditProtocolUIHandler(TuttiUI parentUi, EditProtocolUI ui) {
        super(parentUi.getHandler().getContext(), ui);
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiTableUIHandler methods                                --//
    //------------------------------------------------------------------------//

    @Override
    public EditProtocolSpeciesTableModel getTableModel() {
        return (EditProtocolSpeciesTableModel) getTable().getModel();
    }

    @Override
    public JXTable getTable() {
        return ui.getSpeciesTable();
    }

    @Override
    protected boolean isRowValid(EditProtocolSpeciesRowModel row) {
        return true;
    }

    @Override
    protected void onRowModified(int rowIndex,
                                 EditProtocolSpeciesRowModel row,
                                 String propertyName,
                                 Object oldValue,
                                 Object newValue) {

        recomputeRowValidState(row);
    }


    @Override
    protected void onRowValidStateChanged(int rowIndex,
                                          EditProtocolSpeciesRowModel row,
                                          Boolean oldValue,
                                          Boolean newValue) {
        super.onRowValidStateChanged(rowIndex, row, oldValue, newValue);

        if (row != null && row.isModify() && newValue != null && newValue) {
            // row was modified and is valid, we can save protocol
            // even if after the row becomes again not valid, this is not a
            // problem since we will only save valid data!
            getModel().setModify(true);
        }
    }

    @Override
    protected void saveSelectedRowIfRequired(TuttiBeanMonitor<EditProtocolSpeciesRowModel> rowMonitor,
                                             EditProtocolSpeciesRowModel row) {

        // nothing to save when row changes, model is marked to be save
        // when a row has changed and is valid or other protocol properties are modified
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    public SwingValidator<EditProtocolUIModel> getValidator() {
        return ui.getValidator();
    }

    @Override
    public void beforeInitUI() {

        EditProtocolUIModel model = new EditProtocolUIModel();

        // load cache data

        List<Species> allSpecies = Lists.newArrayList(persistenceService.getAllSpecies());
        model.setAllSpecies(allSpecies);
        Multimap<String, Species> allSpeciesByTaxonId = Multimaps.index(allSpecies,
                                                                        TuttiEntities.GET_TAXON_ID);

        model.setAllSpeciesByTaxonId(allSpeciesByTaxonId);
        Map<String, Species> allReferentSpeciesByTaxonId = TuttiEntities.splitByTaxonId(
                persistenceService.getAllReferentSpecies());

        model.setAllReferentSpeciesByTaxonId(allReferentSpeciesByTaxonId);
        List<TuttiProtocol> protocols = persistenceService.getAllProtocol();
        model.setExistingProtocols(protocols);

        List<Caracteristic> caracteristics = persistenceService.getAllCaracteristic();
        model.setCaracteristics(caracteristics);

        Map<String, Caracteristic> allCaracteristic = TuttiEntities.splitById(caracteristics);
        model.setAllCaracteristic(allCaracteristic);

        // can't load directly model from database here, since we want to
        // fill only the model with rows (transformed from speciesProtocol)
        // As we still don't have the table model at this point, wait the
        // afterUI method to fill model

        listModelIsModify(model);
        ui.setContextValue(model);
    }

    @Override
    public void afterInitUI() {

        initUI(ui);

        EditProtocolUIModel model = getModel();

        TuttiProtocol protocol = ImportProtocolAction.IMPORT_PROTOCOL_ENTRY.getContextValue(ui);

        if (protocol != null) {

            // import a protocol

            ImportProtocolAction.IMPORT_PROTOCOL_ENTRY.removeContextValue(ui);

            model.fromBean(protocol);
            model.setImported(true);
            ui.getSaveWarning().setText(_("tutti.label.protocol.import.warning"));

        } else {

            // load protocol if existing

            if (context.isProtocolFilled()) {

                // load existing protocol
                protocol = persistenceService.getProtocol(context.getProtocolId());
                Boolean mustClone = ui.getContextValue(Boolean.class, MainUIHandler.CLONE_PROTOCOL);

                model.fromBean(protocol);
                if (mustClone != null && mustClone) {
                    ui.setContextValue(false, MainUIHandler.CLONE_PROTOCOL);
                    model.setId(null);
                    model.setCloned(true);

                    ui.getSaveWarning().setText(_("tutti.label.protocol.clone.warning"));

                } else {
                    log.debug(model.getExistingProtocols());
                    model.getExistingProtocols().remove(protocol);
                }

            } else {

                // create new protocol

                if (log.isDebugEnabled()) {
                    log.debug("Will create a new protocol");
                }
            }
        }

        SwingValidator validator = ui.getValidator();
        listenValidatorValid(validator, model);

        registerValidators(validator);

        // create table model

        JXTable table = getTable();

        DefaultTableColumnModelExt columnModel = new DefaultTableColumnModelExt();

        addColumnToModel(columnModel,
                         null,
                         newTableCellRender(Species.class),
                         EditProtocolSpeciesTableModel.SPECIES_ID);

        addColumnToModel(columnModel,
                         null,
                         null,
                         EditProtocolSpeciesTableModel.SURVEY_CODE_ID);

        addLengthClassesColumnToModel(columnModel, model.getLengthClassesPmfmId());

        addBooleanColumnToModel(columnModel, EditProtocolSpeciesTableModel.WEIGHT_ENABLED, table);
        addBooleanColumnToModel(columnModel, EditProtocolSpeciesTableModel.COUNT_IF_NO_FREQUENCY_ENABLED, table);

        addBooleanColumnToModel(columnModel, EditProtocolSpeciesTableModel.SIZE_ENABLED, table);
        addBooleanColumnToModel(columnModel, EditProtocolSpeciesTableModel.SEX_ENABLED, table);
        addBooleanColumnToModel(columnModel, EditProtocolSpeciesTableModel.MATURITY_ENABLED, table);
        addBooleanColumnToModel(columnModel, EditProtocolSpeciesTableModel.AGE_ENABLED, table);

        addBooleanColumnToModel(columnModel, EditProtocolSpeciesTableModel.CALCIFY_SAMPLE_ENABLED, table);

        EditProtocolSpeciesTableModel tableModel =
                new EditProtocolSpeciesTableModel(columnModel);
        table.setModel(tableModel);
        table.setColumnModel(columnModel);

        initTable(table);

        List<Species> speciesList = Lists.newArrayList(model.getAllSpecies());

        List<EditProtocolSpeciesRowModel> rows;

        // build speciesProtocol rows
        if (protocol == null) {
            rows = Lists.newArrayList();
        } else {

            rows = toRows(speciesList, protocol.getSpecies());

            if (log.isDebugEnabled()) {
                log.debug("Will edit protocol with " +
                          rows.size() + " species declared.");
            }
        }

        // set to model ( will propagate to tableModel)
        model.setRows(rows);

        initBeanComboBox(ui.getSpeciesComboBox(), speciesList, null);
        selectFirstInCombo(ui.getSpeciesComboBox());

        initDoubleList(EditProtocolUIModel.PROPERTY_LENGTH_CLASSES_PMFM_ID,
                       ui.getLengthClassesList(),
                       Lists.newArrayList(model.getCaracteristics()),
                       model.getLengthClassesPmfmId());

        initDoubleList(EditProtocolUIModel.PROPERTY_GEAR_PMFM_ID,
                       ui.getGearList(),
                       Lists.newArrayList(model.getCaracteristics()),
                       model.getGearPmfmId());

        initDoubleList(EditProtocolUIModel.PROPERTY_ENVIRONMENT_PMFM_ID,
                       ui.getEnvironmentList(),
                       Lists.newArrayList(model.getCaracteristics()),
                       model.getEnvironmentPmfmId());

        initDoubleList(EditProtocolUIModel.PROPERTY_HYDROLOGY_PMFM_ID,
                       ui.getHydrologyList(),
                       Lists.newArrayList(model.getCaracteristics()),
                       model.getHydrologyPmfmId());

        // if new protocol can already cancel his creation
        model.setModify(model.isCreate());
    }

    public List<EditProtocolSpeciesRowModel> toRows(List<Species> speciesList,
                                                    List<SpeciesProtocol> speciesProtocols) {

        Preconditions.checkNotNull(speciesList);

        EditProtocolUIModel model = getModel();

        Map<String, Species> allReferentSpeciesByTaxonId = model.getAllReferentSpeciesByTaxonId();
        Map<String, Caracteristic> allCaracteristic = model.getAllCaracteristic();

        List<EditProtocolSpeciesRowModel> result = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(speciesProtocols)) {
            for (SpeciesProtocol speciesProtocol : speciesProtocols) {
                Integer taxonId = speciesProtocol.getSpeciesReferenceTaxonId();
                String taxonIdStr = String.valueOf(taxonId);

                // remove all synonyms from available species list
                Collection<Species> allSynonyms = model.getAllSynonyms(taxonIdStr);
                speciesList.removeAll(allSynonyms);

                EditProtocolSpeciesRowModel row = getTableModel().createNewRow();

                // get species referent taxon
                Species species = allReferentSpeciesByTaxonId.get(taxonIdStr);
                row.setSpecies(species);
                row.setLengthStepPmfm(allCaracteristic.get(speciesProtocol.getLengthStepPmfmId()));
                row.fromBean(speciesProtocol);
                result.add(row);
            }
        }

        return result;
    }

    @Override
    public void onCloseUI() {
        if (log.isInfoEnabled()) {
            log.info("closing: " + ui);
        }
        clearValidators();
    }

    @Override
    public boolean canCloseUI(TuttiScreen nextScreen) {
        boolean result = true;
        if (getModel().isModify()) {
            int answer = askSaveBeforeLeaving(_("tutti.dialog.askSaveBeforeLeaving.saveProtocol"));
            result = answer == JOptionPane.NO_OPTION;
            if (answer == JOptionPane.YES_OPTION) {
                ActionEvent event = new ActionEvent(nextScreen, nextScreen.ordinal(), null);
                doAction(ui.getSaveButton(), event);
            }
        }
        return result;
    }

    @Override
    protected void onAfterSelectedRowChanged(int oldRowIndex,
                                             EditProtocolSpeciesRowModel oldRow,
                                             int newRowIndex,
                                             EditProtocolSpeciesRowModel newRow) {

        super.onAfterSelectedRowChanged(oldRowIndex, oldRow, newRowIndex, newRow);
        getModel().setRemoveSpeciesEnabled(newRow != null);
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    public void addDoubleListListeners() {
        String id;
        UpdateSelectedList updateListener;
        EditProtocolUIModel model = getModel();

        id = (String) ui.getLengthClassesList().getClientProperty("_updateListenerId");
        updateListener = (UpdateSelectedList) ui.getLengthClassesList().getClientProperty("_updateListener");
        model.addPropertyChangeListener(id, updateListener);

        id = (String) ui.getEnvironmentList().getClientProperty("_updateListenerId");
        updateListener = (UpdateSelectedList) ui.getEnvironmentList().getClientProperty("_updateListener");
        model.addPropertyChangeListener(id, updateListener);

        id = (String) ui.getGearList().getClientProperty("_updateListenerId");
        updateListener = (UpdateSelectedList) ui.getGearList().getClientProperty("_updateListener");
        model.addPropertyChangeListener(id, updateListener);

        id = (String) ui.getHydrologyList().getClientProperty("_updateListenerId");
        updateListener = (UpdateSelectedList) ui.getHydrologyList().getClientProperty("_updateListener");
        model.addPropertyChangeListener(id, updateListener);
    }

    public void removeDoubleListListeners() {
        String id;
        UpdateSelectedList updateListener;
        EditProtocolUIModel model = getModel();

        id = (String) ui.getLengthClassesList().getClientProperty("_updateListenerId");
        updateListener = (UpdateSelectedList) ui.getLengthClassesList().getClientProperty("_updateListener");

        model.removePropertyChangeListener(id, updateListener);

        id = (String) ui.getEnvironmentList().getClientProperty("_updateListenerId");
        updateListener = (UpdateSelectedList) ui.getEnvironmentList().getClientProperty("_updateListener");
        model.removePropertyChangeListener(id, updateListener);

        id = (String) ui.getGearList().getClientProperty("_updateListenerId");
        updateListener = (UpdateSelectedList) ui.getGearList().getClientProperty("_updateListener");
        model.removePropertyChangeListener(id, updateListener);

        id = (String) ui.getHydrologyList().getClientProperty("_updateListenerId");
        updateListener = (UpdateSelectedList) ui.getHydrologyList().getClientProperty("_updateListener");
        model.removePropertyChangeListener(id, updateListener);
    }

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected void initDoubleList(String propertyId,
                                  BeanDoubleList<Caracteristic> widget,
                                  List<Caracteristic> availableCaracteristics,
                                  List<String> selectedCaracteristics) {

        initBeanList(widget, availableCaracteristics,
                     Lists.<Caracteristic>newArrayList());

        UpdateSelectedList listener = new UpdateSelectedList(
                widget,
                getModel().getAllCaracteristic());
        widget.putClientProperty("_updateListener", listener);
        widget.putClientProperty("_updateListenerId", propertyId);
        listener.select(selectedCaracteristics);
    }

    protected void selectLengthClasses(List<String> ids, JComboBox comboBox) {

        Map<String, Caracteristic> allCaracteristic = getModel().getAllCaracteristic();
        List<Caracteristic> selection = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(ids)) {
            for (String id : ids) {
                selection.add(allCaracteristic.get(id));
            }
        }

        List<Caracteristic> dataToList = Lists.newArrayList(selection);

        // add a null value at first position
        if (!dataToList.isEmpty() && dataToList.get(0) != null) {
            dataToList.add(0, null);
        }
        SwingUtil.fillComboBox(comboBox, dataToList, null);
    }

    protected void addLengthClassesColumnToModel(TableColumnModel model, List<String> selectedIds) {

        Decorator<Caracteristic> decorator =
                getDecorator(Caracteristic.class, null);

        final JComboBox comboBox = new JComboBox();

        getModel().addPropertyChangeListener(EditProtocolUIModel.PROPERTY_LENGTH_CLASSES_PMFM_ID, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                List<String> ids = (List<String>) evt.getNewValue();
                selectLengthClasses(ids, comboBox);
            }
        });
        comboBox.setRenderer(newListCellRender(decorator));

        selectLengthClasses(selectedIds, comboBox);
        ObjectToStringConverter converter = BeanUIUtil.newDecoratedObjectToStringConverter(decorator);
        BeanUIUtil.decorate(comboBox, converter);
        ComboBoxCellEditor editor = new ComboBoxCellEditor(comboBox);


        addColumnToModel(model,
                         editor,
                         newTableCellRender(decorator),
                         EditProtocolSpeciesTableModel.LENGTH_STEP_PMFM_ID);
    }

    protected static class UpdateSelectedList implements PropertyChangeListener {

        private final BeanDoubleListModel<Caracteristic> model;

        private final Map<String, Caracteristic> caracteristicMap;

        public UpdateSelectedList(BeanDoubleList<Caracteristic> doubleList,
                                  Map<String, Caracteristic> caracteristicMap) {
            this.model = doubleList.getModel();
            this.caracteristicMap = caracteristicMap;
        }

        private boolean valueIsAdjusting;

        @Override
        public void propertyChange(PropertyChangeEvent evt) {

            if (!valueIsAdjusting) {

                valueIsAdjusting = true;

                try {
                    List<String> selectedIds = (List<String>) evt.getNewValue();
                    if (log.isInfoEnabled()) {
                        log.info("[" + evt.getPropertyName() + "] selected ids: " + selectedIds);
                    }

                    select(selectedIds);
                } finally {
                    valueIsAdjusting = false;
                }
            }
        }

        public void select(List<String> selectedIds) {

            List<Caracteristic> selection = Lists.newArrayList();
            if (CollectionUtils.isNotEmpty(selectedIds)) {
                for (String selectedId : selectedIds) {
                    selection.add(caracteristicMap.get(selectedId));
                }
            }
            model.setSelected(selection);
        }
    }
}
