package fr.ifremer.tutti.ui.swing;

/*
 * #%L
 * Tutti :: UI
 * $Id: TuttiDataContext.java 513 2013-03-01 20:59:12Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.0.3/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/TuttiDataContext.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import fr.ifremer.tutti.persistence.entities.data.Cruise;
import fr.ifremer.tutti.persistence.entities.data.Program;
import fr.ifremer.tutti.persistence.entities.protocol.TuttiProtocol;
import fr.ifremer.tutti.persistence.entities.referential.Caracteristic;
import fr.ifremer.tutti.persistence.entities.referential.Gear;
import fr.ifremer.tutti.persistence.entities.referential.Person;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.persistence.entities.referential.Vessel;
import fr.ifremer.tutti.service.PersistenceService;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.Closeable;
import java.util.List;

/**
 * Data context of ui.
 * <p/>
 * All shared data must be there to avoid reloading some stuff.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.0.2
 */
public class TuttiDataContext implements Closeable {

    /** Logger. */
    private static final Log log = LogFactory.getLog(TuttiDataContext.class);

    protected Program program;

    protected Cruise cruise;

    protected TuttiProtocol protocol;

    protected List<Caracteristic> caracteristics;

    protected List<Person> persons;

    protected List<Species> species;

    protected List<Species> referentSpecies;

    protected List<Vessel> fishingVessels;

    protected List<Vessel> scientificVessels;

    protected List<Gear> fishingGears;

    protected List<Gear> scientificGears;

    protected PersistenceService service;

    protected final TuttiUIContext uiContext;

    public TuttiDataContext(TuttiUIContext uiContext) {
        this.uiContext = uiContext;
        this.uiContext.addPropertyChangeListener(TuttiUIContext.PROPERTY_PROGRAM_ID, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                program = null;
            }
        });
        this.uiContext.addPropertyChangeListener(TuttiUIContext.PROPERTY_CRUISE_ID, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                cruise = null;
            }
        });
        this.uiContext.addPropertyChangeListener(TuttiUIContext.PROPERTY_PROTOCOL_ID, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                protocol = null;
            }
        });
    }

    public Program getProgram() {
        checkOpened();
        if (program == null) {
            if (uiContext.isProgramFilled()) {
                String id = uiContext.getProgramId();
                if (log.isInfoEnabled()) {
                    log.info("Loading program: " + id);
                }
                program = service.getProgram(id);
            }
        }
        return program;
    }

    public Cruise getCruise() {
        checkOpened();
        if (cruise == null) {
            if (uiContext.isCruiseFilled()) {
                String id = uiContext.getCruiseId();
                if (log.isInfoEnabled()) {
                    log.info("Loading cruise: " + id);
                }
                cruise = service.getCruise(id);
            }
        }
        return cruise;
    }

    public TuttiProtocol getProtocol() {
        checkOpened();
        if (protocol == null) {
            if (uiContext.isProtocolFilled()) {
                String id = uiContext.getProtocolId();
                if (log.isInfoEnabled()) {
                    log.info("Loading cruise: " + id);
                }
                protocol = service.getProtocol(id);
            }
        }
        return protocol;
    }

    public List<Caracteristic> getCaracteristics() {
        if (caracteristics == null) {
            if (log.isInfoEnabled()) {
                log.info("Loading allCaracteristic");
            }
            caracteristics = service.getAllCaracteristic();
        }
        return caracteristics;
    }

    public List<Species> getSpecies() {
        if (species == null) {
            if (log.isInfoEnabled()) {
                log.info("Loading allSpecies");
            }
            species = service.getAllSpecies();
        }
        return species;
    }

    public List<Species> getReferentSpecies() {
        if (referentSpecies == null) {
            if (log.isInfoEnabled()) {
                log.info("Loading allReferentSpecies");
            }
            referentSpecies = service.getAllReferentSpecies();
        }
        return referentSpecies;
    }

    public List<Person> getPersons() {
        checkOpened();
        if (persons == null) {
            if (log.isInfoEnabled()) {
                log.info("Loading allPerson");
            }
            persons = service.getAllPerson();
        }
        return persons;
    }

    public List<Vessel> getFishingVessels() {
        checkOpened();
        if (fishingVessels == null) {
            if (log.isInfoEnabled()) {
                log.info("Loading allfishingVessel");
            }
            fishingVessels = service.getAllFishingVessel();
        }
        return fishingVessels;
    }

    public List<Vessel> getScientificVessels() {
        checkOpened();
        if (scientificVessels == null) {
            if (log.isInfoEnabled()) {
                log.info("Loading allScientificVessel");
            }
            scientificVessels = service.getAllScientificVessel();
        }
        return scientificVessels;
    }

    public List<Gear> getFishingGears() {
        checkOpened();
        if (fishingGears == null) {
            if (log.isInfoEnabled()) {
                log.info("Loading allFishingGear");
            }
            fishingGears = service.getAllFishingGear();
        }
        return fishingGears;
    }

    public List<Gear> getScientificGears() {
        checkOpened();
        if (scientificGears == null) {
            if (log.isInfoEnabled()) {
                log.info("Loading allScientificGear");
            }
            scientificGears = service.getAllScientificGear();
        }
        return scientificGears;
    }

    public void open(PersistenceService persistenceService) {
        close();
        this.service = persistenceService;
    }

    @Override
    public void close() {
        service = null;
        program = null;
        cruise = null;
        program = null;
        resetVessels();
        resetGears();
        resetPersons();
        resetSpecies();
    }

    public void resetGears() {
        fishingGears = null;
        scientificGears = null;
    }

    public void resetVessels() {
        fishingVessels = null;
        scientificVessels = null;
    }

    public void resetPersons() {
        persons = null;
    }

    public void resetSpecies() {
        species = null;
        referentSpecies = null;
    }

    public void resetCaracteristics() {
        caracteristics = null;
    }

    protected void checkOpened() {
        Preconditions.checkState(service != null, "No persistence service assigned!");
    }
}
