package fr.ifremer.tutti.ui.swing.util.action;

/*
 * #%L
 * Tutti :: UI
 * $Id: TuttiUIAction.java 514 2013-03-02 06:25:02Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.0.2/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/util/action/TuttiUIAction.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import fr.ifremer.tutti.ui.swing.content.MainUI;
import fr.ifremer.tutti.ui.swing.util.TuttiExceptionHandler;
import jaxx.runtime.SwingUtil;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTitledPanel;

import javax.swing.AbstractAction;
import javax.swing.ImageIcon;
import javax.swing.SwingUtilities;
import javax.swing.SwingWorker;
import java.awt.Cursor;
import java.awt.event.ActionEvent;
import java.util.Timer;
import java.util.TimerTask;

/**
 * Abstract tutti ui action which launch a {@link AbstractTuttiAction}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.0.2
 */
public class TuttiUIAction<A extends AbstractTuttiAction> extends AbstractAction {

    private static final long serialVersionUID = 1L;

    /** Logger. */
    private static final Log log = LogFactory.getLog(TuttiUIAction.class);

    /**
     * Timer used to launch timerTask (to open load dialog).
     *
     * @since 1.0.2
     */
    private static final Timer t = new Timer();

    /**
     * To deal with errors.
     *
     * @since 1.0.2
     */
    private static final TuttiExceptionHandler exceptionHandler = new TuttiExceptionHandler();

    /**
     * Logic Action to perform.
     *
     * @since 1.0.2
     */
    protected final A action;

    public TuttiUIAction(A action) {

        this.action = action;

        String icon = action.getActionIcon();
        if (icon != null) {
            ImageIcon actionIcon = SwingUtil.createActionIcon(icon);
            putValue(SMALL_ICON, actionIcon);
            putValue(LARGE_ICON_KEY, actionIcon);
        }
        putValue(ACTION_COMMAND_KEY, action.getClass().getName());
        putValue(NAME, action.getActionName());
        putValue(SHORT_DESCRIPTION, action.getActionDescription());
    }

    @Override
    public final void actionPerformed(final ActionEvent event) {

        if (log.isInfoEnabled()) {
            log.info("Task [" + this + "] starting");
        }

        // prepare action
        boolean doAction = action.prepareAction();

        if (doAction) {

            final TuttiActionSwingWorker<A> worker =
                    new TuttiActionSwingWorker<A>(action);

            SwingUtilities.invokeLater(new Runnable() {
                @Override
                public void run() {

                    // make ui busy
                    worker.updateBusyState(true);

                }
            });

            if (log.isInfoEnabled()) {
                log.info("Before execute of action " + action);
            }

            // perform and release action
            worker.execute();
            if (log.isInfoEnabled()) {
                log.info("After execute of action " + action);
            }

        } else {

            // release action
            action.releaseAction();
        }
    }

    //    protected void setMnemonic(int key) {
//        putValue(MNEMONIC_KEY, key);
//    }

    protected static class TuttiActionSwingWorker<A extends AbstractTuttiAction> extends SwingWorker<Void, String> {

        protected final TuttiActionUI actionUI;

        protected final A action;

        protected TimerTask timer;

        protected TuttiActionSwingWorker(A action) {

            this.action = action;
            this.actionUI = action.getContext().getActionUI();
        }

        @Override
        protected Void doInBackground() throws Exception {

            if (!isCancelled()) {

                timer = new TimerTask() {

                    @Override
                    public void run() {
                        if (isCancelled() || isDone()) {

                            if (log.isDebugEnabled()) {
                                log.debug("Task [" + action + "] was already canceled or done, do nothing");
                            }
                        } else {

                            if (log.isDebugEnabled()) {
                                log.debug("Task [" + action + "] is started, show waiting dialog");
                            }

                            actionUI.open(action);

                        }
                    }
                };

                t.schedule(timer, 1000);

                try {

                    action.performAction();


                } finally {
                    if (log.isInfoEnabled()) {
                        log.info("Task [" + this + "] done");
                    }

                    action.releaseAction();
                }
            }
            return null;
        }

        @Override
        protected void done() {
            super.done();

            // close timer

            if (timer != null) {

                timer.cancel();
            }

            actionUI.close();

            updateBusyState(false);

            if (action.isFailed()) {

                exceptionHandler.handle(new TuttiExceptionHandler.TuttiActionException(action, action.getError()));
            }
        }

        protected void updateBusyState(boolean busy) {

            boolean hideBody = action.isHideBody();

            final MainUI mainUI = action.getContext().getMainUI();
            JXTitledPanel body = mainUI.getBody();
            if (busy) {
                // ui bloquee
                if (log.isDebugEnabled()) {
                    log.debug("block ui in busy mode");
                }
                mainUI.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
            } else {
                // ui debloquee
                if (log.isDebugEnabled()) {
                    log.debug("unblock ui in none busy mode");
                }
                mainUI.setCursor(Cursor.getDefaultCursor());
            }
            if (hideBody) {
                body.setVisible(!busy);
            }

        }
    }
}
