package fr.ifremer.tutti.ui.swing;

/*
 * #%L
 * Tutti :: UI
 * $Id: MainUIHandler.java 461 2013-02-25 14:11:54Z blavenier $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.0.1/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/MainUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import fr.ifremer.tutti.persistence.entities.data.Cruise;
import fr.ifremer.tutti.persistence.entities.data.Program;
import fr.ifremer.tutti.persistence.entities.protocol.TuttiProtocol;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.ui.swing.config.TuttiApplicationConfig;
import fr.ifremer.tutti.ui.swing.config.TuttiConfigUI;
import fr.ifremer.tutti.ui.swing.content.cruise.EditCruiseUI;
import fr.ifremer.tutti.ui.swing.content.db.DbManagerUI;
import fr.ifremer.tutti.ui.swing.content.db.OpenDbAction;
import fr.ifremer.tutti.ui.swing.content.home.SelectCruiseUI;
import fr.ifremer.tutti.ui.swing.content.operation.FishingOperationsUI;
import fr.ifremer.tutti.ui.swing.content.program.EditProgramUI;
import fr.ifremer.tutti.ui.swing.content.protocol.EditProtocolUI;
import fr.ifremer.tutti.ui.swing.content.referential.ManageTemporaryReferentialUI;
import fr.ifremer.tutti.ui.swing.util.action.TuttiActionUI;
import jaxx.runtime.swing.AboutPanel;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.widget.SwingSession;

import javax.swing.Icon;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JToolBar;
import java.awt.Desktop;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.net.URL;
import java.util.Calendar;

import static org.nuiton.i18n.I18n._;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class MainUIHandler extends AbstractTuttiUIHandler<TuttiUIContext, MainUI> {

    /** Logger. */
    private static final Log log = LogFactory.getLog(MainUIHandler.class);

    public static final String CLONE_PROTOCOL = "cloneProtocol";

    protected JComponent currentBody;

    protected PersistenceService persistenceService;

    protected MainUIHandler(TuttiUIContext context, MainUI ui) {
        super(context, ui);
        this.persistenceService = context.getService(PersistenceService.class);
        context.setMainUI(ui);
        context.setActionUI(new TuttiActionUI(ui, context));
    }

    protected MainUIHandler(TuttiUIContext context) {
        super(context, null);
        this.persistenceService = null;
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    public void beforeInitUI() {
        context.addPropertyChangeListener(new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                String propertyName = evt.getPropertyName();
                if (TuttiUIContext.ID_TO_SAVE_PROPERTIES.contains(propertyName)) {

                    // change the ui title
                    changeTitle();

                } else if (propertyName.equals(TuttiUIContext.PROPERTY_SCREEN)) {
                    setScreen((TuttiScreen) evt.getNewValue());
                }
            }
        });
        ui.setContextValue(ui, MainUI.class.getName());
    }

    @Override
    public void afterInitUI() {

        initUI(ui);

        // Init SwingSession
        SwingSession swingSession = context.getSwingSession();
        swingSession.add(ui);
        swingSession.save();

        //SwingUtil.getLayer(ui.getBody()).setUI(new BetaLayerUI());

        changeTitle();

        JToolBar bar = ui.getBottomBar();
        ui.getStatus().addWidget(bar, 0);

        if (!context.isDbExist()) {

            // clean db context
            context.clearDbContext();

            // go to manage db screen (to install db)
            context.setScreen(TuttiScreen.MANAGE_DB);
        } else {

            // open tutti db
            OpenDbAction openDbAction = createAction(OpenDbAction.class);
            openDbAction.actionPerformed(null);
        }
    }

    @Override
    public void onCloseUI() {

        context.close();
    }

    @Override
    public SwingValidator<TuttiUIContext> getValidator() {
        return null;
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    public void closeTutti() {
        RunTutti.closeTutti(ui, true);
    }

    public void reloadTutti() {

        // Close the application
        RunTutti.closeTutti(ui, false);

        RunTutti.startTutti(context);
    }

    public void reloadApplication() {
        //TODO
    }

    public void reloadShortcuts() {
        //TODO
    }

    public void showConfig() {

        TuttiConfigUI configUI = new TuttiConfigUI(ui);
        configUI.createUI();
    }

    public void showManageProtocol() {
        context.setScreen(TuttiScreen.EDIT_PROTOCOL);
    }

    public void showSelectCampaign() {
        context.setScreen(TuttiScreen.SELECT_CRUISE);
    }

    public void showFillCatches() {
        context.setValidationContext(TuttiUIContext.VALIDATION_CONTEXT_EDIT);
        context.setScreen(TuttiScreen.EDIT_FISHING_OPERATION);
    }

    public void showValidateCatches() {
        context.setValidationContext(TuttiUIContext.VALIDATION_CONTEXT_VALIDATE);
        context.setScreen(TuttiScreen.EDIT_FISHING_OPERATION);
    }

    public void showAbout() {

        ApplicationConfig applicationConfig =
                context.getConfig().getApplicationConfig();

        String iconPath = applicationConfig.getOption("application.icon.path");
        String name = "tutti-ui-swing";
        String licensePath = "META-INF/" + name + "-LICENSE.txt";
        String thirdPartyPath = "META-INF/" + name + "-THIRD-PARTY.txt";

        AboutPanel about = new AboutPanel();
        about.setTitle(_("tutti.title.about"));
        about.setAboutText(_("tutti.about.message"));

        TuttiApplicationConfig config = context.getConfig();
        int currentYear = Calendar.getInstance().get(Calendar.YEAR);
        int inceptionYear = config.getInceptionYear();
        String years;
        if (currentYear != inceptionYear) {
            years = inceptionYear + "-" + currentYear;
        } else {
            years = inceptionYear + "";
        }

        about.setBottomText(_("tutti.about.bottomText",
                              config.getOrganizationName(),
                              years,
                              config.getVersion()));
        about.setIconPath(iconPath);
        about.setLicenseFile(licensePath);
        about.setThirdpartyFile(thirdPartyPath);
        about.buildTopPanel();
        about.init();
        about.showInDialog(ui, true);

        // register on swing session
        context.getSwingSession().add(about);
    }

    public void gotoSite() {
        TuttiApplicationConfig config = context.getConfig();

        URL siteURL = config.getSiteUrl();

        if (log.isDebugEnabled()) {
            log.debug("goto " + siteURL);
        }
        if (Desktop.isDesktopSupported() &&
            Desktop.getDesktop().isSupported(Desktop.Action.BROWSE)) {
            try {
                Desktop.getDesktop().browse(siteURL.toURI());
            } catch (Exception e) {
                TuttiUIContext.getErrorHelper().showErrorDialog(
                        "Failed to open '" + siteURL + "' in browser", e);
            }
        }
    }

    @Override
    public final void showInformationMessage(String message) {
        ui.getStatus().setStatus(message);
    }

    public void registerValidator(SwingValidator validator) {
        ui.getValidatorMessageWidget().registerValidator(validator);
    }

    @Override
    public void clearValidators() {
        ui.getValidatorMessageWidget().clearValidators();
    }

    public AbstractTuttiUIHandler getCurrentHandler() {
        TuttiUI<?, ?> body = (TuttiUI<?, ?>) currentBody;
        return body.getHandler();
    }

    /**
     * Reload the persistence service and set the new ne into the handler.
     *
     * @since 1.0
     */
    public void reloadPersistenceService() {
        this.persistenceService = context.reloadPersistenceService();
    }

    /**
     * Get the persistence service instance used by the handler.
     *
     * @return the persistence service instance used by the handler
     * @since 1.0
     */
    public PersistenceService getPersistenceService() {
        return persistenceService;
    }

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected void setScreen(TuttiScreen screen) {

        try {

            // close current body (if any)
            if (currentBody != null) {
                TuttiUI<?, ?> body = (TuttiUI<?, ?>) currentBody;
                body.getHandler().onCloseUI();

                context.getSwingSession().save();

                ui.getBody().remove(currentBody);

                currentBody = null;
            }

            if (screen != null) {

                String programId = context.getProgramId();
                String cruiseId = context.getCruiseId();
                String protocolId = context.getProtocolId();

                JComponent screenUI;
                String screenTitle;

                Icon icon;
                switch (screen) {
                    default:
                    case MANAGE_DB:

                        screenUI = new DbManagerUI(ui);
                        screenTitle = _("tutti.title.manageDb");
                        icon = ui.getMenuActionManageDb().getIcon();
                        break;

                    case SELECT_CRUISE:

                        screenUI = new SelectCruiseUI(ui);
                        screenTitle = _("tutti.title.home");
                        icon = ui.getMenuActionSelectCruise().getIcon();
                        break;

                    case EDIT_PROGRAM:

                        if (programId == null) {
                            screenTitle = _("tutti.title.create.program");
                        } else {
                            screenTitle = _("tutti.title.edit.program");
                        }

                        screenUI = new EditProgramUI(ui);
                        icon = ui.getMenuActionEditProgram().getIcon();
                        break;

                    case EDIT_CRUISE:

                        if (cruiseId == null) {
                            screenTitle = _("tutti.title.create.cruise");
                        } else {
                            screenTitle = _("tutti.title.edit.cruise");
                        }

                        screenUI = new EditCruiseUI(ui);
                        icon = ui.getMenuActionEditCruise().getIcon();
                        break;

                    case EDIT_PROTOCOL:

                        if (protocolId == null) {
                            screenTitle = _("tutti.title.create.protocol");
                        } else {
                            screenTitle = _("tutti.title.edit.protocol");
                        }

                        screenUI = new EditProtocolUI(ui);
                        icon = ui.getMenuActionEditProtocol().getIcon();
                        break;

                    case CLONE_PROTOCOL:
                        screenTitle = _("tutti.title.create.protocol");
                        ui.setContextValue(true, CLONE_PROTOCOL);
                        screenUI = new EditProtocolUI(ui);
                        icon = ui.getMenuActionEditProtocol().getIcon();
                        break;

                    case EDIT_FISHING_OPERATION:

                        String validationContext = context.getValidationContext();

                        if (TuttiUIContext.VALIDATION_CONTEXT_EDIT.equals(validationContext)) {

                            // edit catches
                            screenTitle = _("tutti.title.edit.operations", getSelectedCruiseTitle());
                            icon = ui.getMenuActionEditCatches().getIcon();
                        } else {

                            // validate catches
                            screenTitle = _("tutti.title.validate.operations", getSelectedCruiseTitle());
                            icon = ui.getMenuActionValidateCatches().getIcon();
                        }
                        screenUI = new FishingOperationsUI(ui);

                        break;

                    case IMPORT_TEMPORARY_REFERENTIAL:
                        screenTitle = _("tutti.title.import.temporary.referential");
                        screenUI = new ManageTemporaryReferentialUI(ui);
                        icon = ui.getMenuImportTemporaryReferential().getIcon();
                        break;
                }

                this.currentBody = screenUI;
                context.getSwingSession().add(currentBody);
                ui.getBody().setTitle(screenTitle);
                ui.getBody().add(currentBody);
                ui.getBody().setLeftDecoration(new JLabel(icon));

            }
        } catch (Exception e) {

            AbstractTuttiAction mainAction = AbstractTuttiAction.getMainAction();
            if (mainAction != null) {
                mainAction.setFailed(true);
            }

            TuttiUIContext.getErrorHelper().showErrorDialog(
                    "Could not change to screen " + screen, e);

            // go back to home
            context.setScreen(TuttiScreen.SELECT_CRUISE);
        }
    }

    protected void changeTitle() {

        String title = getSelectedCruiseTitle();

        ui.setTitle("Tutti - v " + getConfig().getVersion() + " [ " + title + " ]");
    }

    protected String getSelectedCruiseTitle() {


        String title;

        if (context.isDbLoaded()) {

            String programId = context.getProgramId();

            if (programId == null) {

                // no program selected (so neither cruise)

                title = _("tutti.title.noSelectedProgram");

            } else {

                // selected program

                Program program = persistenceService.getProgram(programId);

                title = _("tutti.title.selectedProgram", program.getName()) + " / ";

                String cruiseId = context.getCruiseId();

                if (cruiseId == null) {

                    // no selected cruise

                    title += _("tutti.title.noSelectedCruise");

                } else {

                    // selected cruise
                    Cruise cruise;
                    cruise = persistenceService.getCruise(cruiseId);

                    if (cruise != null) {
                    	title += _("tutti.title.selectedCruise", cruise.getName());
                    }
                }

                title += " / ";

                String protocolId = context.getProtocolId();

                if (protocolId == null) {

                    // no selected protocol

                    title += _("tutti.title.noSelectedProtocol");

                } else {

                    // selected protocol
                    TuttiProtocol protocol = persistenceService.getProtocol(protocolId);

                    title += _("tutti.title.selectedProtocol", protocol.getName());
                }

            }
        } else {
            title = _("tutti.title.nodb");
        }

        return title;
    }
}
