package fr.ifremer.tutti.ui.swing.content.operation.catches.species.split;

/*
 * #%L
 * Tutti :: UI
 * $Id: SplitSpeciesBatchUIHandler.java 147 2013-01-03 17:01:03Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-0.3/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/catches/species/split/SplitSpeciesBatchUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.ezware.oxbow.swingbits.util.Preconditions;
import com.google.common.collect.Lists;
import fr.ifremer.tutti.persistence.entities.referential.Caracteristic;
import fr.ifremer.tutti.persistence.entities.referential.CaracteristicQualitativeValue;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.ui.swing.TuttiUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.SampleCategory;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.SampleCategoryType;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.SpeciesBatchRowModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.SpeciesBatchUI;
import fr.ifremer.tutti.ui.swing.util.Cancelable;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import fr.ifremer.tutti.ui.swing.util.table.AbstractTuttiTableUIHandler;
import jaxx.runtime.validator.swing.SwingValidatorMessageTableRenderer;
import jaxx.runtime.validator.swing.SwingValidatorUtil;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.table.DefaultTableColumnModelExt;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.List;

/**
 * Handler of {@link SplitSpeciesBatchUI}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.3
 */
public class SplitSpeciesBatchUIHandler
        extends AbstractTuttiTableUIHandler<SplitSpeciesBatchRowModel, SplitSpeciesBatchUIModel> implements Cancelable {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(SplitSpeciesBatchUIHandler.class);

    /**
     * Parent UI.
     *
     * @since 0.3
     */
    private final SpeciesBatchUI parentUi;

    /**
     * UI.
     *
     * @since 0.3
     */
    private final SplitSpeciesBatchUI ui;

    public SplitSpeciesBatchUIHandler(SpeciesBatchUI parentUi,
                                      SplitSpeciesBatchUI ui) {
        super(parentUi.getHandler().getContext(),
              SplitSpeciesBatchRowModel.PROPERTY_CATEGORY_VALUE,
              SplitSpeciesBatchRowModel.PROPERTY_WEIGHT);
        this.parentUi = parentUi;
        this.ui = ui;
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiTableUIHandler methods                                --//
    //------------------------------------------------------------------------//

    @Override
    protected SplitSpeciesBatchTableModel getTableModel() {
        return (SplitSpeciesBatchTableModel) getTable().getModel();
    }

    @Override
    protected JXTable getTable() {
        return ui.getTable();
    }

    @Override
    protected boolean isRowValid(SplitSpeciesBatchRowModel row) {
        //TODO
        return row.getCategoryValue() != null && row.getWeight() != null;
    }

    @Override
    protected void saveSelectedRowIfRequired(TuttiBeanMonitor<SplitSpeciesBatchRowModel> rowMonitor,
                                             SplitSpeciesBatchRowModel row) {
        if (rowMonitor.wasModified()) {

            if (row.isValid()) {
                if (log.isInfoEnabled()) {
                    log.info("Change row that was modified and valid");
                }
            }

            rowMonitor.clearModified();
        }
    }

    @Override
    protected void onRowModified(int rowIndex,
                                 SplitSpeciesBatchRowModel row,
                                 String propertyName,
                                 Object oldValue,
                                 Object newValue) {

        recomputeRowValidState(row);

        if (SplitSpeciesBatchRowModel.PROPERTY_WEIGHT.equals(propertyName)) {

            // Need to recompute the sample weight
            computeSampleWeight(row);
        }
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    protected SplitSpeciesBatchUIModel getModel() {
        return ui.getModel();
    }

    @Override
    public void beforeInitUI() {

        SplitSpeciesBatchUIModel model = new SplitSpeciesBatchUIModel();

        ui.setContextValue(model);
    }

    @Override
    public void afterInitUI() {

        initUI(ui);

        SplitSpeciesBatchUIModel model = getModel();

        initBeanComboBox(ui.getCategoryComboBox(),
                         Lists.<SampleCategoryType>newArrayList(),
                         model.getSelectedCategory());

        // when category changed, remove selected category
        model.addPropertyChangeListener(SplitSpeciesBatchUIModel.PROPERTY_CATEGORY, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {

                SplitSpeciesBatchUIModel source =
                        (SplitSpeciesBatchUIModel) evt.getSource();

                // unselect previous selected category
                source.setSelectedCategory(null);

                // fill comboBox with new list
                ui.getCategoryComboBox().setData((List<SampleCategoryType>) evt.getNewValue());
            }
        });

        // when selected category changed, regenerate the table model + add inside some default rows
        model.addPropertyChangeListener(SplitSpeciesBatchUIModel.PROPERTY_SELECTED_CATEGORY, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {

                SplitSpeciesBatchUIModel source =
                        (SplitSpeciesBatchUIModel) evt.getSource();

                // when selected category change, sample total weight is reset
                source.setSampleWeight(null);

                SampleCategoryType newValue =
                        (SampleCategoryType) evt.getNewValue();
                generateTableModel(newValue);
            }
        });

        generateTableModel(null);

        initTable(getTable());

        SwingValidatorUtil.installUI(ui.getErrorTable(),
                                     new SwingValidatorMessageTableRenderer());

        listenValidatorValid(ui.getValidator(), model);
    }

    @Override
    public void onCloseUI() {
        if (log.isInfoEnabled()) {
            log.info("closing: " + ui);
        }
    }

    //------------------------------------------------------------------------//
    //-- Cancelable methods                                                 --//
    //------------------------------------------------------------------------//

    @Override
    public void cancel() {

        if (log.isInfoEnabled()) {
            log.info("Cancel UI " + ui);
        }

        // evict model from validator
        ui.getValidator().setBean(null);

        // when canceling always invalid model
        getModel().setValid(false);

        // close dialog
        closeDialog(ui);
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    public String decorateSpecies(Species object) {
        String result = object == null ? "" : super.decorate(object);
        return result;
    }

    public void editBatch(SpeciesBatchRowModel batch) {

        // get possible the last used
        List<SampleCategoryType> categories =
                Lists.newArrayList(SampleCategoryType.values());

        if (batch != null) {
            //TODO Use the samplingOrder (+ the one from the table model)
            List<SampleCategoryType> samplingOrder =
                    parentUi.getModel().getSamplingOrder();

            SampleCategory<?> lastCategory = batch.getSampleCategory();

            Preconditions.checkNotNull(
                    lastCategory,
                    "Can't split a species batch with no sample category.");

            if (batch.getSortedUnsortedCategory().isValid()) {
                categories.remove(SampleCategoryType.sortedUnsorted);
            }

            if (batch.getSizeCategory().isValid()) {
                categories.remove(SampleCategoryType.size);
            }

            if (batch.getSexCategory().isValid()) {
                categories.remove(SampleCategoryType.sex);
            }

            if (batch.getMaturityCategory().isValid()) {
                categories.remove(SampleCategoryType.maturity);
            }

            if (batch.getAgeCategory().isValid()) {
                categories.remove(SampleCategoryType.age);
            }
        }

        SplitSpeciesBatchUIModel model = getModel();

        // connect model to validator
        ui.getValidator().setBean(model);

        model.setSampleWeight(null);
        model.setCategory(categories);

        // keep batch (will be used to push back editing entry)
        model.setBatch(batch);
    }

    public void save() {

        if (log.isInfoEnabled()) {
            log.info("Save UI " + ui);
        }

        closeDialog(ui);
    }

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected void computeSampleWeight(SplitSpeciesBatchRowModel row) {

        if (log.isInfoEnabled()) {
            log.info("Will recompute sample weight from row: " + row);
        }

        Float result = 0f;
        List<SplitSpeciesBatchRowModel> rows = getTableModel().getRows();
        for (SplitSpeciesBatchRowModel rowModel : rows) {
            Float weight = rowModel.getWeight();
            if (weight != null) {
                result += weight;
            }
        }
        getModel().setSampleWeight(result);
    }

    protected void generateTableModel(SampleCategoryType category) {

        // when generate a new table model, then reset previous rows from model
        getModel().setRows(null);

        Caracteristic data = null;

        DefaultTableColumnModelExt columnModel =
                new DefaultTableColumnModelExt();

        boolean editableCategoryValue = false;
        if (category != null) {

            switch (category) {

                case sortedUnsorted:
                    data = persistenceService.getSortedUnsortedCaracteristic();
                    break;
                case size:
                    data = persistenceService.getSizeCategoryCaracteristic();
                    break;
                case sex:
                    data = persistenceService.getSexCaracteristic();
                    break;
                case maturity:
                    data = persistenceService.getMaturityCaracteristic();
                    break;
                case age:
                    editableCategoryValue = true;
                    addFloatColumnToModel(columnModel,
                                          SplitSpeciesBatchTableModel.EDITABLE_CATEGORY_VALUE,
                                          TuttiUI.DECIMAL1_PATTERN);
                    break;
            }

            if (data != null) {

                if (log.isDebugEnabled()) {
                    log.debug("Got " + data.sizeQualitativeValue() + " qualitative data to add");
                }
                addColumnToModel(columnModel,
                                 null,
                                 newTableCellRender(CaracteristicQualitativeValue.class),
                                 SplitSpeciesBatchTableModel.READ_ONLY_CATEGORY_VALUE);
            }
            { // Weight

                addFloatColumnToModel(columnModel,
                                      SplitSpeciesBatchTableModel.WEIGHT,
                                      TuttiUI.DECIMAL3_PATTERN);
            }
        }

        // create table model
        SplitSpeciesBatchTableModel tableModel =
                new SplitSpeciesBatchTableModel(columnModel,
                                                getModel(),
                                                editableCategoryValue);

        JXTable table = getTable();

        // remove all listener on tables we could add before
        uninstallTableSaveOnRowChangedSelectionListener();
        uninstallTableKeyListener();

        if (log.isDebugEnabled()) {
            log.debug("Install new table model " + tableModel);
        }
        table.setModel(tableModel);
        table.setColumnModel(columnModel);

        // install table listeners
        installTableSaveOnRowChangedSelectionListener();
        installTableKeyListener(columnModel, table);

        // fill datas

        List<SplitSpeciesBatchRowModel> rows = Lists.newArrayList();

        if (data != null) {

            // add a row for each qualitive value
            for (CaracteristicQualitativeValue qualitativeValue : data.getQualitativeValue()) {
                if (log.isDebugEnabled()) {
                    log.debug("Add QV: " + qualitativeValue);
                }
                SplitSpeciesBatchRowModel newRow = tableModel.createNewRow();
                newRow.setCategoryValue(qualitativeValue);
                rows.add(newRow);
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Will add " + rows.size() + " rows in table model " +
                      "(can add a first empty row? " + editableCategoryValue + ").");
        }

        getModel().setRows(rows);
    }

}