package fr.ifremer.tutti.ui.swing.util.table;

/*
 * #%L
 * Tutti :: UI
 * $Id: AbstractTuttiTableUIHandler.java 81 2012-12-17 07:39:44Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-0.2/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/util/table/AbstractTuttiTableUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import fr.ifremer.tutti.ui.swing.AbstractTuttiBeanUIModel;
import fr.ifremer.tutti.ui.swing.AbstractTuttiUIHandler;
import fr.ifremer.tutti.ui.swing.TuttiUIContext;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.swing.JAXXWidgetUtil;
import jaxx.runtime.swing.editor.bean.BeanUIUtil;
import jaxx.runtime.swing.editor.cell.NumberCellEditor;
import jaxx.runtime.swing.renderer.DecoratorTableCellRenderer;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.autocomplete.ComboBoxCellEditor;
import org.jdesktop.swingx.autocomplete.ObjectToStringConverter;
import org.jdesktop.swingx.table.TableColumnExt;
import org.nuiton.util.decorator.Decorator;

import javax.swing.JComboBox;
import javax.swing.JTable;
import javax.swing.border.LineBorder;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import java.awt.Color;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Enumeration;
import java.util.List;
import java.util.Set;

import static org.nuiton.i18n.I18n._;

/**
 * @param <R> type of a row
 * @param <M> type of the ui model
 * @author tchemit <chemit@codelutin.com>
 * @since 0.2
 */
public abstract class AbstractTuttiTableUIHandler<R extends AbstractTuttiBeanUIModel, M extends AbstractTuttiTableUIModel<?, R, M>> extends AbstractTuttiUIHandler<M> {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(AbstractTuttiTableUIHandler.class);

    /**
     * Monitor the selected row (save it only if something has changed).
     *
     * @since 0.2
     */
    private final TuttiBeanMonitor<R> rowMonitor;

    protected abstract JXTable getTable();

    protected abstract AbstractTuttiTableModel<R> getTableModel();

    protected abstract TableColumnModel createTableColumnModel();

    protected abstract void onRowModified(R row,
                                          String propertyName,
                                          Object oldValue,
                                          Object newValue);

    protected abstract void onRowValidStateChanged(R row,
                                                   Boolean oldValue,
                                                   Boolean newValue);

    protected abstract void onRowModifyStateChanged(R row,
                                                    Boolean oldValue,
                                                    Boolean newValue);


    protected String[] getRowPropertiesToIgnore() {
        return ArrayUtils.EMPTY_STRING_ARRAY;
    }

    protected AbstractTuttiTableUIHandler(TuttiUIContext context,
                                          String... properties) {
        super(context);

        rowMonitor = new TuttiBeanMonitor<R>(properties);

        // listen when bean is changed
        rowMonitor.addPropertyChangeListener(TuttiBeanMonitor.PROPERTY_BEAN, new PropertyChangeListener() {

            final Set<String> propertiesToSkip =
                    Sets.newHashSet(getRowPropertiesToIgnore());

            final PropertyChangeListener l = new PropertyChangeListener() {
                @Override
                public void propertyChange(PropertyChangeEvent evt) {
                    String propertyName = evt.getPropertyName();

                    R row = (R) evt.getSource();

                    Object oldValue = evt.getOldValue();
                    Object newValue = evt.getNewValue();

                    if (AbstractTuttiBeanUIModel.PROPERTY_VALID.equals(propertyName)) {
                        onRowValidStateChanged(row,
                                               (Boolean) oldValue,
                                               (Boolean) newValue);
                    } else if (AbstractTuttiBeanUIModel.PROPERTY_MODIFY.equals(propertyName)) {
                        onRowModifyStateChanged(row,
                                                (Boolean) oldValue,
                                                (Boolean) newValue);
                    } else if (!propertiesToSkip.contains(propertyName)) {

                        onRowModified(row,
                                      propertyName,
                                      oldValue,
                                      newValue);
                    }
                }
            };

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                R oldValue = (R) evt.getOldValue();
                R newValue = (R) evt.getNewValue();
                if (log.isInfoEnabled()) {
                    log.info("Monitor row changed from " +
                             oldValue + " to " + newValue);
                }
                if (oldValue != null) {
                    oldValue.removePropertyChangeListener(l);
                }
                if (newValue != null) {
                    newValue.addPropertyChangeListener(l);
                }
            }
        });
    }

    protected KeyListener installTableKeyListener(TableColumnModel columnModel,
                                                  JTable table) {

        AbstractTuttiTableModel<R> model = getTableModel();
        final MoveToNextEditableCellAction nextCellAction =
                MoveToNextEditableCellAction.newAction(model, table);
        final MoveToPreviousEditableCellAction previousCellAction =
                MoveToPreviousEditableCellAction.newAction(model, table);

        final MoveToNextEditableRowAction nextRowAction =
                MoveToNextEditableRowAction.newAction(model, table);
        final MoveToPreviousEditableRowAction previousRowAction =
                MoveToPreviousEditableRowAction.newAction(model, table);

        // Key adapter à ajouter sur les éditeurs où l'on souhaite gérer les
        // touches "entrer", "gauche", "droite" de facon personnalisée.
        KeyAdapter keyAdapter = new KeyAdapter() {

            @Override
            public void keyPressed(KeyEvent e) {
                if (e.getKeyCode() == KeyEvent.VK_ENTER ||
                    e.getKeyCode() == KeyEvent.VK_RIGHT ||
                    e.getKeyCode() == KeyEvent.VK_TAB) {
                    e.consume();
                    nextCellAction.actionPerformed(null);
                } else if (e.getKeyCode() == KeyEvent.VK_LEFT) {
                    e.consume();
                    previousCellAction.actionPerformed(null);
                } else if (e.getKeyCode() == KeyEvent.VK_UP) {
                    e.consume();
                    previousRowAction.actionPerformed(null);
                } else if (e.getKeyCode() == KeyEvent.VK_DOWN) {
                    e.consume();
                    nextRowAction.actionPerformed(null);
                }
            }
        };
        table.addKeyListener(keyAdapter);

        Enumeration<TableColumn> columns = columnModel.getColumns();
        while (columns.hasMoreElements()) {
            TableColumn tableColumn = columns.nextElement();
            TableCellEditor cellEditor = tableColumn.getCellEditor();
            if (cellEditor instanceof NumberCellEditor) {
                NumberCellEditor editor = (NumberCellEditor) cellEditor;
                editor.getNumberEditor().getTextField().addKeyListener(keyAdapter);
            }
        }
        return keyAdapter;
    }

    protected void addColumnToModel(TableColumnModel model,
                                    TableCellEditor editor,
                                    TableCellRenderer renderer,
                                    ColumnIdentifier<R> identifier) {

        TableColumn col = new TableColumnExt(model.getColumnCount());
        col.setCellEditor(editor);
        col.setCellRenderer(renderer);
        col.setHeaderValue(_(identifier.getHeaderI18nKey()));
        //TODO Use tip in a header renderer

        col.setIdentifier(identifier);
        model.addColumn(col);
    }

    protected void addColumnToModel(TableColumnModel model,
                                    ColumnIdentifier<R> identifier) {

        addColumnToModel(model, null, null, identifier);
    }

    protected void addFloatColumnToModel(TableColumnModel model,
                                         ColumnIdentifier<R> identifier,
                                         String numberPattern) {

        NumberCellEditor<Float> editor =
                JAXXWidgetUtil.newNumberTableCellEditor(Float.class, false);
        editor.getNumberEditor().setSelectAllTextOnError(true);
        editor.getNumberEditor().getTextField().setBorder(new LineBorder(Color.GRAY, 2));
        editor.getNumberEditor().setNumberPattern(numberPattern);

        addColumnToModel(model, editor, null, identifier);

    }

    protected void addIntegerColumnToModel(TableColumnModel model,
                                           ColumnIdentifier<R> identifier,
                                           String numberPattern) {

        NumberCellEditor<Integer> editor =
                JAXXWidgetUtil.newNumberTableCellEditor(Integer.class, false);
        editor.getNumberEditor().setSelectAllTextOnError(true);
        editor.getNumberEditor().getTextField().setBorder(new LineBorder(Color.GRAY, 2));
        editor.getNumberEditor().setNumberPattern(numberPattern);

        addColumnToModel(model, editor, null, identifier);
    }


    protected void addBooleanColumnToModel(TableColumnModel model,
                                           ColumnIdentifier<R> identifier,
                                           JTable table) {

        addColumnToModel(model,
                         table.getDefaultEditor(Boolean.class),
                         table.getDefaultRenderer(Boolean.class),
                         identifier);
    }

    protected <B> void addComboDataColumnToModel(TableColumnModel model,
                                                 ColumnIdentifier<R> identifier,
                                                 Decorator<B> decorator,
                                                 List<B> data) {
        JComboBox comboBox = new JComboBox();
        comboBox.setRenderer(newListCellRender(decorator));

        // add a null value at first position
        if (!data.isEmpty() && data.get(0) != null) {
            data.add(0, null);
        }
        SwingUtil.fillComboBox(comboBox, data, null);

        ObjectToStringConverter converter = BeanUIUtil.newDecoratedObjectToStringConverter(decorator);
        BeanUIUtil.decorate(comboBox, converter);
        ComboBoxCellEditor editor = new ComboBoxCellEditor(comboBox);

        addColumnToModel(model,
                         editor,
                         newTableCellRender(decorator),
                         identifier);
    }

    protected void listenRowsFromModel() {
        getModel().addPropertyChangeListener(AbstractTuttiTableUIModel.PROPERTY_ROWS, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                onModelRowsChanged((List<R>) evt.getNewValue());
            }
        });
    }

    protected void onModelRowsChanged(List<R> rows) {
        getTableModel().setRows(rows);
    }

    protected <O> TableCellRenderer newTableCellRender(Class<O> type) {

        return newTableCellRender(type, null);
    }

    protected <O> TableCellRenderer newTableCellRender(Class<O> type, String name) {

        Decorator<O> decorator = getDecorator(type, name);

        TableCellRenderer result = newTableCellRender(decorator);
        return result;
    }

    protected <O> TableCellRenderer newTableCellRender(Decorator<O> decorator) {

        Preconditions.checkNotNull(decorator);

        DecoratorTableCellRenderer result = new DecoratorTableCellRenderer(decorator);
        return result;
    }

    protected TuttiBeanMonitor<R> getRowMonitor() {
        return rowMonitor;
    }

}
