package fr.ifremer.tutti.ui.swing.content.operation.species;

/*
 * #%L
 * Tutti :: UI
 * $Id: SpeciesBatchTreeModel.java 77 2012-12-15 16:05:45Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-0.2/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/species/SpeciesBatchTreeModel.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import fr.ifremer.tutti.persistence.entities.referential.SortedUnsortedCategory;
import fr.ifremer.tutti.ui.swing.util.TuttiUIUtil;
import org.apache.commons.collections.CollectionUtils;

import java.io.Serializable;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 0.2
 */
public class SpeciesBatchTreeModel implements Serializable {

    private static final long serialVersionUID = 1L;

    /**
     * Root node (with no sampling on it.
     *
     * @since 0.2
     */
    protected final SpeciesBatchTreeNode root;

    /**
     * Property which are used to sub sampling.
     *
     * @since 0.2
     */
    protected String[] samplingOrder;

    /**
     * Mapping from row to node.
     *
     * @since 0.2
     */
    protected final Map<SpeciesBatchRowModel, SpeciesBatchTreeNode> rowToNode;

    /**
     * Mapping from node to row.
     *
     * @since 0.2
     */
    protected final Multimap<SpeciesBatchTreeNode, SpeciesBatchRowModel> nodeToRow;

    public SpeciesBatchTreeModel(String... samplingOrder) {
        this.samplingOrder = samplingOrder;
        root = new SpeciesBatchTreeNode();
        rowToNode = Maps.newHashMap();
        nodeToRow = HashMultimap.create();
    }

    public void populate(List<SpeciesBatchRowModel> rows) {

        clear();

        for (SpeciesBatchRowModel row : rows) {

            SpeciesBatchTreeNode node = getSamplingNode(row);

            // check if row is valid
            boolean rowIsValid = isValid(row, node);

            // set it in row
            row.setValid(rowIsValid);
        }
    }

    public String[] getSamplingOrder() {
        return samplingOrder;
    }

    public void setSamplingOrder(String... samplingOrder) {
        this.samplingOrder = samplingOrder;
        clear();
    }

    protected void clear() {
        // clear tree representation
        root.removeAllChildren();

        // clear mappings
        rowToNode.clear();
        nodeToRow.clear();
    }

    public SpeciesBatchTreeNode removeNodeFromCache(SpeciesBatchRowModel row) {
        SpeciesBatchTreeNode result = rowToNode.remove(row);
        if (result != null) {

            nodeToRow.remove(result, row);
        }
        return result;
    }

    public SpeciesBatchTreeNode getSamplingNode(SpeciesBatchRowModel row) {

        // search frist in cache
        SpeciesBatchTreeNode result = rowToNode.get(row);

        if (result == null) {

            // not in cache must build the path
            result = getSamplingNode(root, 0, row);

            if (result == root) {

                // this means an empty row, so no node for it
                result = null;

            } else {

                // found a row with a sampling, add it in cache
                rowToNode.put(row, result);
                nodeToRow.put(result, row);
            }
        }
        return result;
    }

    protected SpeciesBatchTreeNode getSamplingNode(SpeciesBatchTreeNode node,
                                                   int samplingIndex,
                                                   SpeciesBatchRowModel row) {

        String samplingKey = samplingOrder[samplingIndex];

        Serializable samplingValue = (Serializable)
                TuttiUIUtil.getProperty(row, samplingKey);

        SpeciesBatchTreeNode result;

        if (samplingValue == null) {

            // was already on last sampling node
            result = node;
        } else {

            result = node.getChild(samplingValue);

            if (result == null) {

                // new node, creates it and add it as child of current node
                result = new SpeciesBatchTreeNode(samplingKey, samplingValue);
                node.add(result);
            }

            if (samplingIndex < samplingOrder.length) {

                // can try to find yet another sampling level
                result = getSamplingNode(result, samplingIndex + 1, row);
            }
        }

        return result;
    }

    public SamplingContext createSamplingContext(SpeciesBatchTreeNode rootNode) {

        Set<SpeciesBatchRowModel> childRows = Sets.newHashSet();

        SpeciesBatchRowModel superSamplingRow = getNodeToRow(rootNode);

        float totalWeight = 0f;
        for (int i = 0, nbChildren = rootNode.getChildCount(); i < nbChildren; i++) {
            SpeciesBatchTreeNode node = rootNode.getChildAt(i);

            SpeciesBatchRowModel row = getNodeToRow(node);
            if (row != null) {
                SortedUnsortedCategory sortedUnsortedCategory = row.getSortedUnsortedCategory();
                if (sortedUnsortedCategory != null &&
                    "unsorted".equals(sortedUnsortedCategory.getName())) {

                    // never take account of a such child

                } else {

                    childRows.add(row);
                    Float weight = row.getWeight();
                    if (weight != null) {
                        totalWeight += weight;
                    }
                }
            }
        }
        SamplingContext result = new SamplingContext(rootNode,
                                                     superSamplingRow,
                                                     childRows,
                                                     totalWeight);
        return result;
    }

    public SpeciesBatchRowModel getNodeToRow(SpeciesBatchTreeNode node) {
        Collection<SpeciesBatchRowModel> rows = nodeToRow.get(node);

        SpeciesBatchRowModel result = null;
        if (CollectionUtils.isNotEmpty(rows)) {
            for (SpeciesBatchRowModel next : rows) {
                if (next.isValid()) {
                    result = next;
                    break;
                }
            }
        }
        return result;
    }

    public boolean isValid(SpeciesBatchRowModel row, SpeciesBatchTreeNode newNode) {

        // row must be on a sampling and having a weight
        boolean result = row.getWeight() != null && newNode != null;

        // make sure there is not already another row for this node
        if (result && CollectionUtils.size(nodeToRow.get(newNode)) > 1) {
            result = false;
        }

        return result;
    }

    public static class SamplingContext {

        private final SpeciesBatchTreeNode rootNode;

        private final SpeciesBatchRowModel superSamplingRow;

        private final Set<SpeciesBatchRowModel> samplingRows;

        private final float totalWeight;

        public SamplingContext(SpeciesBatchTreeNode rootNode,
                               SpeciesBatchRowModel superSamplingRow,
                               Set<SpeciesBatchRowModel> samplingRows,
                               float totalWeight) {
            this.rootNode = rootNode;
            this.superSamplingRow = superSamplingRow;
            this.samplingRows = samplingRows;
            this.totalWeight = totalWeight;
        }

        public SpeciesBatchTreeNode getRootNode() {
            return rootNode;
        }

        public SpeciesBatchRowModel getSuperSamplingRow() {
            return superSamplingRow;
        }

        public Set<SpeciesBatchRowModel> getSamplingRows() {
            return samplingRows;
        }

        public float getTotalWeight() {
            return totalWeight;
        }

        public void applyNewSampleValues(float samplingWeight,
                                         Float samplingRatio) {

            for (SpeciesBatchRowModel row : samplingRows) {
                row.setSampleWeight(samplingWeight);
                row.setSamplingRatio(samplingRatio);
            }
        }
    }
}
