package fr.ifremer.tutti.ui.swing.content.operation.fishing.environment;

/*
 * #%L
 * Tutti :: UI
 * $Id: EnvironmentTabUIHandler.java 146 2013-01-02 18:51:13Z kmorin $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-0.2.5/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/fishing/environment/EnvironmentTabUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import fr.ifremer.tutti.persistence.entities.CaracteristicMap;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.data.FishingOperation;
import fr.ifremer.tutti.persistence.entities.referential.Caracteristic;
import fr.ifremer.tutti.ui.swing.content.operation.EditFishingOperationUI;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import fr.ifremer.tutti.ui.swing.util.editor.CaracteristicValueEditor;
import fr.ifremer.tutti.ui.swing.util.editor.CaracteristicValueRenderer;
import fr.ifremer.tutti.ui.swing.util.table.AbstractTuttiTableUIHandler;
import fr.ifremer.tutti.ui.swing.util.table.CaracteristicRow;
import java.util.Collection;
import java.util.List;
import jaxx.runtime.swing.editor.bean.BeanComboBox;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.table.DefaultTableColumnModelExt;

/**
 * @author kmorin
 * @since 0.3
 */
public class EnvironmentTabUIHandler extends AbstractTuttiTableUIHandler<EnvironmentRowModel, EnvironmentTabUIModel> {

    private final static Log log =
            LogFactory.getLog(EnvironmentTabUIHandler.class);

    /**
     * UI.
     *
     * @since 0.3
     */
    protected final EnvironmentTabUI ui;

    public EnvironmentTabUIHandler(EditFishingOperationUI parentUi,
                                   EnvironmentTabUI ui) {
        super(parentUi.getHandler().getContext(),
              EnvironmentRowModel.PROPERTY_VALUE);
        this.ui = ui;
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiTableUIHandler methods                                --//
    //------------------------------------------------------------------------//

    @Override
    protected EnvironmentTableModel getTableModel() {
        return (EnvironmentTableModel) getTable().getModel();
    }

    @Override
    protected JXTable getTable() {
        return ui.getEnvironmentTable();
    }

    @Override
    protected boolean isRowValid(EnvironmentRowModel row) {
        //TODO
        return row.getValue() != null;
    }

    @Override
    protected void onRowModified(int rowIndex,
                                 EnvironmentRowModel row,
                                 String propertyName,
                                 Object oldValue,
                                 Object newValue) {
        recomputeRowValidState(row);
    }

    @Override
    protected void saveSelectedRowIfRequired(TuttiBeanMonitor<EnvironmentRowModel> rowMonitor,
                                             EnvironmentRowModel row) {
        if (row.isValid()) {
            // there is a valid bean attached to the monitor

            if (rowMonitor.wasModified()) {

                // monitored bean was modified, save it
                if (log.isInfoEnabled()) {
                    log.info("Row " + row + " was modified, will save it");
                }

                showInformationMessage(
                        "[ Trait - Environnement ] " +
                        "Sauvegarde des modifications de " + row + ".");

                saveRow(row);

                // clear modified flag on the monitor
                rowMonitor.clearModified();
            }
        }
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    protected EnvironmentTabUIModel getModel() {
        return ui.getModel();
    }

    @Override
    public void beforeInitUI() {
        EnvironmentTabUIModel model = new EnvironmentTabUIModel();
        ui.setContextValue(model);
    }

    @Override
    public void afterInitUI() {
        initUI(ui);

        initBeanComboBox(ui.getNewRowKey(),
                         Lists.<Caracteristic>newArrayList(), null);

        JXTable table = getTable();

        // create table column model
        DefaultTableColumnModelExt columnModel = new DefaultTableColumnModelExt();

        {

            addColumnToModel(columnModel,
                             null,
                             newTableCellRender(Caracteristic.class),
                             EnvironmentTableModel.KEY);
        }

        {

            addColumnToModel(columnModel,
                             new CaracteristicValueEditor(context),
                             new CaracteristicValueRenderer(context),
                             EnvironmentTableModel.VALUE);
        }

        // create table model
        EnvironmentTableModel tableModel =
                new EnvironmentTableModel(columnModel);

        table.setModel(tableModel);
        table.setColumnModel(columnModel);
        initTable(table);
    }

    @Override
    public void onCloseUI() {
        if (log.isInfoEnabled()) {
            log.info("closing: " + ui);
        }
    }
    
    @Override
    protected void onAfterSelectedRowChanged(int oldRowIndex,
                                             EnvironmentRowModel oldRow,
                                             int newRowIndex,
                                             EnvironmentRowModel newRow) {
        
        super.onAfterSelectedRowChanged(oldRowIndex, oldRow, newRowIndex, newRow);
        getModel().setRemoveCaracteristicEnabled(newRow != null);
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    /** Adds a row with the parameter selected in the combo box */
    public void addRow() {
        BeanComboBox<Caracteristic> keyCombo = ui.getNewRowKey();
        Caracteristic selectedItem = (Caracteristic) keyCombo.getSelectedItem();
        
        EnvironmentRowModel row = getTableModel().createNewRow();
        row.setKey(selectedItem);
        getTableModel().addNewRow(row);
        
        EnvironmentTableModel tableModel = getTableModel();
        int rowIndex = tableModel.getRowCount() - 1;
        tableModel.fireTableRowsInserted(rowIndex, rowIndex);

        keyCombo.removeItem(selectedItem);
        selectFirstInCombo(keyCombo);
    }

    /** Resets the table with the data from the database */
    public void reset(FishingOperation fishingOperation) {
        EnvironmentTableModel tableModel = getTableModel();
        EnvironmentTabUIModel model = getModel();
        model.setFishingOperation(fishingOperation);

        CaracteristicMap environmentCaracteristics = fishingOperation.getEnvironmentCaracteristics();
        if (environmentCaracteristics == null) {
            environmentCaracteristics = new CaracteristicMap();
        }

        List<EnvironmentRowModel> rows = Lists.newArrayList();
        Collection<Caracteristic> caracteristics = environmentCaracteristics.keySet();
        
        for (Caracteristic key : caracteristics) {
            EnvironmentRowModel newRow = tableModel.createNewRow();
            newRow.setKey(key);
            newRow.setValue(environmentCaracteristics.get(key));
            rows.add(newRow);
        }

        model.setRows(rows);

        List<Caracteristic> caracteristicList = Lists.newArrayList();

        for (Caracteristic caracteristic : model.getAvailableCaracteristics()) {
            if (!caracteristics.contains(caracteristic)) {
                caracteristicList.add(caracteristic);
            }
        }

        ui.getNewRowKey().setData(caracteristicList);
        selectFirstInCombo(ui.getNewRowKey());
    }
    
    public void removeCaracteristic() {
        int rowIndex = getTable().getSelectedRow();

        Preconditions.checkState(rowIndex != -1,
                                 "Cant remove caracteristic if no caracteristic selected");
        
        EnvironmentRowModel row = getTableModel().getEntry(rowIndex);
        
        FishingOperation fishingOperation = getModel().getFishingOperation();
        Preconditions.checkNotNull(fishingOperation);

        fishingOperation.getEnvironmentCaracteristics().remove(row.getKey());
        
        if (TuttiEntities.isNew(fishingOperation)) {

            fishingOperation = persistenceService.createFishingOperation(fishingOperation);
        } else {
            persistenceService.saveFishingOperation(fishingOperation);
        }
        
        //add the row in the combo
        BeanComboBox keyCombo = ui.getNewRowKey();
        keyCombo.addItem(row.getKey());
        selectFirstInCombo(keyCombo);
        
        // remove the row from the model
        getModel().getRows().remove(rowIndex);

        // refresh all the table
        getTableModel().fireTableRowsDeleted(rowIndex, rowIndex);
        
    }
    
    //------------------------------------------------------------------------//
    //-- Protected methods                                                  --//
    //------------------------------------------------------------------------//
    
    protected void saveRow(EnvironmentRowModel row) {

        FishingOperation fishingOperation = getModel().getFishingOperation();
        Preconditions.checkNotNull(fishingOperation);

        if (fishingOperation.getEnvironmentCaracteristics() == null) {
            fishingOperation.setEnvironmentCaracteristics(new CaracteristicMap());
        }
        fishingOperation.getEnvironmentCaracteristics().put(row.getKey(), row.getValue());
        
        if (TuttiEntities.isNew(fishingOperation)) {

            fishingOperation = persistenceService.createFishingOperation(fishingOperation);
        } else {
            persistenceService.saveFishingOperation(fishingOperation);
        }
    }
}
