package fr.ifremer.tutti.ui.swing;

/*
 * #%L
 * Tutti :: UI
 * $Id: TuttiUIUtil.java 15 2012-11-24 13:10:24Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-0.1/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/TuttiUIUtil.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import jaxx.runtime.JAXXUtil;
import jaxx.runtime.SwingUtil;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.FileUtil;

import javax.swing.Action;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JRootPane;
import javax.swing.KeyStroke;
import javax.swing.UIManager;
import java.awt.Component;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;

/**
 * Created: 14/06/12
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public final class TuttiUIUtil {

    /** Logger. */
    private static final Log log = LogFactory.getLog(TuttiUIUtil.class);

    private TuttiUIUtil() {
        // never instanciate util class
    }

    public static JDialog openInDialog(JComponent ui,
                                       JFrame frame,
                                       String title,
                                       Action closeAction) {
        return openInDialog(ui, frame, title, closeAction, null);
    }

    public static JDialog openInDialog(JComponent ui,
                                       JFrame frame,
                                       String title,
                                       Action closeAction,
                                       Dimension dim) {

        JDialog result = new JDialog(frame, false);
        result.setTitle(title);
        result.add(ui);
        result.setResizable(true);
        if (dim == null) {
            result.setSize(550, 450);
        } else {
            result.setSize(dim);
        }
        if (closeAction != null) {
            JRootPane rootPane = result.getRootPane();
            rootPane.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(KeyStroke.getKeyStroke("ESCAPE"), "close");
            rootPane.getActionMap().put("close", closeAction);
        }
        result.addWindowListener(new WindowAdapter() {

            @Override
            public void windowClosed(WindowEvent e) {
                Component ui = (Component) e.getSource();
                if (log.isInfoEnabled()) {
                    log.info("Destroy ui " + ui);
                }
                JAXXUtil.destroy(ui);
            }
        });
        SwingUtil.center(frame, result);
        result.setVisible(true);
        return result;
    }

    /**
     * Choisir un fichier via un sélecteur graphique de fichiers.
     *
     * @param parent      le component swing appelant le controle
     * @param title       le titre du dialogue de sélection
     * @param buttonLabel le label du boutton d'acceptation
     * @param incoming    le fichier de base à utilier
     * @param filters     les filtres + descriptions sur le sélecteur de
     *                    fichiers
     * @return le fichier choisi ou le fichier incoming si l'opération a été
     *         annulée
     */
    public static File chooseFile(Component parent,
                                  String title,
                                  String buttonLabel,
                                  File incoming,
                                  String... filters) {
        File oldBasedir = FileUtil.getCurrentDirectory();
        if (incoming != null) {
            File basedir;
            if (incoming.isFile()) {
                basedir = incoming.getParentFile();
            } else {
                basedir = incoming;
            }
            if (basedir.exists()) {
                FileUtil.setCurrentDirectory(basedir);
            }
        }
        File file = FileUtil.getFile(title, buttonLabel, parent, filters);
        if (log.isDebugEnabled()) {
            log.debug(title + " : " + file);
        }
        FileUtil.setCurrentDirectory(oldBasedir);
        File result = file == null ? incoming : file;
        return result;
    }

    /**
     * Choisir un répertoire via un sélecteur graphique de fichiers.
     *
     * @param parent      le component swing appelant le controle
     * @param title       le titre de la boite de dialogue de sléection
     * @param buttonLabel le label de l'action d'acceptation
     * @param incoming    le fichier de base à utiliser
     * @return le répertoire choisi ou le répertoire incoming si l'opération a
     *         été annulée
     */
    public static File chooseDirectory(Component parent,
                                       String title,
                                       String buttonLabel,
                                       File incoming) {
        File oldBasedir = FileUtil.getCurrentDirectory();
        if (incoming != null) {
            File basedir;
            if (incoming.isFile()) {
                basedir = incoming.getParentFile();
            } else {
                basedir = incoming;
            }
            if (basedir.exists()) {
                FileUtil.setCurrentDirectory(basedir);
            }
        }
        String file = FileUtil.getDirectory(parent, title, buttonLabel);
        if (log.isDebugEnabled()) {
            log.debug(title + " : " + file);
        }
        FileUtil.setCurrentDirectory(oldBasedir);
        return file == null ? incoming : new File(file);
    }

    public static void updateBusyState(Component ui, boolean busy) {
        if (busy) {
            // ui bloquee
            if (log.isDebugEnabled()) {
                log.debug("block ui in busy mode");
            }
            ui.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
        } else {
            // ui debloquee
            if (log.isDebugEnabled()) {
                log.debug("unblock ui in none busy mode");
            }
            ui.setCursor(Cursor.getDefaultCursor());
        }
    }

    public static void showSuccessMessage(Component ui, String message) {
        JOptionPane.showMessageDialog(
                ui,
                message,
                "Success",
                JOptionPane.INFORMATION_MESSAGE,
                UIManager.getIcon("info")
        );
    }

    public static void showErrorMessage(Component ui, String message) {
        JOptionPane.showMessageDialog(
                ui,
                message,
                "Error",
                JOptionPane.ERROR_MESSAGE,
                UIManager.getIcon("error")
        );
    }

    public static boolean askQuestion(Component ui, String message) {
        int i = JOptionPane.showConfirmDialog(
                ui,
                message,
                "Question?",
                JOptionPane.YES_NO_OPTION);
        return i == JOptionPane.YES_OPTION;
    }
}
