package fr.ifremer.tutti.ichtyometer.feed;

/*
 * #%L
 * Tutti :: Ichtyometer API
 * $Id: FeedReader.java 1574 2014-02-06 17:34:03Z tchemit $
 * $HeadURL: https://svn.codelutin.com/tutti/tags/tutti-3.4.3/tutti-ichtyometer/src/main/java/fr/ifremer/tutti/ichtyometer/feed/FeedReader.java $
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.io.Closeables;
import fr.ifremer.tutti.ichtyometer.IchtyometerClient;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.microedition.io.StreamConnection;
import javax.swing.event.EventListenerList;
import java.io.Closeable;
import java.io.DataInputStream;
import java.io.IOException;

/**
 * To read some records from a ichtyometer in feed mode.
 * <p/>
 * Created on 1/24/14.
 *
 * @author Tony Chemit <chemit@codelutin.com>
 * @since 3.1
 */
public class FeedReader implements Closeable {

    /** Logger. */
    private static final Log log = LogFactory.getLog(FeedReader.class);

    /**
     * Ichtyometer client given in the {@code start} method.
     */
    protected IchtyometerClient client;

    /**
     * Reader runnable code (to be used in a thread).
     */
    protected ReadingRunnable readingRunnable;

    /**
     * To keep list of {@link FeedReaderListener} listeners.
     */
    protected final EventListenerList listenerList;

    public FeedReader() {
        listenerList = new EventListenerList();
    }

    public void start(IchtyometerClient client) throws IOException {

        Preconditions.checkNotNull(client, "client can not be null");
        Preconditions.checkState(client.isOpen(), "client must be opened");
        this.client = client;

        // get connection
        StreamConnection connection = client.openConnection();

        // get input stream
        DataInputStream dataInputStream = connection.openDataInputStream();

        // create the reader runnable
        readingRunnable = new ReadingRunnable(dataInputStream);

        // start the reader thread
        new Thread(readingRunnable).start();

        if (log.isDebugEnabled()) {
            log.debug("Ready to read remote device...");
        }
    }

    @Override
    public void close() throws IOException {

        Preconditions.checkNotNull(client, "client can not be null");
        Preconditions.checkState(client.isOpen(), "client must be opened");

        try {
            readingRunnable.stop();

        } finally {

            Closeables.close(client, true);
        }
    }

    public String getClientName() {
        return client.getName();
    }

    public void addFeedModeReaderListener(FeedReaderListener listener) {
        listenerList.add(FeedReaderListener.class, listener);
    }

    public void removeFeedModeReaderListener(FeedReaderListener listener) {
        listenerList.remove(FeedReaderListener.class, listener);
    }

    protected class ReadingRunnable implements Runnable {

        /**
         * Input stream to read new record.
         */
        protected final DataInputStream dataInputStream;

        /**
         * Flag to stop the runnable.
         */
        protected boolean stop;

        protected ReadingRunnable(DataInputStream dataInputStream) {
            this.dataInputStream = dataInputStream;
        }

        protected void stop() {
            this.stop = true;
        }

        @Override
        public void run() {

            if (log.isInfoEnabled()) {
                log.info("Reader thread start... " + this);
            }

            while (!stop) {

                try {
                    // get a new record
                    FeedReaderRecord readerRecord = readRecord(dataInputStream);

                    if (!stop) {

                        // send new record to listeners
                        FeedReaderEvent e = new FeedReaderEvent(FeedReader.this, readerRecord);
                        for (FeedReaderListener listener : listenerList.getListeners(FeedReaderListener.class)) {
                            listener.recordRead(e);
                        }
                    }
                } catch (IOException e) {
                    if (log.isErrorEnabled()) {
                        log.error("Could not read record", e);
                    }
                }
            }

            if (log.isInfoEnabled()) {
                log.info("Reader thread stop..." + this);
            }
        }

        protected FeedReaderRecord readRecord(DataInputStream dataInputStream) throws IOException {
            String result = "";

            // wait until got a @

            boolean complete = false;

            while (!complete) {

                if (stop) {
                    break;
                }
                while (dataInputStream.available() > 0) {
                    if (stop) {
                        break;
                    }
                    int c = dataInputStream.read();

                    if (c == '@') {

                        complete = true;
                    } else {

                        result += (char) c;
                    }
                }
            }

            FeedReaderRecord readerRecord = null;

            if (!stop) {

                int lastComma = result.lastIndexOf(',');
                result = result.substring(0, lastComma);
                lastComma = result.lastIndexOf(',')+1;
                String record = result.substring(0, lastComma);
                String crc = result.substring(lastComma);

                if (log.isDebugEnabled()) {
                    log.debug(String.format("Record: %s / %s", record, crc));
                }
                readerRecord = new FeedReaderRecord(record, crc);
            }
            return readerRecord;
        }
    }
}
