/*
 * #%L
 * IsisFish
 * 
 * $Id: EvaluatorHelper.java 3969 2014-04-17 16:48:13Z echatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2006 - 2012 Ifremer, Code Lutin, Cédric Pineau, Benjamin Poussin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.util;

import static org.nuiton.i18n.I18n.t;

import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.apache.commons.io.FileUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import fr.ifremer.isisfish.IsisFish;
import fr.ifremer.isisfish.IsisFishRuntimeException;
import fr.ifremer.isisfish.simulator.SimulationContext;
import java.util.HashMap;
import org.apache.commons.lang3.StringUtils;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Permet d'evaluer les equations ecritent en Java
 * 
 * Created: 3 juil. 2006 23:44:48
 *
 * @author poussin
 * @version $Revision: 3969 $
 *
 * Last update: $Date: 2014-04-17 18:48:13 +0200 (jeu., 17 avril 2014) $
 * by : $Author: echatellier $
 */
public class EvaluatorHelper {

    /** Logger for this class. */
    private static Log log = LogFactory.getLog(EvaluatorHelper.class);

    /**
     * Regex to match import:
     * - "^(import ...;)" for import at line beginning
     * - ";(import ...;)" for multiple import on same line
     */
    static protected Pattern grepImportPattern = Pattern.compile("(?:^\\s*|(?<=;)\\s*)(import[^;]+;)", Pattern.DOTALL + Pattern.MULTILINE);

    private static final String HASH_CACHE_KEY = "__hashCache__";

    protected static String getHashCache(File fileCheckSum) {
        String result = "";

        // FIXME echatellier c'est pas possible que cela renvoit null
        SimulationContext context = SimulationContext.get();
        if (context == null) {
            // on est pas dans une simulation
            // on verifie dans le fichier
            if (fileCheckSum.exists()) {
                try {
                    result = FileUtils.readFileToString(fileCheckSum);
                } catch (IOException eee) {
                    log.info("Can't read old checkSum:  " + fileCheckSum, eee);
                }
            }
        } else {
            // on est dans une simulation, on verifie dans le cache de la simulation
            String key = "__hashCache__";
            Map<String, String> cache = (Map<String, String>)context.getValue(key);
            if (cache != null) {
                result = StringUtils.defaultString(cache.get(fileCheckSum.getPath()));
            }
        }

        return result;
    }

    protected static void setHashCache(File fileCheckSum, String hashcode) {
        SimulationContext context = SimulationContext.get();
        if (context == null) {
            // on est pas dans une simulation
            // on ecrit dans le fichier
            try {
                FileUtils.writeStringToFile(fileCheckSum, hashcode);
            } catch (IOException eee) {
                log.info("Can't write checkSum:  " + fileCheckSum, eee);
            }
        } else {
            // on est dans une simulation, on verifie dans le cache de la simulation
            Map<String, String> cache = (Map<String, String>)context.getValue(HASH_CACHE_KEY);
            if (cache == null) {
                context.setValue(HASH_CACHE_KEY, cache = new HashMap<String, String>());
            }

            cache.put(fileCheckSum.getPath(), hashcode);
        }
    }

    protected static String normalizeClassName(String name) {
        StringBuilder result = new StringBuilder(name);
        for (int i=0; i<result.length(); i++) {
            char c = result.charAt(i);
            if (!Character.isJavaIdentifierPart(c)) {
                result.setCharAt(i, '_');
            }
        }
        return result.toString();
    }

    /**
     * Verifie si un script (prescript/equation) est syntaxiquement correct.
     * 
     * @param javaInterface
     * @param script
     * @param out output writer (can be null for non output)
     * @return 0 si ok
     */
    public static int check(Class javaInterface, String script, PrintWriter out) {
        try {
            File src = File.createTempFile("check", "equation");
            src.deleteOnExit();
            String packageName = null;
            String className = normalizeClassName(src.getName());
            
            src = new File(src.getParentFile(), className + ".java");
            src.deleteOnExit();
            
            // recherche la methode de l'interface
            Method [] methods = javaInterface.getDeclaredMethods();
            Method interfaceMethod = methods[0];
            
            String content = generateContent(packageName, className, interfaceMethod, script);

            FileUtils.writeStringToFile(src, content, "utf-8");

            int compileResult = CompileHelper.compile(src.getParentFile(), src, src.getParentFile(), out);
            File dest = new File(src.getParentFile(), className + ".class");
            dest.deleteOnExit();

            return compileResult;

        } catch (Exception eee) {
            log.warn("Can't check equation", eee);
            return -10000;
        }
    }

    /**
     * Evalue une equation.
     * 
     * @param packageName le nom de package de la classe
     * @param className le nom de la classe
     * @param javaInterface l'interface que la classe doit etendre,
     *        cette interface n'a qu'un methode
     * @param script le code de la methode
     * @param args les arguments a utiliser pour l'appel de la methode
     * @return la valeur retourné par la methode
     */
    public static Object evaluate(String packageName, String className,
            Class javaInterface, String script, Object... args) {
        className = normalizeClassName(className);

        Object result = null;
        Class clazz = null;

        // recherche la methode de l'interface
        Method [] methods = javaInterface.getDeclaredMethods();
        Method interfaceMethod = methods[0];

        String classname = packageName + "." + className;

        File fileRootSrc = IsisFish.config.getCompileDirectory();
        File fileCheckSum = new File(fileRootSrc, packageName + File.separator + className + ".hashCode");
        File fileSrc = new File(fileRootSrc, packageName + File.separator + className + ".java");
        //File fileDest = new File(fileRootSrc, packageName + File.separator + className + ".class");

        // if equation's Java file exists, check the checksum 
        String oldCheckSum = getHashCache(fileCheckSum);
        String newCheckSum = Integer.toString(script.hashCode());
        boolean checkSumEquals = newCheckSum.equals(oldCheckSum);

        // if Java file's checkSum is not equals to script's checkSum
        // generate new Java file
        if (!checkSumEquals) {

            String content = generateContent(packageName, className, interfaceMethod, script);
            try {
                // force writing to UTF-8
                // fix compilation issue : unmappable characters
                FileUtils.writeStringToFile(fileSrc, content, "utf-8");
                setHashCache(fileCheckSum, Integer.toString(script.hashCode()));
                setHashCache(fileCheckSum, Integer.toString(script.hashCode()));
            } catch (IOException zzz) {
                throw new IsisFishRuntimeException(t("isisfish.error.save.script.compilation", fileSrc), zzz);
            }

            compile(fileRootSrc, fileSrc);
        }

        // try to load class
        try {
            ClassLoader cl = IsisFish.config.getScriptClassLoader();
            clazz = cl.loadClass(classname);
        } catch (Exception ex) {

            // force to compile it again; may happen if hashcode
            // is correct, but .class file can't be loaded
            compile(fileRootSrc, fileSrc);

            try {
                // second load
                ClassLoader cl = IsisFish.config.getScriptClassLoader();
                clazz = cl.loadClass(classname);
            } catch (Exception ex2) {
                throw new IsisFishRuntimeException(t("isisfish.error.compile.script", fileSrc), ex2);
            }
        }

        result = invoke(clazz, interfaceMethod, args);

        return result;
    }

    protected static void compile(File fileRootSrc, File fileSrc) {
        try {
            List<File> classpath = new ArrayList<File>();
            classpath.add(fileRootSrc.getAbsoluteFile());
            classpath.add(IsisFish.config.getDatabaseDirectory().getAbsoluteFile());
            int compileResult = CompileHelper.compile(classpath, Collections.singletonList(fileSrc), fileRootSrc, null);

            if (compileResult != 0) {
                throw new IsisFishRuntimeException(t("isisfish.error.compile.script", compileResult, fileSrc));
            }
        } catch (Exception zzz) {
            throw new IsisFishRuntimeException(t("isisfish.error.compile.script", fileSrc), zzz);
        }
    }

    /**
     * Generate script content.
     * 
     * Warning, content are always on a unique single line (without \n) for debugging purpose.
     * 
     * @param packageName
     * @param className
     * @param interfaceMethod
     * @param script
     * @return script return (or null)
     */
    protected static String generateContent(String packageName, String className, Method interfaceMethod, String script) {

        StringBuilder imports = new StringBuilder();
        StringBuilder code = new StringBuilder();

        grepImport(script, imports, code);

        String content = "";
        if (packageName != null && !"".equals(packageName)) {
            content += "package " + packageName + ";";
        }

        // add common used imports
        content += "import java.util.*;";
        content += "import java.io.*;";
        content += "import fr.ifremer.isisfish.entities.*;";
        content += "import fr.ifremer.isisfish.types.*;";
        content += "import org.nuiton.math.matrix.*;";
        content += "import org.apache.commons.logging.*;";
        content += imports.toString();
        
        // generate content (do not add \n here, this help debug)
        content += "public class " + className + " implements " + interfaceMethod.getDeclaringClass().getName() + " {";
        content += "private static Log log = LogFactory.getLog(" + className + ".class);";
        content += "public " + interfaceMethod.getReturnType().getName() +  " " + interfaceMethod.getName() + "(";

        Args args = interfaceMethod.getAnnotation(Args.class);
        String [] names = args.value();

        String [] stringTypes;
        ArgTypes argTypes = interfaceMethod.getAnnotation(ArgTypes.class);
        if (argTypes != null) {
            stringTypes = argTypes.value();
        } else {
            stringTypes = new String[names.length];
            Class [] types = interfaceMethod.getParameterTypes();
            for (int i=0; i<types.length; i++) {
                stringTypes[i] = types[i].getName();
            }
        }

        for (int i=0; i<names.length; i++) {
            content += stringTypes[i] + " " + names[i];
            if (i+1<names.length) {
                content += ", ";
            }
        }
        content += ") throws Exception {";
        content += code.toString();
        content += "\n}\n}\n";

        return content;
    }

    /**
     * looking for import in code. return all import as found in code in imports args
     * all other code are put in others
     *
     * @param code
     * @param imports
     * @param others
     */
    protected static void grepImport(String code, StringBuilder imports, StringBuilder others) {
        Matcher matches = grepImportPattern.matcher(code);
        int pos = 0;
        while (matches.find()) {
            int begin = matches.start(1);
            int end = matches.end(1);

            others.append(code.substring(pos, begin));
            imports.append(code.substring(begin, end));
            pos = end;

        }

        others.append(code.substring(pos));
    }

    protected static Object invoke(Class clazz, Method interfaceMethod, Object... args) {

        Object result;
        try {
            Method method = clazz.getDeclaredMethod(interfaceMethod.getName(),
                    interfaceMethod.getParameterTypes());

            Object eq = clazz.newInstance();
            result = method.invoke(eq, args);

        } catch (Exception eee) {
            throw new IsisFishRuntimeException(t("isisfish.error.invoke.method", interfaceMethod, clazz.getName()), eee);
        }
        return result;
    }
}
