/*
 * #%L
 * IsisFish
 * 
 * $Id: FileOffsetReader.java 3969 2014-04-17 16:48:13Z echatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2002 - 2010 Ifremer, CodeLutin, Benjamin Poussin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.logging.io;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.EOFException;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;

/**
 * A simple implementation of {@link OffsetReader} using a file for
 * dealing with offsets.
 * 
 * The offset of lines are pre-computed in a file {@link #offsetFile}, generated
 * by {@link #createOffsets(LineReader)}. or
 * {@link #createOffsets(LineReader , LineReader)}.
 *
 * @author chemit
 */
public class FileOffsetReader extends OffsetReader {

    protected static final Log log = LogFactory.getLog(FileOffsetReader.class);

    /** offset file to read */
    protected File offsetFile;

    /** offset file reader */
    protected RandomAccessFile offsetReader;

    /** writer used while creation or update */
    protected RandomAccessFile writer;

    public FileOffsetReader(File file) {
        this.offsetFile = file;
    }

    public long getOffset(long position) throws IOException {
        // a long is on 8 bits
        offsetReader.seek(position * 8);
        // use current stream and seek to right position
        long result;
        try {
            result = offsetReader.readLong();
        } catch (EOFException e) {
            result = -1;
        }
        return result;
    }

    @Override
    public void open(LineReader reader) throws IOException {
        super.open(reader);
        try {
            offsetReader = new RandomAccessFile(this.offsetFile, "r");
        } catch (FileNotFoundException e) {
            throw new IllegalArgumentException(e);
        }
        if (!needCreate()) {
            // we know extacly the number of line from the size of offsets file
            this.nbLines = offsetReader.length() / 8;
        }
    }

    @Override
    public void close() throws IOException {
        IOException error=null;
        if (offsetReader != null) {
            try {
                offsetReader.close();
            } catch (IOException e) {
                error=e;
            }
            offsetReader=null;
        }
        offsetFile.deleteOnExit();
        if (error!=null) {
            throw error;
        }
    }

    @Override
    public synchronized void update(LineReader lineReader) throws IOException {
        // first close actually readers
        offsetReader.close();
        // goto end of file (from old length)
        writer = new RandomAccessFile(offsetFile.getAbsolutePath(), "rw");
        try {
            super.update(lineReader);
        } finally {
            writer.close();
            writer=null;
            // reopen readers
            this.offsetReader = new RandomAccessFile(this.offsetFile, "r");
        }
    }

    @Override
    public void deleteOffsetFile() {
        if (offsetFile != null && offsetFile.exists()) {
            offsetFile.delete();
        }
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("(offsetfile:").append(offsetFile.getName()).append(", nbLines:").append(getNbLines()).append(')');
        return sb.toString();
    }

    protected void storeOffset(long position, long offset) throws IOException {
        writer.writeLong(offset);
    }

    protected boolean needCreate() throws IOException {
        return !offsetFile.exists();
    }

    @Override
    protected long createOffsets(LineReader lineReader) throws IOException {
        try {
            writer = new RandomAccessFile(offsetFile, "rw");
            return super.createOffsets(lineReader);
        } finally {
            if (writer != null) {
                writer.close();
                writer=null;
            }
        }
    }

    @Override
    protected long createOffsets(LineReader lineReader, LineReader parent) throws IOException {
        try {
            writer = new RandomAccessFile(offsetFile, "rw");
            return super.createOffsets(lineReader, parent);
        } finally {
            if (writer != null) {
                writer.close();
                writer=null;
            }
        }
    }
}
