/*
 * #%L
 * EchoBase :: UI
 * 
 * $Id: ManageExportQuery.java 842 2013-08-15 10:53:16Z tchemit $
 * $HeadURL: https://forge.codelutin.com/svn/echobase/tags/echobase-2.3/echobase-ui/src/main/java/fr/ifremer/echobase/ui/actions/exportQuery/ManageExportQuery.java $
 * %%
 * Copyright (C) 2011 Ifremer, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package fr.ifremer.echobase.ui.actions.exportQuery;

import com.opensymphony.xwork2.Preparable;
import fr.ifremer.echobase.entities.EchoBaseUser;
import fr.ifremer.echobase.entities.ExportQuery;
import fr.ifremer.echobase.services.service.InternalDbPersistenceService;
import fr.ifremer.echobase.services.service.exportquery.ExportQueryInvalidNameException;
import fr.ifremer.echobase.services.service.exportquery.ExportQueryNameAlreadyExistException;
import fr.ifremer.echobase.services.service.exportquery.ExportQueryService;
import fr.ifremer.echobase.ui.actions.EchoBaseActionSupport;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.Map;

/**
 * To manage {@link ExportQuery}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class ManageExportQuery extends EchoBaseActionSupport implements Preparable {

    private static final long serialVersionUID = 1L;

    /** Logger. */
    private static final Log log = LogFactory.getLog(ManageExportQuery.class);

    /** Selected query loaded from database if his id is not empty. */
    protected ExportQuery query;

    public ExportQuery getQuery() {
        if (query == null) {
            query = exportQueryService.newExportQuery();
        }
        return query;
    }

    protected boolean newQuery;

    public void setNewQuery(boolean newQuery) {
        this.newQuery = newQuery;
    }

    public boolean isNewQuery() {
        return newQuery;
    }

    protected boolean newLibreOfficeQuery;

    public void setNewLibreOfficeQuery(boolean newLibreOfficeQuery) {
        this.newLibreOfficeQuery = newLibreOfficeQuery;
    }

    public boolean isNewLibreOfficeQuery() {
        return newLibreOfficeQuery;
    }

    protected boolean queryExists;

    public boolean isQueryExists() {
        return StringUtils.isNotEmpty(getQuery().getTopiaId());
    }

    protected boolean canUpdateQuery;

    public boolean isCanUpdateQuery() {
        return canUpdateQuery;
    }

    /** All available queries from database. */
    protected Map<String, String> queries;

    public Map<String, String> getQueries() {
        return queries;
    }

    public Map<String, String> getTableNames() {
        return queries;
    }

    public String saveQuery() throws Exception {

        String result = INPUT;

        boolean safeSql = checkQuery();

        if (safeSql) {
            try {
                query = exportQueryService.createOrUpdate(
                        getQuery(), getEchoBaseSession().getUser());

                result = SUCCESS;
            } catch (ExportQueryInvalidNameException e) {
                addFieldError("query.name",
                              _("echobase.error.query.invalid.name"));
            } catch (ExportQueryNameAlreadyExistException e) {
                addFieldError("query.name",
                              _("echobase.error.query.name.already.exists"));
            }
        }
        return result;
    }

    public String cloneQuery() throws Exception {

        getQuery().setTopiaId(null);
        getQuery().setName(getQuery().getName() + "-clone");
        query = exportQueryService.createOrUpdate(getQuery(),
                                                  getEchoBaseSession().getUser()
        );
        return SUCCESS;
    }

    public String confirmDeleteQuery() throws Exception {

        return SUCCESS;
    }

    public String deleteQuery() throws Exception {

        //TODO : do validation
        exportQueryService.delete(getQuery().getTopiaId());
        query = null;
        return SUCCESS;
    }

    @Override
    public String execute() {

        if (StringUtils.isNotEmpty(getQuery().getTopiaId())) {

            // test query
            checkQuery();
        }
        return SUCCESS;
    }

    @Override
    public void prepare() throws Exception {

        queries = internalDbPersistenceService.loadSortAndDecorate(ExportQuery.class);

        if (queries.isEmpty()) {

            // no query saved
            addFlashMessage(_("echobase.info.no.sqlQuery.saved"));
        } else {

            String selectedQueryId = getQuery().getTopiaId();

            if (!isQueryExists()) {

                if (isNewQuery() || isNewLibreOfficeQuery()) {

                    // new query in progress
                    addFlashMessage(_("echobase.info.new.sqlQuery.inprogress"));
                } else {

                    // no query selected
                    addFlashMessage(_("echobase.info.no.sqlQuery.selected"));
                }
            } else {

                // load query from database
                query = exportQueryService.getExportQuery(selectedQueryId);

                EchoBaseUser echoBaseUser =
                        getEchoBaseSession().getUser();
                canUpdateQuery = echoBaseUser.isAdmin() ||
                                 echoBaseUser.getEmail().equals(
                                         query.getLastModifiedUser());

                if (!canUpdateQuery) {

                    // this user can not update selected query
                    addFlashMessage(
                            _("echobase.info.sqlQuery.not.modifiable"));
                }
            }
        }
    }

    private boolean checkQuery() {

        boolean result;
        try {
            exportQueryService.testSql(getQuery().getSqlQuery());
            result = true;
        } catch (Exception e) {
            Throwable cause = e.getCause();
            if (log.isWarnEnabled()) {
                log.warn("Invalid sql ", cause);
            }
            addFieldError("query.sqlQuery",
                          _("echobase.error.invalid.sql", cause.getMessage()));
            result = false;
        }
        return result;
    }

    //------------------------------------------------------------------------//
    //-- Injected objects                                                     //
    //------------------------------------------------------------------------//

    protected transient ExportQueryService exportQueryService;

    public void setExportQueryService(ExportQueryService exportQueryService) {
        this.exportQueryService = exportQueryService;
    }

    protected transient InternalDbPersistenceService internalDbPersistenceService;

    public void setInternalDbPersistenceService(InternalDbPersistenceService internalDbPersistenceService) {
        this.internalDbPersistenceService = internalDbPersistenceService;
    }
}