/*
 * #%L
 * EchoBase :: UI
 * 
 * $Id: ManageUser.java 116 2011-11-24 23:06:43Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/echobase/trunk/echobase-ui/src/main/java/fr/ifremer/echobase/ui/actions/user/ManageUser.java $
 * %%
 * Copyright (C) 2011 Ifremer, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package fr.ifremer.echobase.ui.actions.user;

import com.opensymphony.xwork2.Preparable;
import fr.ifremer.echobase.entities.EchoBaseUser;
import fr.ifremer.echobase.entities.EchoBaseUserDTO;
import fr.ifremer.echobase.entities.EchoBaseUserDTOImpl;
import fr.ifremer.echobase.services.UserService;
import fr.ifremer.echobase.ui.actions.EchoBaseActionSupport;
import fr.ifremer.echobase.ui.actions.EditActionEnum;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Action to manage user (create - update - change password,...)
 *
 * @author sletellier <letellier@codelutin.com>
 * @since 0.1
 */
public class ManageUser extends EchoBaseActionSupport implements Preparable {

    protected static final Log log = LogFactory.getLog(ManageUser.class);

    private static final long serialVersionUID = 1L;

    public static final String BACK_TO_LIST = "backToList";

    protected transient UserService userService;

    protected EchoBaseUserDTO user;

    protected String userEditAction;

    public String getUserEditAction() {
        return userEditAction;
    }

    public void setUserEditAction(String userEditAction) {
        this.userEditAction = userEditAction;
    }

    protected UserService getUserService() {
        if (userService == null) {
            userService = newService(UserService.class);
        }
        return userService;
    }

    @Override
    public void prepare() throws Exception {

        String userId = getUser().getId();
        if (!StringUtils.isEmpty(userId)) {

            // load user
            user = getUserService().getEntityById(EchoBaseUser.class, userId).toDTO();

            log.info("Selected user " + user.getEmail());
        }
    }

    public String doCreateOrUpdate() throws Exception {
        EchoBaseUserDTO user = getUser();
        String userEmail = user.getEmail();

        if (log.isInfoEnabled()) {
            log.info("will create user  " + userEmail);
        }

        // create user
        getUserService().createOrUpdate(user);
        return BACK_TO_LIST;
    }

    public String doDelete() throws Exception {
        EchoBaseUserDTO user = getUser();
        String userEmail = user.getEmail();

        if (log.isInfoEnabled()) {
            log.info("will delete user " + userEmail);
        }
        getUserService().delete(user);
        return BACK_TO_LIST;
    }

    @Override
    public void validate() {

        EditActionEnum action = getEditActionEnum();

        log.info("Edit action : " + action);

        if (action == null) {

            // no validation (no edit action)
            return;
        }

        EchoBaseUserDTO user = getUser();
        String userEmail = user.getEmail();

        switch (action) {

            case CREATE:

                // login + password required
                if (StringUtils.isEmpty(userEmail)) {

                    // empty user login
                    addFieldError("user.email",
                                  _("echobase.error.required.email"));
                } else {

                    // check login not already used
                    EchoBaseUser login;
                    try {
                        login = getUserService().getUserByEmail(userEmail);
                    } catch (Exception e) {

                        // could not get user
                        throw new IllegalStateException(
                                "Could not obtain user " + userEmail, e);
                    }
                    if (login != null) {
                        addFieldError("user.email",
                                      _("echobase.error.email.already.used"));
                    }
                }

                String userPassword = user.getPassword();
                if (StringUtils.isEmpty(userPassword)) {

                    // empty user password
                    addFieldError("user.password",
                                  _("echobase.error.required.password"));
                }

                break;
            case EDIT:

                // at the moment nothing to validate
                break;
            case DELETE:

                // Do nothing
            default:
                // nothing to validate
        }
    }

    public EchoBaseUserDTO getUser() {
        if (user == null) {
            user = new EchoBaseUserDTOImpl();
        }
        return user;
    }

    protected EditActionEnum getEditActionEnum() {
        if (userEditAction == null) {
            return null;
        }
        return EditActionEnum.valueOf(userEditAction);
    }
}
